"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Task = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * A task that can be performed on the project. Modeled as a series of shell
 * commands and subtasks.
 */
class Task {
    constructor(name, props = {}) {
        this.name = name;
        this._description = props.description;
        this.condition = props.condition;
        this.cwd = props.cwd;
        this._locked = false;
        this._env = props.env ?? {};
        this._steps = props.steps ?? [];
        this.requiredEnv = props.requiredEnv;
        if (props.exec && props.steps) {
            throw new Error("cannot specify both exec and steps");
        }
        if (props.exec) {
            this.exec(props.exec, { receiveArgs: props.receiveArgs });
        }
    }
    /**
     * Forbid additional changes to this task.
     */
    lock() {
        this._locked = true;
    }
    /**
     * Returns the description of this task.
     */
    get description() {
        return this._description;
    }
    /**
     * Sets the description of this task.
     */
    set description(desc) {
        this._description = desc;
    }
    /**
     * Reset the task so it no longer has any commands.
     * @param command the first command to add to the task after it was cleared.
     */
    reset(command, options = {}) {
        this.assertUnlocked();
        while (this._steps.length) {
            this._steps.shift();
        }
        if (command) {
            this.exec(command, options);
        }
    }
    /**
     * Executes a shell command
     * @param command Shell command
     * @param options Options
     */
    exec(command, options = {}) {
        this.assertUnlocked();
        this._steps.push({ exec: command, ...options });
    }
    /**
     * Execute a builtin task.
     *
     * Builtin tasks are programs bundled as part of projen itself and used as
     * helpers for various components.
     *
     * In the future we should support built-in tasks from external modules.
     *
     * @param name The name of the builtin task to execute (e.g.
     * `release/resolve-version`).
     */
    builtin(name) {
        this.assertUnlocked();
        this._steps.push({ builtin: name });
    }
    /**
     * Say something.
     * @param message Your message
     * @param options Options
     */
    say(message, options = {}) {
        this.assertUnlocked();
        this._steps.push({ say: message, ...options });
    }
    /**
     * Adds a command at the beginning of the task.
     * @param shell The command to add.
     *
     * @deprecated use `prependExec()`
     */
    prepend(shell, options = {}) {
        this.assertUnlocked();
        this.prependExec(shell, options);
    }
    /**
     * Spawns a sub-task.
     * @param subtask The subtask to execute.
     */
    spawn(subtask, options = {}) {
        this.assertUnlocked();
        this._steps.push({ spawn: subtask.name, ...options });
    }
    /**
     * Adds a command at the beginning of the task.
     * @param shell The command to add.
     */
    prependExec(shell, options = {}) {
        this.assertUnlocked();
        this._steps.unshift({
            exec: shell,
            ...options,
        });
    }
    /**
     * Adds a spawn instruction at the beginning of the task.
     * @param subtask The subtask to execute.
     */
    prependSpawn(subtask, options = {}) {
        this.assertUnlocked();
        this._steps.unshift({
            spawn: subtask.name,
            ...options,
        });
    }
    /**
     * Says something at the beginning of the task.
     * @param message Your message
     */
    prependSay(message, options = {}) {
        this.assertUnlocked();
        this._steps.unshift({
            say: message,
            ...options,
        });
    }
    /**
     * Adds an environment variable to this task.
     * @param name The name of the variable
     * @param value The value. If the value is surrounded by `$()`, we will
     * evaluate it within a subshell and use the result as the value of the
     * environment variable.
     */
    env(name, value) {
        this.assertUnlocked();
        this._env[name] = value;
    }
    /**
     * Returns an immutable copy of all the step specifications of the task.
     */
    get steps() {
        // If the list of steps is a Lazy value, we can't know what the steps
        // are until synthesis occurs, so just return an empty array.
        if (!Array.isArray(this._steps)) {
            return [];
        }
        return [...this._steps];
    }
    /**
     * Renders a task spec into the manifest.
     *
     * @internal
     */
    _renderSpec() {
        return {
            name: this.name,
            description: this.description,
            env: this._env,
            requiredEnv: this.requiredEnv,
            steps: this._steps,
            condition: this.condition,
            cwd: this.cwd,
        };
    }
    assertUnlocked() {
        if (this._locked) {
            throw new Error(`Task "${this.name}" is locked for changes`);
        }
    }
}
exports.Task = Task;
_a = JSII_RTTI_SYMBOL_1;
Task[_a] = { fqn: "projen.Task", version: "0.65.76" };
//# sourceMappingURL=data:application/json;base64,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