"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Eslint = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const common_1 = require("../common");
const component_1 = require("../component");
const json_1 = require("../json");
const yaml_1 = require("../yaml");
const prettier_1 = require("./prettier");
/**
 * Represents eslint configuration.
 */
class Eslint extends component_1.Component {
    constructor(project, options) {
        super(project);
        this._plugins = new Array();
        this._extends = new Array();
        this.nodeProject = project;
        project.addDevDeps("eslint@^8", "@typescript-eslint/eslint-plugin@^5", "@typescript-eslint/parser@^5", "eslint-import-resolver-node", "eslint-import-resolver-typescript", "eslint-plugin-import", "json-schema");
        if (options.aliasMap) {
            project.addDevDeps("eslint-import-resolver-alias");
        }
        const devdirs = options.devdirs ?? [];
        const dirs = [...options.dirs, ...devdirs];
        const fileExtensions = options.fileExtensions ?? [".ts"];
        this._allowDevDeps = new Set((devdirs ?? []).map((dir) => `**/${dir}/**`));
        const lintProjenRc = options.lintProjenRc ?? true;
        const lintProjenRcFile = options.lintProjenRcFile ?? common_1.PROJEN_RC;
        const eslint = project.addTask("eslint", {
            description: "Runs eslint against the codebase",
            exec: [
                "eslint",
                `--ext ${fileExtensions.join(",")}`,
                "--fix",
                "--no-error-on-unmatched-pattern",
                ...dirs,
                ...(lintProjenRc && lintProjenRcFile ? [lintProjenRcFile] : []),
            ].join(" "),
        });
        project.testTask.spawn(eslint);
        // exclude some files
        project.npmignore?.exclude("/.eslintrc.json");
        this._formattingRules = {
            // see https://github.com/typescript-eslint/typescript-eslint/blob/master/packages/eslint-plugin/docs/rules/indent.md
            indent: ["off"],
            "@typescript-eslint/indent": ["error", 2],
            // Style
            quotes: ["error", "single", { avoidEscape: true }],
            "comma-dangle": ["error", "always-multiline"],
            "comma-spacing": ["error", { before: false, after: true }],
            "no-multi-spaces": ["error", { ignoreEOLComments: false }],
            "array-bracket-spacing": ["error", "never"],
            "array-bracket-newline": ["error", "consistent"],
            "object-curly-spacing": ["error", "always"],
            "object-curly-newline": ["error", { multiline: true, consistent: true }],
            "object-property-newline": [
                "error",
                { allowAllPropertiesOnSameLine: true },
            ],
            "keyword-spacing": ["error"],
            "brace-style": ["error", "1tbs", { allowSingleLine: true }],
            "space-before-blocks": ["error"],
            curly: ["error", "multi-line", "consistent"],
            "@typescript-eslint/member-delimiter-style": ["error"],
            // Require semicolons
            semi: ["error", "always"],
            // Max line lengths
            "max-len": [
                "error",
                {
                    code: 150,
                    ignoreUrls: true,
                    ignoreStrings: true,
                    ignoreTemplateLiterals: true,
                    ignoreComments: true,
                    ignoreRegExpLiterals: true,
                },
            ],
            // Don't unnecessarily quote properties
            "quote-props": ["error", "consistent-as-needed"],
        };
        this.rules = {
            // Require use of the `import { foo } from 'bar';` form instead of `import foo = require('bar');`
            "@typescript-eslint/no-require-imports": ["error"],
            // Require all imported dependencies are actually declared in package.json
            "import/no-extraneous-dependencies": [
                "error",
                {
                    // Only allow importing devDependencies from "devdirs".
                    devDependencies: () => this.renderDevDepsAllowList(),
                    optionalDependencies: false,
                    peerDependencies: true,
                },
            ],
            // Require all imported libraries actually resolve (!!required for import/no-extraneous-dependencies to work!!)
            "import/no-unresolved": ["error"],
            // Require an ordering on all imports
            "import/order": [
                "warn",
                {
                    groups: ["builtin", "external"],
                    alphabetize: { order: "asc", caseInsensitive: true },
                },
            ],
            // Cannot import from the same module twice
            "no-duplicate-imports": ["error"],
            // Cannot shadow names
            "no-shadow": ["off"],
            "@typescript-eslint/no-shadow": ["error"],
            // Required spacing in property declarations (copied from TSLint, defaults are good)
            "key-spacing": ["error"],
            // No multiple empty lines
            "no-multiple-empty-lines": ["error"],
            // One of the easiest mistakes to make
            "@typescript-eslint/no-floating-promises": ["error"],
            // Make sure that inside try/catch blocks, promises are 'return await'ed
            // (must disable the base rule as it can report incorrect errors)
            "no-return-await": ["off"],
            "@typescript-eslint/return-await": ["error"],
            // Useless diff results
            "no-trailing-spaces": ["error"],
            // Must use foo.bar instead of foo['bar'] if possible
            "dot-notation": ["error"],
            // Are you sure | is not a typo for || ?
            "no-bitwise": ["error"],
            // Member ordering
            "@typescript-eslint/member-ordering": [
                "error",
                {
                    default: [
                        "public-static-field",
                        "public-static-method",
                        "protected-static-field",
                        "protected-static-method",
                        "private-static-field",
                        "private-static-method",
                        "field",
                        // Constructors
                        "constructor",
                        // Methods
                        "method",
                    ],
                },
            ],
        };
        // Overrides for .projenrc.js
        this.overrides = [
            {
                files: [lintProjenRcFile || common_1.PROJEN_RC],
                rules: {
                    "@typescript-eslint/no-require-imports": "off",
                    "import/no-extraneous-dependencies": "off",
                },
            },
        ];
        this.ignorePatterns = options.ignorePatterns ?? [
            "*.js",
            `!${lintProjenRcFile || common_1.PROJEN_RC}`,
            "*.d.ts",
            "node_modules/",
            "*.generated.ts",
            "coverage",
        ];
        const tsconfig = options.tsconfigPath ?? "./tsconfig.json";
        this.addPlugins("@typescript-eslint");
        this.addPlugins("import");
        this.addExtends("plugin:import/typescript");
        this.config = {
            env: {
                jest: true,
                node: true,
            },
            root: true,
            plugins: () => this._plugins,
            parser: "@typescript-eslint/parser",
            parserOptions: {
                ecmaVersion: 2018,
                sourceType: "module",
                project: tsconfig,
            },
            extends: () => this._extends,
            settings: {
                "import/parsers": {
                    "@typescript-eslint/parser": [".ts", ".tsx"],
                },
                "import/resolver": {
                    ...(options.aliasMap && {
                        alias: {
                            map: Object.entries(options.aliasMap).map(([k, v]) => [k, v]),
                            extensions: options.aliasExtensions,
                        },
                    }),
                    node: {},
                    typescript: {
                        project: tsconfig,
                        ...(options.tsAlwaysTryTypes !== false && { alwaysTryTypes: true }),
                    },
                },
            },
            ignorePatterns: this.ignorePatterns,
            rules: () => ({ ...this._formattingRules, ...this.rules }),
            overrides: this.overrides,
        };
        if (options.yaml) {
            new yaml_1.YamlFile(project, ".eslintrc.yml", {
                obj: this.config,
                marker: true,
            });
        }
        else {
            new json_1.JsonFile(project, ".eslintrc.json", {
                obj: this.config,
                // https://eslint.org/docs/latest/user-guide/configuring/configuration-files#comments-in-configuration-files
                marker: true,
                allowComments: true,
            });
        }
        // if the user enabled prettier explicitly _or_ if the project has a
        // `Prettier` component, we shall tweak our configuration accordingly.
        if (options.prettier || prettier_1.Prettier.of(project)) {
            this.enablePrettier();
        }
    }
    /**
     * Returns the singletone Eslint component of a project or undefined if there is none.
     */
    static of(project) {
        const isEslint = (c) => c instanceof Eslint;
        return project.components.find(isEslint);
    }
    /**
     * Add an eslint rule.
     */
    addRules(rules) {
        for (const [k, v] of Object.entries(rules)) {
            this.rules[k] = v;
        }
    }
    /**
     * Adds an eslint plugin
     * @param plugins The names of plugins to add
     */
    addPlugins(...plugins) {
        this._plugins.push(...plugins);
    }
    /**
     * Add an eslint override.
     */
    addOverride(override) {
        this.overrides.push(override);
    }
    /**
     * Do not lint these files.
     */
    addIgnorePattern(pattern) {
        this.ignorePatterns.push(pattern);
    }
    /**
     * Adds an `extends` item to the eslint configuration.
     * @param extendList The list of "extends" to add.
     */
    addExtends(...extendList) {
        this._extends.push(...extendList);
    }
    /**
     * Add a glob file pattern which allows importing dev dependencies.
     * @param pattern glob pattern.
     */
    allowDevDeps(pattern) {
        this._allowDevDeps.add(pattern);
    }
    /**
     * Enables prettier for code formatting.
     */
    enablePrettier() {
        this.nodeProject.addDevDeps("prettier", "eslint-plugin-prettier", "eslint-config-prettier");
        this.addPlugins("prettier");
        this._formattingRules = {
            "prettier/prettier": ["error"],
        };
        this.addExtends("prettier", "plugin:prettier/recommended");
    }
    renderDevDepsAllowList() {
        return Array.from(this._allowDevDeps);
    }
}
exports.Eslint = Eslint;
_a = JSII_RTTI_SYMBOL_1;
Eslint[_a] = { fqn: "projen.javascript.Eslint", version: "0.65.76" };
//# sourceMappingURL=data:application/json;base64,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