"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bundler = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path_1 = require("path");
const component_1 = require("../component");
const dependencies_1 = require("../dependencies");
const util_1 = require("./util");
/**
 * Adds support for bundling JavaScript applications and dependencies into a
 * single file. In the future, this will also supports bundling websites.
 */
class Bundler extends component_1.Component {
    /**
     * Creates a `Bundler`.
     */
    constructor(project, options = {}) {
        super(project);
        this.esbuildVersion = options.esbuildVersion;
        this.bundledir = options.assetsDir ?? "assets";
        this.addToPreCompile = options.addToPreCompile ?? true;
        this.loaders = options.loaders;
    }
    /**
     * Returns the `Bundler` instance associated with a project or `undefined` if
     * there is no Bundler.
     * @param project The project
     * @returns A bundler
     */
    static of(project) {
        const isBundler = (o) => o instanceof Bundler;
        return project.components.find(isBundler);
    }
    /**
     * Gets or creates the singleton "bundle" task of the project.
     *
     * If the project doesn't have a "bundle" task, it will be created and spawned
     * during the pre-compile phase.
     */
    get bundleTask() {
        if (!this._task) {
            this.addBundlingSupport();
            this._task = this.project.tasks.addTask("bundle", {
                description: "Prepare assets",
            });
            // install the bundle task into the pre-compile phase.
            if (this.addToPreCompile) {
                this.project.preCompileTask.spawn(this._task);
            }
        }
        return this._task;
    }
    /**
     * Adds a task to the project which bundles a specific entrypoint and all of
     * its dependencies into a single javascript output file.
     *
     * @param entrypoint The relative path of the artifact within the project
     * @param options Bundling options
     */
    addBundle(entrypoint, options) {
        const name = util_1.renderBundleName(entrypoint);
        const outdir = path_1.join(this.bundledir, name);
        const outfile = path_1.join(outdir, options.outfile ?? "index.js");
        const args = [
            "esbuild",
            "--bundle",
            entrypoint,
            `--target="${options.target}"`,
            `--platform="${options.platform}"`,
            `--outfile="${outfile}"`,
        ];
        const tsconfig = options.tsconfigPath ?? false;
        if (tsconfig) {
            args.push(`--tsconfig="${tsconfig}"`);
        }
        for (const x of options.externals ?? []) {
            args.push(`--external:${x}`);
        }
        const sourcemap = options.sourcemap ?? false;
        if (sourcemap) {
            args.push("--sourcemap");
        }
        const loaders = options.loaders ?? false ? options.loaders : this.loaders ?? false;
        if (loaders) {
            for (let [extension, loader] of Object.entries(loaders)) {
                args.push(`--loader:.${extension}=${loader}`);
            }
        }
        const bundleTask = this.project.addTask(`bundle:${name}`, {
            description: `Create a JavaScript bundle from ${entrypoint}`,
            exec: args.join(" "),
        });
        this.bundleTask.spawn(bundleTask);
        if (options.executable ?? false) {
            bundleTask.exec(`chmod +x ${outfile}`);
        }
        let watchTask;
        const watch = options.watchTask ?? true;
        if (watch) {
            watchTask = this.project.addTask(`bundle:${name}:watch`, {
                description: `Continuously update the JavaScript bundle from ${entrypoint}`,
                exec: `${args.join(" ")} --watch`,
            });
        }
        return {
            bundleTask: bundleTask,
            watchTask: watchTask,
            outdir: outdir,
            outfile: outfile,
        };
    }
    /**
     * Add bundling support to a project. This is called implicitly when
     * `bundleTask` is referenced first. It adds the dependency on `esbuild`,
     * gitignore/npmignore, etc.
     */
    addBundlingSupport() {
        const ignoreEntry = `/${this.bundledir}/`;
        this.project.addGitIgnore(ignoreEntry);
        this.project.addPackageIgnore(`!${ignoreEntry}`); // include in tarball
        const dep = this.esbuildVersion
            ? `esbuild@${this.esbuildVersion}`
            : "esbuild";
        this.project.deps.addDependency(dep, dependencies_1.DependencyType.BUILD);
    }
}
exports.Bundler = Bundler;
_a = JSII_RTTI_SYMBOL_1;
Bundler[_a] = { fqn: "projen.javascript.Bundler", version: "0.65.76" };
//# sourceMappingURL=data:application/json;base64,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