"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CiConfiguration = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const case_1 = require("case");
const component_1 = require("../component");
const yaml_1 = require("../yaml");
/**
 * CI for GitLab.
 * A CI is a configurable automated process made up of one or more stages/jobs.
 * @see https://docs.gitlab.com/ee/ci/yaml/
 */
class CiConfiguration extends component_1.Component {
    constructor(project, name, options) {
        super(project);
        /**
         * Defines default scripts that should run *after* all jobs. Can be overriden by the job level `afterScript`.
         */
        this.defaultAfterScript = [];
        /**
         * Defines default scripts that should run *before* all jobs. Can be overriden by the job level `afterScript`.
         */
        this.defaultBeforeScript = [];
        /**
         * A default list of additional Docker images to run scripts in. The service image is linked to the image specified in the  image parameter.
         */
        this.defaultServices = [];
        /**
         * Used to select a specific runner from the list of all runners that are available for the project.
         */
        this.defaultTags = [];
        /**
         * Can be `Include` or `Include[]`. Each `Include` will be a string, or an
         * object with properties for the method if including external YAML file. The external
         * content will be fetched, included and evaluated along the `.gitlab-ci.yml`.
         */
        this.include = [];
        /**
         * Groups jobs into stages. All jobs in one stage must complete before next stage is
         * executed. Defaults to ['build', 'test', 'deploy'].
         */
        this.stages = [];
        /**
         * Global variables that are passed to jobs.
         * If the job already has that variable defined, the job-level variable takes precedence.
         */
        this.variables = {};
        /**
         * The jobs in the CI configuration.
         */
        this.jobs = {};
        this.project = project;
        this.name = path.parse(name).name;
        this.path =
            this.name === "gitlab-ci"
                ? ".gitlab-ci.yml"
                : `.gitlab/ci-templates/${name.toLocaleLowerCase()}.yml`;
        this.file = new yaml_1.YamlFile(this.project, this.path, {
            obj: () => this.renderCI(),
            // GitLab needs to read the file from the repository in order to work.
            committed: true,
        });
        const defaults = options?.default;
        if (defaults) {
            this.defaultAfterScript.push(...(defaults.afterScript ?? []));
            this.defaultArtifacts = defaults.artifacts;
            defaults.beforeScript &&
                this.defaultBeforeScript.push(...defaults.beforeScript);
            this.defaultCache = defaults.cache;
            this.defaultImage = defaults.image;
            this.defaultInterruptible = defaults.interruptible;
            this.defaultRetry = defaults.retry;
            defaults.services && this.addServices(...defaults.services);
            defaults.tags && this.defaultTags.push(...defaults.tags);
            this.defaultTimeout = defaults.timeout;
        }
        this.pages = options?.pages;
        this.workflow = options?.workflow;
        if (options?.stages) {
            this.addStages(...options.stages);
        }
        if (options?.variables) {
            this.addJobs(options.variables);
        }
        if (options?.jobs) {
            this.addJobs(options.jobs);
        }
    }
    /**
     * Add additional yml/yaml files to the CI includes
     * @param includes The includes to add.
     */
    addIncludes(...includes) {
        for (const additional of includes) {
            this.assertIsValidInclude(additional);
            for (const existing of this.include) {
                if (this.areEqualIncludes(existing, additional)) {
                    throw new Error(`${this.name}: GitLab CI ${existing} already contains one or more templates specified in ${additional}.`);
                }
            }
            this.include.push(additional);
        }
    }
    /**
     * Throw an error if the provided Include is invalid.
     * @see https://gitlab.com/gitlab-org/gitlab/-/blob/master/lib/gitlab/ci/config/external/mapper.rb
     * @param include the Include to validate.
     */
    assertIsValidInclude(include) {
        const combos = [
            include.local,
            include.file && include.project,
            include.remote,
            include.template,
        ];
        const len = combos.filter((x) => Boolean(x)).length;
        if (len !== 1) {
            throw new Error(`${this.name}: GitLab CI include ${include} contains ${len} property combination(s).
        A valid include configuration specifies *one* of the following property combinations.
        * local
        * file, project
        * remote
        * template  
        `);
        }
    }
    /**
     * Check if the equality of Includes.
     * @see https://gitlab.com/gitlab-org/gitlab/-/blob/master/lib/gitlab/ci/config/external/mapper.rb
     * @param x First include to compare.
     * @param y Second include to compare.
     * @returns Whether the includes are equal.
     */
    areEqualIncludes(x, y) {
        if (x.local === y.local && x.local !== undefined) {
            return true;
        }
        else if (x.template === y.template && x.template !== undefined) {
            return true;
        }
        else if (x.remote === y.remote && x.remote !== undefined) {
            return true;
        }
        else if (x.project === y.project && x.ref === y.ref) {
            const xFiles = x.file ? x.file : [];
            const yFiles = y.file ? y.file : [];
            const allFiles = xFiles.concat(yFiles);
            return new Set(allFiles).size !== allFiles.length;
        }
        return false;
    }
    /**
     * Add additional services.
     * @param services The services to add.
     */
    addServices(...services) {
        for (const additional of services) {
            for (const existing of this.defaultServices) {
                if (additional.name === existing.name &&
                    additional.alias === existing.alias) {
                    throw new Error(`${this.name}: GitLab CI already contains service ${additional}.`);
                }
            }
            this.defaultServices.push(additional);
        }
    }
    /**
     * Add a globally defined variable to the CI configuration.
     * @param variables The variables to add.
     */
    addGlobalVariables(variables) {
        for (const [key, value] of Object.entries(variables)) {
            if (this.variables[key] !== undefined) {
                throw new Error(`${this.name}: GitLab CI already contains variable ${key}.`);
            }
            this.variables[key] = value;
        }
    }
    /**
     * Add stages to the CI configuration if not already present.
     * @param stages stages to add.
     */
    addStages(...stages) {
        for (const stage of stages) {
            if (!this.stages.includes(stage)) {
                this.stages.push(stage);
            }
        }
    }
    /**
     * Add jobs and their stages to the CI configuration.
     * @param jobs Jobs to add.
     */
    addJobs(jobs) {
        for (const [key, value] of Object.entries(jobs)) {
            if (this.jobs[key] !== undefined) {
                throw new Error(`${this.name}: GitLab CI already contains job ${key}.`);
            }
            this.jobs[key] = value;
            if (value.stage) {
                this.addStages(value.stage);
            }
        }
    }
    renderCI() {
        return {
            default: this.renderDefault(),
            include: this.include.length > 0 ? snakeCaseKeys(this.include) : undefined,
            pages: snakeCaseKeys(this.pages),
            services: this.defaultServices.length > 0
                ? snakeCaseKeys(this.defaultServices)
                : undefined,
            variables: Object.entries(this.variables).length > 0 ? this.variables : undefined,
            workflow: snakeCaseKeys(this.workflow),
            stages: this.stages.length > 0 ? this.stages : undefined,
            ...snakeCaseKeys(this.jobs),
        };
    }
    renderDefault() {
        const defaults = {
            afterScript: this.defaultAfterScript.length > 0
                ? this.defaultAfterScript
                : undefined,
            artifacts: this.defaultArtifacts,
            beforeScript: this.defaultBeforeScript.length > 0
                ? this.defaultBeforeScript
                : undefined,
            cache: this.defaultCache,
            image: this.defaultImage,
            interruptible: this.defaultInterruptible,
            retry: this.defaultRetry,
            services: this.defaultServices.length > 0 ? this.defaultServices : undefined,
            tags: this.defaultTags.length > 0 ? this.defaultTags : undefined,
            timeout: this.defaultTimeout,
        };
        return Object.values(defaults).filter((x) => x).length
            ? snakeCaseKeys(defaults)
            : undefined;
    }
}
exports.CiConfiguration = CiConfiguration;
_a = JSII_RTTI_SYMBOL_1;
CiConfiguration[_a] = { fqn: "projen.gitlab.CiConfiguration", version: "0.65.76" };
function snakeCaseKeys(obj) {
    if (typeof obj !== "object" || obj == null) {
        return obj;
    }
    if (Array.isArray(obj)) {
        return obj.map(snakeCaseKeys);
    }
    const result = {};
    for (let [k, v] of Object.entries(obj)) {
        if (typeof v === "object" && v != null && k !== "variables") {
            v = snakeCaseKeys(v);
        }
        result[case_1.snake(k)] = v;
    }
    return result;
}
//# sourceMappingURL=data:application/json;base64,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