"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GithubWorkflow = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const case_1 = require("case");
const _resolve_1 = require("../_resolve");
const component_1 = require("../component");
const util_1 = require("../util");
const yaml_1 = require("../yaml");
/**
 * Workflow for GitHub.
 *
 * A workflow is a configurable automated process made up of one or more jobs.
 *
 * @see https://docs.github.com/en/actions/reference/workflow-syntax-for-github-actions
 */
class GithubWorkflow extends component_1.Component {
    constructor(github, name, options = {}) {
        super(github.project);
        this.events = {};
        this.jobs = {};
        this.name = name;
        this.concurrency = options.concurrency;
        this.projenCredentials = github.projenCredentials;
        const workflowsEnabled = github.workflowsEnabled || options.force;
        if (workflowsEnabled) {
            this.file = new yaml_1.YamlFile(this.project, `.github/workflows/${name.toLocaleLowerCase()}.yml`, {
                obj: () => this.renderWorkflow(),
                // GitHub needs to read the file from the repository in order to work.
                committed: true,
            });
        }
    }
    /**
     * Add events to triggers the workflow.
     *
     * @param events The event(s) to trigger the workflow.
     */
    on(events) {
        this.events = {
            ...this.events,
            ...events,
        };
    }
    /**
     * Adds a single job to the workflow.
     * @param id The job name (unique within the workflow)
     * @param job The job specification
     */
    addJob(id, job) {
        this.addJobs({ [id]: job });
    }
    /**
     * Add jobs to the workflow.
     *
     * @param jobs Jobs to add.
     */
    addJobs(jobs) {
        // verify that job has a "permissions" statement to ensure workflow can
        // operate in repos with default tokens set to readonly
        for (const [id, job] of Object.entries(jobs)) {
            if (!job.permissions) {
                throw new Error(`${id}: all workflow jobs must have a "permissions" clause to ensure workflow can operate in restricted repositories`);
            }
        }
        // verify that job has a "runsOn" statement to ensure a worker can be selected appropriately
        for (const [id, job] of Object.entries(jobs)) {
            if (!("uses" in job)) {
                if ("runsOn" in job && job.runsOn.length === 0) {
                    throw new Error(`${id}: at least one runner selector labels must be provided in "runsOn" to ensure a runner instance can be selected`);
                }
            }
        }
        this.jobs = {
            ...this.jobs,
            ...jobs,
        };
    }
    renderWorkflow() {
        return {
            name: this.name,
            on: snakeCaseKeys(this.events),
            concurrency: this.concurrency,
            jobs: renderJobs(this.jobs),
        };
    }
}
exports.GithubWorkflow = GithubWorkflow;
_a = JSII_RTTI_SYMBOL_1;
GithubWorkflow[_a] = { fqn: "projen.github.GithubWorkflow", version: "0.65.76" };
function snakeCaseKeys(obj) {
    if (typeof obj !== "object" || obj == null) {
        return obj;
    }
    if (Array.isArray(obj)) {
        return obj.map(snakeCaseKeys);
    }
    const result = {};
    for (let [k, v] of Object.entries(obj)) {
        if (typeof v === "object" && v != null) {
            v = snakeCaseKeys(v);
        }
        result[case_1.snake(k)] = v;
    }
    return result;
}
function renderJobs(jobs) {
    const result = {};
    for (const [name, job] of Object.entries(jobs)) {
        result[name] = renderJob(job);
    }
    return result;
    /** @see https://docs.github.com/en/actions/reference/workflow-syntax-for-github-actions */
    function renderJob(job) {
        const steps = new Array();
        // https://docs.github.com/en/actions/using-workflows/reusing-workflows#supported-keywords-for-jobs-that-call-a-reusable-workflow
        if ("uses" in job) {
            return {
                name: job.name,
                needs: arrayOrScalar(job.needs),
                if: job.if,
                permissions: util_1.kebabCaseKeys(job.permissions),
                concurrency: job.concurrency,
                uses: job.uses,
                with: job.with,
                secrets: job.secrets,
            };
        }
        if (job.tools) {
            steps.push(...setupTools(job.tools));
        }
        const userDefinedSteps = util_1.kebabCaseKeys(_resolve_1.resolve(job.steps), false);
        steps.push(...userDefinedSteps);
        return {
            name: job.name,
            needs: arrayOrScalar(job.needs),
            "runs-on": arrayOrScalar(job.runsOn),
            permissions: util_1.kebabCaseKeys(job.permissions),
            environment: job.environment,
            concurrency: job.concurrency,
            outputs: renderJobOutputs(job.outputs),
            env: job.env,
            defaults: util_1.kebabCaseKeys(job.defaults),
            if: job.if,
            steps: steps.map(renderStep),
            "timeout-minutes": job.timeoutMinutes,
            strategy: renderJobStrategy(job.strategy),
            "continue-on-error": job.continueOnError,
            container: job.container,
            services: job.services,
        };
    }
    function renderJobOutputs(output) {
        if (output == null) {
            return undefined;
        }
        const rendered = {};
        for (const [name, { stepId, outputName }] of Object.entries(output)) {
            rendered[name] = `\${{ steps.${stepId}.outputs.${outputName} }}`;
        }
        return rendered;
    }
    function renderJobStrategy(strategy) {
        if (strategy == null) {
            return undefined;
        }
        const rendered = {
            "max-parallel": strategy.maxParallel,
            "fail-fast": strategy.failFast,
        };
        if (strategy.matrix) {
            const matrix = {
                include: strategy.matrix.include,
                exclude: strategy.matrix.exclude,
            };
            for (const [key, values] of Object.entries(strategy.matrix.domain ?? {})) {
                if (key in matrix) {
                    // A domain key was set to `include`, or `exclude`:
                    throw new Error(`Illegal job strategy matrix key: ${key}`);
                }
                matrix[key] = values;
            }
            rendered.matrix = matrix;
        }
        return rendered;
    }
    function renderStep(step) {
        return {
            name: step.name,
            id: step.id,
            if: step.if,
            uses: step.uses,
            env: step.env,
            run: step.run,
            with: step.with,
            "continue-on-error": step.continueOnError,
            "timeout-minutes": step.timeoutMinutes,
            "working-directory": step.workingDirectory,
        };
    }
}
function arrayOrScalar(arr) {
    if (arr == null || arr.length === 0) {
        return arr;
    }
    if (arr.length === 1) {
        return arr[0];
    }
    return arr;
}
function setupTools(tools) {
    const steps = [];
    if (tools.java) {
        steps.push({
            uses: "actions/setup-java@v3",
            with: { distribution: "temurin", "java-version": tools.java.version },
        });
    }
    if (tools.node) {
        steps.push({
            uses: "actions/setup-node@v3",
            with: { "node-version": tools.node.version },
        });
    }
    if (tools.python) {
        steps.push({
            uses: "actions/setup-python@v4",
            with: { "python-version": tools.python.version },
        });
    }
    if (tools.go) {
        steps.push({
            uses: "actions/setup-go@v3",
            with: { "go-version": tools.go.version },
        });
    }
    if (tools.dotnet) {
        steps.push({
            uses: "actions/setup-dotnet@v3",
            with: { "dotnet-version": tools.dotnet.version },
        });
    }
    return steps;
}
//# sourceMappingURL=data:application/json;base64,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