"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GithubCredentials = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Represents a method of providing GitHub API access for projen workflows.
 */
class GithubCredentials {
    constructor(options) {
        this.options = options;
    }
    /**
     * Provide API access through a GitHub personal access token.
     *
     * The token must be added as a secret to the GitHub repo, and the name of the
     * secret can be specified here.
     *
     * @see https://docs.github.com/en/authentication/keeping-your-account-and-data-secure/creating-a-personal-access-token
     * @default - a secret named "PROJEN_GITHUB_TOKEN"
     */
    static fromPersonalAccessToken(options = {}) {
        return new GithubCredentials({
            setupSteps: [],
            tokenRef: `\${{ secrets.${options.secret ?? "PROJEN_GITHUB_TOKEN"} }}`,
        });
    }
    /**
     * Provide API access through a GitHub App.
     *
     * The GitHub App must be installed on the GitHub repo, its App ID and a
     * private key must be added as secrets to the repo. The name of the secrets
     * can be specified here.
     *
     * @see https://docs.github.com/en/developers/apps/building-github-apps/creating-a-github-app
     * @see https://projen.io/github.html#github-app
     * @default - app id stored in "PROJEN_APP_ID" and private key stored in "PROJEN_APP_PRIVATE_KEY"
     */
    static fromApp(options = {}) {
        const appIdSecret = options.appIdSecret ?? "PROJEN_APP_ID";
        const privateKeySecret = options.privateKeySecret ?? "PROJEN_APP_PRIVATE_KEY";
        return new GithubCredentials({
            setupSteps: [
                {
                    name: "Generate token",
                    id: "generate_token",
                    uses: "tibdex/github-app-token@7ce9ffdcdeb2ba82b01b51d6584a6a85872336d4",
                    with: {
                        app_id: `\${{ secrets.${appIdSecret} }}`,
                        private_key: `\${{ secrets.${privateKeySecret} }}`,
                    },
                },
            ],
            tokenRef: "${{ steps.generate_token.outputs.token }}",
        });
    }
    /**
     * Setup steps to obtain GitHub credentials.
     */
    get setupSteps() {
        return [...this.options.setupSteps];
    }
    /**
     * The value to use in a workflow when a GitHub token is expected. This
     * typically looks like "${{ some.path.to.a.value }}".
     */
    get tokenRef() {
        return this.options.tokenRef;
    }
}
exports.GithubCredentials = GithubCredentials;
_a = JSII_RTTI_SYMBOL_1;
GithubCredentials[_a] = { fqn: "projen.github.GithubCredentials", version: "0.65.76" };
//# sourceMappingURL=data:application/json;base64,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