"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DependencyType = exports.Dependencies = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const common_1 = require("./common");
const component_1 = require("./component");
const json_1 = require("./json");
/**
 * The `Dependencies` component is responsible to track the list of dependencies
 * a project has, and then used by project types as the model for rendering
 * project-specific dependency manifests such as the dependencies section
 * `package.json` files.
 *
 * To add a dependency you can use a project-type specific API such as
 * `nodeProject.addDeps()` or use the generic API of `project.deps`:
 */
class Dependencies extends component_1.Component {
    /**
     * Adds a dependencies component to the project.
     * @param project The parent project
     */
    constructor(project) {
        super(project);
        this._deps = new Array();
        // this is not really required at the moment, but actually quite useful as a
        // checked-in source of truth for dependencies and will potentially be
        // valuable in the future for CLI tools.
        if (!project.ejected) {
            new json_1.JsonFile(project, Dependencies.MANIFEST_FILE, {
                omitEmpty: true,
                obj: () => this.toJson(),
            });
        }
    }
    /**
     * Returns the coordinates of a dependency spec.
     *
     * Given `foo@^3.4.0` returns `{ name: "foo", version: "^3.4.0" }`.
     * Given `bar@npm:@bar/legacy` returns `{ name: "bar", version: "npm:@bar/legacy" }`.
     */
    static parseDependency(spec) {
        const scope = spec.startsWith("@");
        if (scope) {
            spec = spec.substr(1);
        }
        const [module, ...version] = spec.split("@");
        const name = scope ? `@${module}` : module;
        if (version.length == 0) {
            return { name };
        }
        else {
            return { name, version: version?.join("@") };
        }
    }
    /**
     * A copy of all dependencies recorded for this project.
     *
     * The list is sorted by type->name->version
     */
    get all() {
        return [...this._deps].sort(compareDeps).map(normalizeDep);
    }
    /**
     * Returns a dependency by name.
     *
     * Fails if there is no dependency defined by that name or if `type` is not
     * provided and there is more then one dependency type for this dependency.
     *
     * @param name The name of the dependency
     * @param type The dependency type. If this dependency is defined only for a
     * single type, this argument can be omitted.
     *
     * @returns a copy (cannot be modified)
     */
    getDependency(name, type) {
        const dep = this.tryGetDependency(name, type);
        if (!dep) {
            const msg = type
                ? `there is no ${type} dependency defined on "${name}"`
                : `there is no dependency defined on "${name}"`;
            throw new Error(msg);
        }
        return dep;
    }
    /**
     * Returns a dependency by name.
     *
     * Returns `undefined` if there is no dependency defined by that name or if
     * `type` is not provided and there is more then one dependency type for this
     * dependency.
     *
     * @param name The name of the dependency
     * @param type The dependency type. If this dependency is defined only for a
     * single type, this argument can be omitted.
     *
     * @returns a copy (cannot be modified) or undefined if there is no match
     */
    tryGetDependency(name, type) {
        const idx = this.tryGetDependencyIndex(name, type);
        if (idx === -1) {
            return undefined;
        }
        return {
            ...normalizeDep(this._deps[idx]),
        };
    }
    /**
     * Adds a dependency to this project.
     * @param spec The dependency spec in the format `MODULE[@VERSION]` where
     * `MODULE` is the package-manager-specific module name and `VERSION` is an
     * optional semantic version requirement (e.g. `^3.4.0`).
     * @param type The type of the dependency.
     */
    addDependency(spec, type, metadata = {}) {
        this.project.logger.debug(`${type}-dep ${spec}`);
        const dep = {
            ...Dependencies.parseDependency(spec),
            type,
            metadata,
        };
        const existingDepIndex = this.tryGetDependencyIndex(dep.name, type);
        if (existingDepIndex !== -1) {
            this.project.logger.debug(`updating existing ${dep.type}-dep ${dep.name} with more specific version/metadata`);
            this._deps[existingDepIndex] = dep;
        }
        else {
            this._deps.push(dep);
        }
        return dep;
    }
    /**
     * Removes a dependency.
     * @param name The name of the module to remove (without the version)
     * @param type The dependency type. This is only required if there the
     * dependency is defined for multiple types.
     */
    removeDependency(name, type) {
        const removeIndex = this.tryGetDependencyIndex(name, type);
        if (removeIndex === -1) {
            return;
        }
        this._deps.splice(removeIndex, 1);
    }
    tryGetDependencyIndex(name, type) {
        const deps = this._deps.filter((d) => d.name === name);
        if (deps.length === 0) {
            return -1; // not found
        }
        if (!type) {
            if (deps.length > 1) {
                throw new Error(`"${name}" is defined for multiple dependency types: ${deps
                    .map((d) => d.type)
                    .join(",")}. Please specify dependency type`);
            }
            type = deps[0].type;
        }
        return this._deps.findIndex((dep) => dep.name === name && dep.type === type);
    }
    toJson() {
        if (this._deps.length === 0) {
            return undefined;
        }
        return {
            dependencies: this._deps.sort(compareDeps).map(normalizeDep),
        };
    }
}
exports.Dependencies = Dependencies;
_a = JSII_RTTI_SYMBOL_1;
Dependencies[_a] = { fqn: "projen.Dependencies", version: "0.65.76" };
/**
 * The project-relative path of the deps manifest file.
 */
Dependencies.MANIFEST_FILE = path.posix.join(common_1.PROJEN_DIR, "deps.json");
function normalizeDep(d) {
    const obj = {};
    for (const [k, v] of Object.entries(d)) {
        if (v == undefined) {
            continue;
        }
        if (typeof v === "object" && Object.keys(v).length === 0) {
            continue;
        }
        if (Array.isArray(v) && v.length === 0) {
            continue;
        }
        obj[k] = v;
    }
    return obj;
}
function compareDeps(d1, d2) {
    return specOf(d1).localeCompare(specOf(d2));
    function specOf(dep) {
        let spec = dep.type + ":" + dep.name;
        if (dep.version) {
            spec += "@" + dep.version;
        }
        return spec;
    }
}
/**
 * Type of dependency.
 */
var DependencyType;
(function (DependencyType) {
    /**
     * The dependency is required for the program/library during runtime.
     */
    DependencyType["RUNTIME"] = "runtime";
    /**
     * The dependency is required at runtime but expected to be installed by the
     * consumer.
     */
    DependencyType["PEER"] = "peer";
    /**
     * The dependency is bundled and shipped with the module, so consumers are not
     * required to install it.
     */
    DependencyType["BUNDLED"] = "bundled";
    /**
     * The dependency is required to run the `build` task.
     */
    DependencyType["BUILD"] = "build";
    /**
     * The dependency is required to run the `test` task.
     */
    DependencyType["TEST"] = "test";
    /**
     * The dependency is required for development (e.g. IDE plugins).
     */
    DependencyType["DEVENV"] = "devenv";
    /**
     * Transient dependency that needs to be overwritten.
     *
     * Available for Node packages
     */
    DependencyType["OVERRIDE"] = "override";
})(DependencyType = exports.DependencyType || (exports.DependencyType = {}));
//# sourceMappingURL=data:application/json;base64,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