"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaExtension = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path_1 = require("path");
const case_1 = require("case");
const component_1 = require("../component");
const javascript_1 = require("../javascript");
const source_code_1 = require("../source-code");
const internal_1 = require("./internal");
const lambda_function_1 = require("./lambda-function");
/**
 * Create a Lambda Extension
 */
class LambdaExtension extends component_1.Component {
    constructor(project, options) {
        super(project);
        const basePath = path_1.join(path_1.dirname(options.entrypoint), path_1.basename(options.entrypoint, internal_1.TYPESCRIPT_LAMBDA_EXTENSION_EXT));
        const name = options.name ?? path_1.basename(basePath);
        const bundler = javascript_1.Bundler.of(project);
        if (!bundler) {
            throw new Error("No bundler found. Please add a Bundler component to your project.");
        }
        const compatibleRuntimes = options.compatibleRuntimes ?? [
            lambda_function_1.LambdaRuntime.NODEJS_18_X,
            lambda_function_1.LambdaRuntime.NODEJS_16_X,
            lambda_function_1.LambdaRuntime.NODEJS_14_X,
            lambda_function_1.LambdaRuntime.NODEJS_12_X,
        ];
        if (compatibleRuntimes.length === 0) {
            throw new Error("Compatible runtimes must include at least one runtime");
        }
        // Use the lowest runtime version to bundle
        const [bundlerRuntime] = compatibleRuntimes.sort((a, b) => a.functionRuntime.localeCompare(b.functionRuntime));
        // Allow extension code to import dev-deps since they are only needed
        // during bundling
        const eslint = javascript_1.Eslint.of(project);
        eslint?.allowDevDeps(options.entrypoint);
        const bundle = bundler.addBundle(options.entrypoint, {
            platform: bundlerRuntime.esbuildPlatform,
            target: bundlerRuntime.esbuildTarget,
            externals: ["aws-sdk"],
            outfile: `extensions/${name}`,
            // Make the output executable because Lambda expects to run
            // extensions as stand-alone programs alongside the main lambda
            // process.
            executable: true,
            ...options.bundlingOptions,
        });
        const constructFile = options.constructFile ?? `${basePath}-layer-version.ts`;
        new LambdaLayerConstruct(project, {
            constructFile: constructFile,
            constructName: options.constructName,
            assetDir: bundle.outdir,
            compatibleRuntimes: compatibleRuntimes,
            description: `Provides a Lambda Extension \`${name}\` from ${internal_1.convertToPosixPath(options.entrypoint)}`,
            cdkDeps: options.cdkDeps,
        });
    }
}
exports.LambdaExtension = LambdaExtension;
_a = JSII_RTTI_SYMBOL_1;
LambdaExtension[_a] = { fqn: "projen.awscdk.LambdaExtension", version: "0.65.76" };
class LambdaLayerConstruct extends source_code_1.SourceCode {
    constructor(project, options) {
        super(project, options.constructFile);
        const src = this;
        const cdkDeps = options.cdkDeps;
        const constructName = options.constructName ?? case_1.pascal(path_1.basename(options.constructFile, ".ts"));
        const propsType = `${constructName}Props`;
        const assetDir = path_1.relative(path_1.dirname(options.constructFile), options.assetDir);
        if (src.marker) {
            src.line(`// ${src.marker}`);
        }
        src.line("import * as path from 'path';");
        if (cdkDeps.cdkMajorVersion === 1) {
            src.line("import * as lambda from '@aws-cdk/aws-lambda';");
            src.line("import { Construct } from '@aws-cdk/core';");
            cdkDeps.addV1Dependencies("@aws-cdk/aws-lambda");
            cdkDeps.addV1Dependencies("@aws-cdk/core");
        }
        else {
            src.line("import * as lambda from 'aws-cdk-lib/aws-lambda';");
            src.line("import { Construct } from 'constructs';");
        }
        src.line();
        src.line("/**");
        src.line(` * Props for ${constructName}`);
        src.line(" */");
        src.open(`export interface ${propsType} extends lambda.LayerVersionOptions {`);
        src.close("}");
        src.line();
        src.line("/**");
        src.line(` * ${options.description}`);
        src.line(" */");
        src.open(`export class ${constructName} extends lambda.LayerVersion {`);
        src.open(`constructor(scope: Construct, id: string, props?: ${propsType}) {`);
        src.open("super(scope, id, {");
        src.line(`description: ${encodeCodeString(options.description)},`);
        src.line("...props,");
        src.open("compatibleRuntimes: [");
        for (const runtime of options.compatibleRuntimes) {
            src.line(`new lambda.Runtime('${runtime.functionRuntime}', lambda.RuntimeFamily.NODEJS),`);
        }
        src.close("],");
        src.open(`code: lambda.Code.fromAsset(path.join(__dirname,`);
        src.line(`${encodeCodeString(internal_1.convertToPosixPath(assetDir))})),`);
        src.close();
        src.close("});");
        src.close("}");
        src.close("}");
    }
}
/**
 * Encodes a string for embedding in source code.
 */
function encodeCodeString(value) {
    const json = JSON.stringify(value);
    const escapedString = json.substring(1, json.length - 1).replace(/'/g, "\\'");
    return `'${escapedString}'`;
}
//# sourceMappingURL=data:application/json;base64,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