# Making FermiLAT gamma-ray light curves with wtlike 
> Quickly create a light curve for any 4FGL source, on any time scale, with optional Bayesian Block analysis 


#### github links:  
[this document](https://tburnett.github.io/wtlike/),   [the repository](https://github.com/tburnett/wtlike)

## Introduction
`wtlike`(Perhaps pronounced "DUB-Tee-like"), is a library optimized for interactive exploration in a [Jupyter notebook](https://jupyter.org/) with access to all Fermi-LAT data, and to analyze the time dependence of any source in the
4FGL catalog on any time scale, with the option of performing a [Bayesian Block](https://arxiv.org/pdf/1207.5578.pdf)  partition to select optimal time intervals. The source can be identified by the 4FGL name, or any equivalent common name.

Here is a minimal demo:

```python
from wtlike import *
config = Config()
valid = config.valid
if valid:
    with Timer() as t:
        wtl = WtLike('3C 273',week_range=(9,3*52+9))
        print(t)
```

    SourceData:  4FGL J1229.0+0202: Restoring from cache with key "P88Y3157_weeks_9-165"
    SourceData: Source 3C 273 with:
    	 data:        33,210 photons from 2008-08-04 to 2011-08-03
    	 exposure:   713,320 intervals,  average effective area 2783 cm^2 for 21.3 Ms
    	 rates:  source 2.02e-07/s, background 3.58e-07/s,
    CellData.rebin: Bin photon data into 156 1-week bins from 54683.0 to 55775.0
    LightCurve: select 156 cells for fitting with e>35 & n>2
    elapsed time: 0.7s (0.0 min)


This created a `WtLike` object, loading the first 3 years of data, by specifying weeks from #9, the first data-taking week.
The reason to specify only the first three years here is to avoid the 10 min or so that extracting the furl 13+ years would take for this demo. If that had already been done, then using an included [cache system](https://tburnett.github.io/wtlike/config.html#Cache), it only has to be done once.

Now ask it to make a plot:

```python
if config.valid: 
    wtl.plot(UTC=1);
```


    
![png](docs/images/output_3_0.png)
    


This assumes that the name for the source, in this case the historically famous first [quasar](https://en.wikipedia.org/wiki/Quasar#Background) to be discovered, can be associated with a 4FGL catalog source. The plot shows, as a function of UTC (or MJD if desired) time, weekly measurements of deviations of the flux relative to the average of the 12-year interval used to define the 4FGL-DR3 catalog.



## Overview

This package has code that was developed with the [nbdev](https://nbdev.fast.ai/) code/tests/documentation environment from the [github package lat-timing](https://github.com/tburnett/lat-timing) to generate light curves of Fermi-LAT sources.  
It is based on a [paper](https://arxiv.org/pdf/1910.00140.pdf) by Matthew Kerr, which derives the [weighted likelihood formalism](https://tburnett.github.io/wtlike/loglike#The-Kerr-likelihood-formula) used here, specifically with
the [Bayesian Block](https://arxiv.org/pdf/1207.5578.pdf) to detect and characterize variability of a gamma-ray source.

There are several innovative design features that significantly improve the speed and portability.

* Condensed photon and spacecraft data. 
* Weight tables
* A cache to improve interactivity
* Likelihood functions fit to a modified Poisson function
* Unbinned likelihood 
* A simple user interface


### How it works

Historically, gamma-ray source measurements have used two methods:
1. For a fixed time interval, where the energy and 
position are used, with a model including all potential sources and a model of the detector response to 
define the likelihood. This has been the only way to study weak sources. A light curve must apply this to
each time interval.
2. Or, for very bright flares, for example GRBs or AGN flares, one can simply count the number of photons within a
circular region, that is, aperture photometry.

Matthew Kerr [introduced](https://arxiv.org/pdf/1910.00140.pdf) a third method, basically counting photons but using information from a static
likelihood analysis to assign a "weight" to each photon, the probability for being from the source in question, then optimizing this likelihood. This assumes that the only thing changing is the flux of
the source.  He calls it "retrospective", since the analysis for the full time is then applied back to the individual photons. Another way of looking at it is to make the assumption that the time dependence of a source's photon flux factorizes according to the energy and spatitial dependences.  

### Likelihood evaluation

In a significant modification from Kerr's implemetation as described in that paper, we evaluate weights for each photon by a table lookup.

Assumptions:
* Source photons are completely contained in the dataset boundaries (the "ROI").
* The instrument response is constant with time.
* The background pattern is constant. (Clearly violated if a neighboring source varies!)

The likelihood evaluation is implemented in the module  [loglike](https://tburnett.github.io/wtlike/loglike).

### Photon Data

*Fermi* data are retrieved from the Fermi-LAT weekly data files extracted from the [GSFC FTP server](https://heasarc.gsfc.nasa.gov/FTP/fermi/data/lat/weekly), 
with subfolders for the photon data, `photon` and spacecraft data, `spacecraft`. It is [described here](http://fermi.gsfc.nasa.gov/ssc/data/access/http://fermi.gsfc.nasa.gov/ssc/data/access/). The files are organized into individual weeks, starting UTC midnight each Thursday. Files for the most recent week are updated daily.

We convert each pair of photon and spacecraft files to two DataFrame tables with the minimum information needed to evaluate the likelihood, as compressed as possible. Particularly, the energy and position are binned. Details can be seen in the module [data_man](https://tburnett.github.io/wtlike/data_man).  

The entire data set (SOURCE class, energy>100 MeV) in this format occupies ~2 GB.

### Select Data for a source

All further processing uses a subset of the photons within a cone, currently $4^\circ$, about the selected source, and 
evaluates the exposure during 30-s intervals for this direction.  In the class
[`SourceData`](https://tburnett.github.io/wtlike/source_data#SourceData), implemented in the module 
[`source_data`](https://tburnett.github.io/wtlike/source_data) we
1. Extract photons
2. Evaluate exposure, using the effective area tables and a spectral assumption.
3. Determine the weight for each photon, using the table for the source. See the module [weights](https://tburnett.github.io/wtlike/weights) for details.

The result is a photon DataFrame, containing for each photon, the time $t$ in MJD units, $w$.

This class is a superclass of the user interface class `WtLike` introduced above.

### Partition into cells

A "cell", the terminology used by Kerr, the set of photons in a specific time interval. The class 
[CellData](https://tburnett.github.io/wtlike/cell_data#CellData)
in the module [cell_data](https://tburnett.github.io/wtlike/cell_data), a subclass of SourceData manages this.

This class is instantiated with a tuple to define the binning in time. Denoted by `(a, b, c)`, the elements are:
* `a`--start time 
* `b`--stop time
* `c`-- bin size in days, but 0 means orbit-based, intervals are contiguous eposure.

For the start and stop, values > 50000 are interpreted as MJD. Otherwise they are relative to start if positive
or stop if negative for the full dataset, both rounded to a full day. Zero means actual start for `a` and stop for `b`.
The default binning is simply `(0, 0, 7)` for weekly bins with the full dataset. Hourly for the most recent week would be `(-7, 0, 1/24)`.

A DataFrame table of the cells is created as a data member `cells`, with content

* `t` -- MJD time at cell center
* `tw` -- cell width in days
* `e` -- cell exposure, for reference 
* `n` -- the number of photons
* `w` -- a list of `n` weights
* `S` -- expected number source photons, the nominal source rate times the total exposure.
* `B` -- expected number of background photons (unused)

### Views

`CellData` implements a method `view`, which accepts a binning tuple as an argument, returns a *copy* of the current object, which can be a subclass, assigning to it the binning. Thus the view has all the attributes of its parent, but
with a different set of cells. 

So the following creates a new WtLike object that we generated above, rebins a copy with 25-day bins in the first 100 days, generates a list of the cells, then removes it since it wasn't assigned a reference variable.


```python
if config.valid:
    print(wtl.view(0,100,25).cells)
```

    CellData.rebin: Bin photon data into 4 25-day bins from 54683.0 to 54783.0
    LightCurve: select 4 cells for fitting with e>125 & n>2
             t    tw            e     n  \
    0  54695.5  25.0  1488.182373   553   
    1  54720.5  25.0  1868.273926  1438   
    2  54745.5  25.0  1450.828979  1183   
    3  54770.5  25.0  1929.570801  1175   
    
                                                       w           S           B  
    0  [4.8901367e-01, 0.8066406, 0.11303711, 0.19165...  300.871887  532.046936  
    1  [0.4345703, 0.6064453, 0.0690918, 0.056274414,...  377.716553  667.935242  
    2  [0.33911133, 0.3100586, 0.7089844, 0.06994629,...  293.320007  518.692566  
    3  [0.09112549, 0.58251953, 0.07537842, 0.3457031...  390.109192  689.849792  


### Evaluate Likelihoods and make light curve plots

The class [`LightCurve`](https://tburnett.github.io/wtlike/lightcurve), implemented in the module [`lightcurve`](https://tburnett.github.io/wtlike#LightCurve) is a subclass of `SourceData`.
An instance invokes its superclass to generate the set of cells, then evaluates the likelihoods.

#### Poisson-like Likelihood
We fit the likelihood for each cell, using only a few evaluations, to a [3-parameter Poisson-like formula](https://tburnett.github.io/poisson). Two advantages of this are:
* efficiency -- for large numbers of photons, this is much faster
* convenience -- the [`Poisson`](https://tburnett.github.io/wtlike/poisson.html#Poisson) object implements functions that return the TS, 95% CL, maximum, and uncertainties, using the [incomplete gamma function](https://docs.scipy.org/doc/scipy/reference/generated/scipy.special.gammainc.html).

This results in a DataFrame `fits` containing
* `t, tw, n`, from the cell, and
* `fit`, the `Poisson` object

#### Light curve plots

The function `Likelihood.plot` actually invokes [`flux_plot`](https://tburnett.github.io/lightcurve#flux_plot)

### Apply Bayesian Blocks
The class [`WtLike`](https://tburnett.github.io/wtlike/90-main.html#WtLike), 
implememted in the module [`main`](https://tburnett.github.io/wtlike/90-main.html), adds an implementation of the Bayesian Block (BB) algorithm, from the module [`bayesian`](https://tburnett.github.io/wtlike)  using likelihood instead of counts.
There we have two subclasses of `astropy.stats.bayesian_blocks`, [`CountFitness`](https://tburnett.github.io/wtlike/bayesian.html#CountFitness) and the default [`LikelihoodFitness`](https://tburnett.github.io/wtlike/bayesian.html#LikelihoodFitness).
    
This code creates partitions between boundaries of a set of cells. Usage is via a special view, 
[bb_view`](https://tburnett.github.io/wtlike/main#WtLike.bb_view)
                     

```python
if config.valid:
    bb = wtl.bb_view()
    bb.plot(UTC=1);
```

    Bayesian Blocks: partitioning 156 cells using LikelihoodFitness with penalty 5%
    	found 43 / 156 blocks.
    LightCurve: Loaded 43 / 43 cells for fitting



    
![png](docs/images/output_14_1.png)
    


As you see, this made 94 blocks from the 656 weeks, fit each, and overplotted in on the weekly light curve.
<details class="description">
    <summary>Code details ...</summary>
    
```python
# collapse-hide

@ipynb_doc
def pgram(wtl):
    """
    ### Kerr Periodogram

    One can analyze the frequency spectrum by generating a 
    [Kerr periodogram](https://arxiv.org/pdf/1910.00140.pdf). 
    A `WtLike` object has a `periodogram` function that returns a `TimeSeries` object.
    {out}
    {fig}
    """
    with capture_hide('Setup output') as out:
        p =wtl.periodogram()
        p.power_plot(pmax=100 );
    fig = plt.gcf()
    return locals()

if config.valid: pgram(wtl)
```

</details>


### Kerr Periodogram

One can analyze the frequency spectrum by generating a 
[Kerr periodogram](https://arxiv.org/pdf/1910.00140.pdf). 
A `WtLike` object has a `periodogram` function that returns a `TimeSeries` object.
<details  class="nbdoc-description" >  <summary> Setup output </summary>  <div style="margin-left: 5%;"><pre>CellData.rebin: Bin photon data into 26256 1-hour bins from 54683.0 to 55777.0<br></pre></div> </details>
<figure style="margin-left: 5%" title="Figure 1">   <img src="data:image/png;base64,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" alt="Figure 1"  <br> </figure>


<details class="description">
    <summary>Code details ...</summary>
    
```python
# collapse-hide

@ipynb_doc
def aitdemo(week, nside, name='3C 454.3', size=40):
    """
    ## Aitoff and ZEA projections of a HEALPix map
    
    Choose week {week}, with a large 3C 454.3 flare.
    
    The all-sky Aitoff image of the photon count map, counts per nside={nside} pixel.
    {fig1}
    
    And zoom to small region, with a ZEA projection, a ${size}^\circ$ square.
    {fig2}
    """
    from wtlike.skymaps import  SquareWCS
    from wtlike.sources import findsource
    from utilities import healpix as hpx
    
    cntmap = DataView( FermiInterval(7)[week-1], nside=nside).count_map()
    
    kw = dict(vmin=1, vmax=100, )
    fig1 = hpx.HPmap(cntmap, f'week {week}', unit='counts/pixel', sigma=0.5).ait_plot(**kw);
    
    wcs = SquareWCS(findsource(name, gal=True), size=size,)
    fig2 = wcs.plot(cntmap, title=name,  cb_kw=dict( shrink=0.8), **kw)
    return locals()


week=79
nside=64

if valid: aitdemo(week, nside, fignum=1)
```

</details>


## Aitoff and ZEA projections of a HEALPix map

Choose week 79, with a large 3C 454.3 flare.

The all-sky Aitoff image of the photon count map, counts per nside=64 pixel.
<figure style="margin-left: 5%" title="Figure 1">   <img src="data:image/png;base64,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" alt="Figure 1"  <br> </figure>

And zoom to small region, with a ZEA projection, a $40^\circ$ square.
<figure style="margin-left: 5%" title="Figure 2">   <img src="data:image/png;base64,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" alt="Figure 2"  <br> </figure>



## Simulation

Finally, a simulation option is available. See the [tutorial](https://tburnett.github.io/wtlike/tutorial/)  for an example

## Installation

Note that this is in beta mode. 
It depends on the packages:  matplotlib, pandas, scipy, astropy, scipy and healpy, which pip will install if needed.

To install from pyPI:

```
pip install wtlike
```
or to upgrade:
```
pip install -U wtlike
```


## Input data

There are three data sources which `wtlike` needs to function:

-	The photon/spacecraft data, one file per week
-	A table of weights for each source
-	An effective area table, a grid in energy/angle for "front" and "back" events 

These must be found in a folder, which by default is `~/wtlike-data`. In that folder there must be (perhaps links to) three folders named `data_files`, `weight_files`, `aeff_files`, and a zip file, `weight_file.zip`.  A current copy can be found in my [Google Drive](https://drive.google.com/drive/folders/1YxQRpZ4MI945-GZ2bK3CgTo6ViqoQ2Xt?lfhs=2).

## Google colab setup

An easy way to test, or even use, this system is with [Goggle's colab](https://colab.research.google.com/). To use it, you need a Google account.
All the data files are in my shared folder [wtlike-data](https://drive.google.com/drive/folders/1YxQRpZ4MI945-GZ2bK3CgTo6ViqoQ2Xt?lfhs=2).
It includes a [setup notebook](https://colab.research.google.com/drive/1Dy29VRE3M28W_uMJruDxrBElFpyg0Iyl) which demonstrates setting up a free Google-hosted VM to run wtlike code and makes a few demo plots.
