#!/usr/bin/python
"""
    Copyright (c) 2016-present,
    Jaguar0625, gimre, BloodyRookie, Tech Bureau, Corp. All rights reserved.

    This file is part of Catapult.

    Catapult is free software: you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Catapult is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with Catapult. If not, see <http://www.gnu.org/licenses/>.
"""

from __future__ import annotations
from .BlockFeeMultiplierDto import BlockFeeMultiplierDto
from .DifficultyDto import DifficultyDto
from .EntityTypeDto import EntityTypeDto
from .GeneratorUtils import GeneratorUtils
from .Hash256Dto import Hash256Dto
from .HeightDto import HeightDto
from .KeyDto import KeyDto
from .NetworkTypeDto import NetworkTypeDto
from .SignatureDto import SignatureDto
from .TimestampDto import TimestampDto


class BlockHeaderBuilder:
    """Binary layout for a block header."""

    # pylint: disable-msg=line-too-long
    def __init__(self, signature: SignatureDto, signerPublicKey: KeyDto, version: int, network: NetworkTypeDto, type_: EntityTypeDto, height: HeightDto, timestamp: TimestampDto, difficulty: DifficultyDto, previousBlockHash: Hash256Dto, transactionsHash: Hash256Dto, receiptsHash: Hash256Dto, stateHash: Hash256Dto, beneficiaryPublicKey: KeyDto, feeMultiplier: BlockFeeMultiplierDto):
        """Constructor.
        Args:
            signature: Entity signature.
            signerPublicKey: Entity signer's public key.
            version: Entity version.
            network: Entity network.
            type_: Entity type.
            height: Block height.
            timestamp: Number of milliseconds elapsed since creation of nemesis block.
            difficulty: Block difficulty.
            previousBlockHash: Previous block hash.
            transactionsHash: Hash of the transactions in this block.
            receiptsHash: Hash of the receipts generated by this block.
            stateHash: Hash of the global chain state at this block.
            beneficiaryPublicKey: Beneficiary public key designated by harvester.
            feeMultiplier: Fee multiplier applied to block transactions.
        """
        self.verifiableEntityHeader_Reserved1 = 0
        self.signature = signature
        self.signerPublicKey = signerPublicKey
        self.entityBody_Reserved1 = 0
        self.version = version
        self.network = network
        self.type_ = type_
        self.height = height
        self.timestamp = timestamp
        self.difficulty = difficulty
        self.previousBlockHash = previousBlockHash
        self.transactionsHash = transactionsHash
        self.receiptsHash = receiptsHash
        self.stateHash = stateHash
        self.beneficiaryPublicKey = beneficiaryPublicKey
        self.feeMultiplier = feeMultiplier
        self.blockHeader_Reserved1 = 0

    @classmethod
    def loadFromBinary(cls, payload: bytes) -> BlockHeaderBuilder:
        """Creates an instance of BlockHeaderBuilder from binary payload.
        Args:
            payload: Byte payload to use to serialize the object.
        Returns:
            Instance of BlockHeaderBuilder.
        """
        bytes_ = bytes(payload)
        # gen: _load_from_binary_simple
        # pylint: disable=unused-variable
        size = GeneratorUtils.bufferToUint(GeneratorUtils.getBytes(bytes_, 4))
        bytes_ = bytes_[4:]
        # gen: _load_from_binary_simple
        # pylint: disable=unused-variable
        verifiableEntityHeader_Reserved1 = GeneratorUtils.bufferToUint(GeneratorUtils.getBytes(bytes_, 4))
        bytes_ = bytes_[4:]
        # gen: _load_from_binary_custom
        signature = SignatureDto.loadFromBinary(bytes_)
        bytes_ = bytes_[signature.getSize():]
        # gen: _load_from_binary_custom
        signerPublicKey = KeyDto.loadFromBinary(bytes_)
        bytes_ = bytes_[signerPublicKey.getSize():]
        # gen: _load_from_binary_simple
        # pylint: disable=unused-variable
        entityBody_Reserved1 = GeneratorUtils.bufferToUint(GeneratorUtils.getBytes(bytes_, 4))
        bytes_ = bytes_[4:]
        # gen: _load_from_binary_simple
        version = GeneratorUtils.bufferToUint(GeneratorUtils.getBytes(bytes_, 1))
        bytes_ = bytes_[1:]
        # gen: _load_from_binary_custom
        network = GeneratorUtils.bufferToUint(GeneratorUtils.getBytes(bytes_, 1))
        bytes_ = bytes_[1:]
        # gen: _load_from_binary_custom
        type_ = GeneratorUtils.bufferToUint(GeneratorUtils.getBytes(bytes_, 2))
        bytes_ = bytes_[2:]
        # gen: _load_from_binary_custom
        height = HeightDto.loadFromBinary(bytes_)
        bytes_ = bytes_[height.getSize():]
        # gen: _load_from_binary_custom
        timestamp = TimestampDto.loadFromBinary(bytes_)
        bytes_ = bytes_[timestamp.getSize():]
        # gen: _load_from_binary_custom
        difficulty = DifficultyDto.loadFromBinary(bytes_)
        bytes_ = bytes_[difficulty.getSize():]
        # gen: _load_from_binary_custom
        previousBlockHash = Hash256Dto.loadFromBinary(bytes_)
        bytes_ = bytes_[previousBlockHash.getSize():]
        # gen: _load_from_binary_custom
        transactionsHash = Hash256Dto.loadFromBinary(bytes_)
        bytes_ = bytes_[transactionsHash.getSize():]
        # gen: _load_from_binary_custom
        receiptsHash = Hash256Dto.loadFromBinary(bytes_)
        bytes_ = bytes_[receiptsHash.getSize():]
        # gen: _load_from_binary_custom
        stateHash = Hash256Dto.loadFromBinary(bytes_)
        bytes_ = bytes_[stateHash.getSize():]
        # gen: _load_from_binary_custom
        beneficiaryPublicKey = KeyDto.loadFromBinary(bytes_)
        bytes_ = bytes_[beneficiaryPublicKey.getSize():]
        # gen: _load_from_binary_custom
        feeMultiplier = BlockFeeMultiplierDto.loadFromBinary(bytes_)
        bytes_ = bytes_[feeMultiplier.getSize():]
        # gen: _load_from_binary_simple
        # pylint: disable=unused-variable
        blockHeader_Reserved1 = GeneratorUtils.bufferToUint(GeneratorUtils.getBytes(bytes_, 4))
        bytes_ = bytes_[4:]
        return BlockHeaderBuilder(signature, signerPublicKey, version, network, type_, height, timestamp, difficulty, previousBlockHash, transactionsHash, receiptsHash, stateHash, beneficiaryPublicKey, feeMultiplier)

    def getVerifiableEntityHeader_Reserved1(self) -> int:
        """Gets reserved padding to align Signature on 8-byte boundary.
        Returns:
            Reserved padding to align Signature on 8-byte boundary.
        """
        return self.verifiableEntityHeader_Reserved1

    def getSignature(self) -> SignatureDto:
        """Gets entity signature.
        Returns:
            Entity signature.
        """
        return self.signature

    def getSignerPublicKey(self) -> KeyDto:
        """Gets entity signer's public key.
        Returns:
            Entity signer's public key.
        """
        return self.signerPublicKey

    def getEntityBody_Reserved1(self) -> int:
        """Gets reserved padding to align end of EntityBody on 8-byte boundary.
        Returns:
            Reserved padding to align end of EntityBody on 8-byte boundary.
        """
        return self.entityBody_Reserved1

    def getVersion(self) -> int:
        """Gets entity version.
        Returns:
            Entity version.
        """
        return self.version

    def getNetwork(self) -> NetworkTypeDto:
        """Gets entity network.
        Returns:
            Entity network.
        """
        return self.network

    def getType_(self) -> EntityTypeDto:
        """Gets entity type.
        Returns:
            Entity type.
        """
        return self.type_

    def getHeight(self) -> HeightDto:
        """Gets block height.
        Returns:
            Block height.
        """
        return self.height

    def getTimestamp(self) -> TimestampDto:
        """Gets number of milliseconds elapsed since creation of nemesis block.
        Returns:
            Number of milliseconds elapsed since creation of nemesis block.
        """
        return self.timestamp

    def getDifficulty(self) -> DifficultyDto:
        """Gets block difficulty.
        Returns:
            Block difficulty.
        """
        return self.difficulty

    def getPreviousBlockHash(self) -> Hash256Dto:
        """Gets previous block hash.
        Returns:
            Previous block hash.
        """
        return self.previousBlockHash

    def getTransactionsHash(self) -> Hash256Dto:
        """Gets hash of the transactions in this block.
        Returns:
            Hash of the transactions in this block.
        """
        return self.transactionsHash

    def getReceiptsHash(self) -> Hash256Dto:
        """Gets hash of the receipts generated by this block.
        Returns:
            Hash of the receipts generated by this block.
        """
        return self.receiptsHash

    def getStateHash(self) -> Hash256Dto:
        """Gets hash of the global chain state at this block.
        Returns:
            Hash of the global chain state at this block.
        """
        return self.stateHash

    def getBeneficiaryPublicKey(self) -> KeyDto:
        """Gets beneficiary public key designated by harvester.
        Returns:
            Beneficiary public key designated by harvester.
        """
        return self.beneficiaryPublicKey

    def getFeeMultiplier(self) -> BlockFeeMultiplierDto:
        """Gets fee multiplier applied to block transactions.
        Returns:
            Fee multiplier applied to block transactions.
        """
        return self.feeMultiplier

    def getBlockHeader_Reserved1(self) -> int:
        """Gets reserved padding to align end of BlockHeader on 8-byte boundary.
        Returns:
            Reserved padding to align end of BlockHeader on 8-byte boundary.
        """
        return self.blockHeader_Reserved1

    def getSize(self) -> int:
        """Gets the size of the object.
        Returns:
            Size in bytes.
        """
        size = 0
        size += 4  # size
        size += 4  # verifiableEntityHeader_Reserved1
        size += self.signature.getSize()
        size += self.signerPublicKey.getSize()
        size += 4  # entityBody_Reserved1
        size += 1  # version
        size += 1  # network
        size += 2  # type_
        size += self.height.getSize()
        size += self.timestamp.getSize()
        size += self.difficulty.getSize()
        size += self.previousBlockHash.getSize()
        size += self.transactionsHash.getSize()
        size += self.receiptsHash.getSize()
        size += self.stateHash.getSize()
        size += self.beneficiaryPublicKey.getSize()
        size += self.feeMultiplier.getSize()
        size += 4  # blockHeader_Reserved1
        return size

    def serialize(self) -> bytes:
        """Serializes an object to bytes.
        Returns:
            Serialized bytes.
        """
        bytes_ = bytes()
        sizeBytes = GeneratorUtils.uintToBuffer(self.getSize(), 4)
        bytes_ = GeneratorUtils.concatTypedArrays(bytes_, sizeBytes)
        verifiableEntityHeader_Reserved1Bytes = GeneratorUtils.uintToBuffer(self.getVerifiableEntityHeader_Reserved1(), 4)
        bytes_ = GeneratorUtils.concatTypedArrays(bytes_, verifiableEntityHeader_Reserved1Bytes)
        signatureBytes = self.signature.serialize()
        bytes_ = GeneratorUtils.concatTypedArrays(bytes_, signatureBytes)
        signerPublicKeyBytes = self.signerPublicKey.serialize()
        bytes_ = GeneratorUtils.concatTypedArrays(bytes_, signerPublicKeyBytes)
        entityBody_Reserved1Bytes = GeneratorUtils.uintToBuffer(self.getEntityBody_Reserved1(), 4)
        bytes_ = GeneratorUtils.concatTypedArrays(bytes_, entityBody_Reserved1Bytes)
        versionBytes = GeneratorUtils.uintToBuffer(self.getVersion(), 1)
        bytes_ = GeneratorUtils.concatTypedArrays(bytes_, versionBytes)
        networkBytes = GeneratorUtils.uintToBuffer(self.network, 1)
        bytes_ = GeneratorUtils.concatTypedArrays(bytes_, networkBytes)
        type_Bytes = GeneratorUtils.uintToBuffer(self.type_, 2)
        bytes_ = GeneratorUtils.concatTypedArrays(bytes_, type_Bytes)
        heightBytes = self.height.serialize()
        bytes_ = GeneratorUtils.concatTypedArrays(bytes_, heightBytes)
        timestampBytes = self.timestamp.serialize()
        bytes_ = GeneratorUtils.concatTypedArrays(bytes_, timestampBytes)
        difficultyBytes = self.difficulty.serialize()
        bytes_ = GeneratorUtils.concatTypedArrays(bytes_, difficultyBytes)
        previousBlockHashBytes = self.previousBlockHash.serialize()
        bytes_ = GeneratorUtils.concatTypedArrays(bytes_, previousBlockHashBytes)
        transactionsHashBytes = self.transactionsHash.serialize()
        bytes_ = GeneratorUtils.concatTypedArrays(bytes_, transactionsHashBytes)
        receiptsHashBytes = self.receiptsHash.serialize()
        bytes_ = GeneratorUtils.concatTypedArrays(bytes_, receiptsHashBytes)
        stateHashBytes = self.stateHash.serialize()
        bytes_ = GeneratorUtils.concatTypedArrays(bytes_, stateHashBytes)
        beneficiaryPublicKeyBytes = self.beneficiaryPublicKey.serialize()
        bytes_ = GeneratorUtils.concatTypedArrays(bytes_, beneficiaryPublicKeyBytes)
        feeMultiplierBytes = self.feeMultiplier.serialize()
        bytes_ = GeneratorUtils.concatTypedArrays(bytes_, feeMultiplierBytes)
        blockHeader_Reserved1Bytes = GeneratorUtils.uintToBuffer(self.getBlockHeader_Reserved1(), 4)
        bytes_ = GeneratorUtils.concatTypedArrays(bytes_, blockHeader_Reserved1Bytes)
        return bytes_
