"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const cf = require("@aws-cdk/aws-cloudfront");
const origins = require("@aws-cdk/aws-cloudfront-origins");
const iam = require("@aws-cdk/aws-iam");
const firehose = require("@aws-cdk/aws-kinesisfirehose");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const extensions = require("../extensions");
test('minimal usage', () => {
    // GIVEN
    const deliveryStreamName = 'test';
    const app = new cdk.App();
    const stack = new cdk.Stack(app, 'global-data-ingestion');
    // WHEN
    /**
     * Kinesis Firehose components
     */
    const firehoseRole = new iam.Role(stack, 'FirehoseRole', {
        assumedBy: new iam.ServicePrincipal('firehose.amazonaws.com'),
    });
    const firehoseDataBucket = new s3.Bucket(stack, 'FirehoseDataBucket', {
        autoDeleteObjects: true,
        removalPolicy: cdk.RemovalPolicy.DESTROY,
    });
    firehoseDataBucket.grantReadWrite(firehoseRole);
    const firehoseStream = new firehose.CfnDeliveryStream(stack, 'FirehoseStream', {
        deliveryStreamName,
        deliveryStreamType: 'DirectPut',
        extendedS3DestinationConfiguration: {
            bucketArn: firehoseDataBucket.bucketArn,
            bufferingHints: {
                intervalInSeconds: 60,
                sizeInMBs: 5,
            },
            roleArn: firehoseRole.roleArn,
            errorOutputPrefix: 'error',
        },
    });
    /**
     * Cloudfront and Lambda@Edge
     */
    const ext = new extensions.GlobalDataIngestion(stack, 'GlobalDataIngestion', {
        firehoseStreamName: firehoseStream.deliveryStreamName,
    });
    const dist = new cf.Distribution(stack, 'dist', {
        defaultBehavior: {
            allowedMethods: cf.AllowedMethods.ALLOW_ALL,
            origin: new origins.HttpOrigin('aws.amazon.com'),
            edgeLambdas: [ext],
        },
    });
    /**
     * Outputs
     */
    new cdk.CfnOutput(stack, 'distributionDomainName', {
        value: dist.distributionDomainName,
    });
    new cdk.CfnOutput(stack, 'firehoseStreamName', {
        value: firehoseStream.deliveryStreamName,
    });
    new cdk.CfnOutput(stack, 'firehoseDataBucket', {
        value: firehoseDataBucket.bucketName,
    });
    // THEN
    expect(assert_1.SynthUtils.synthesize(stack).template).toMatchSnapshot();
    expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            DefaultCacheBehavior: {
                AllowedMethods: [
                    'GET',
                    'HEAD',
                    'OPTIONS',
                    'PUT',
                    'PATCH',
                    'POST',
                    'DELETE',
                ],
                LambdaFunctionAssociations: [
                    {
                        EventType: 'viewer-request',
                        IncludeBody: true,
                    },
                ],
                ViewerProtocolPolicy: 'allow-all',
            },
        },
    });
});
//# sourceMappingURL=data:application/json;base64,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