"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const cf = require("@aws-cdk/aws-cloudfront");
const s3 = require("@aws-cdk/aws-s3");
const aws_s3_deployment_1 = require("@aws-cdk/aws-s3-deployment");
const cdk = require("@aws-cdk/core");
const extensions = require("../extensions");
test('minimal usage', () => {
    // GIVEN
    const app = new cdk.App();
    const stack = new cdk.Stack(app, 'demo-stack');
    // WHEN
    // create the cloudfront distribution with extension(s)
    const customErrorPage = new extensions.CustomErrorPage(stack, 'CustomErrorPage');
    // create s3 bucket
    const bucket = new s3.Bucket(customErrorPage, 'demoBucket', {
        autoDeleteObjects: true,
        removalPolicy: cdk.RemovalPolicy.DESTROY,
        websiteIndexDocument: 'index.html',
        websiteErrorDocument: 'error.html',
    });
    // Put demo Object to Bucket.
    new aws_s3_deployment_1.BucketDeployment(customErrorPage, 'Deployment', {
        sources: [aws_s3_deployment_1.Source.asset(path.join(__dirname, './'))],
        destinationBucket: bucket,
        retainOnDelete: false,
    });
    // cloudFront OriginAccessIdentity for bucket
    const originAccessIdentity = new cf.OriginAccessIdentity(customErrorPage, 'OriginAccessIdentity', {
        comment: `CloudFront OriginAccessIdentity for ${bucket.bucketName}`,
    });
    // cloudfront distribution
    const distribution = new cf.CloudFrontWebDistribution(stack, 'distribution', {
        originConfigs: [
            {
                s3OriginSource: {
                    originAccessIdentity,
                    s3BucketSource: bucket,
                },
                behaviors: [{
                        isDefaultBehavior: true,
                        defaultTtl: cdk.Duration.seconds(10),
                        lambdaFunctionAssociations: [customErrorPage],
                    }],
            },
        ],
    });
    new cdk.CfnOutput(stack, 'distributionDomainName', {
        value: distribution.distributionDomainName,
    });
    // THEN
    expect(assert_1.SynthUtils.synthesize(stack).template).toMatchSnapshot();
    expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            DefaultCacheBehavior: {
                AllowedMethods: [
                    'GET',
                    'HEAD',
                ],
                CachedMethods: [
                    'GET',
                    'HEAD',
                ],
                Compress: true,
                DefaultTTL: 10,
                ForwardedValues: {
                    Cookies: {
                        Forward: 'none',
                    },
                    QueryString: false,
                },
                LambdaFunctionAssociations: [
                    {
                        EventType: 'origin-response',
                        IncludeBody: false,
                    },
                ],
                TargetOriginId: 'origin1',
                ViewerProtocolPolicy: 'redirect-to-https',
            },
            DefaultRootObject: 'index.html',
        },
    });
});
//# sourceMappingURL=data:application/json;base64,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