# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['preql', 'preql.core', 'preql.docstring', 'preql.jup_kernel']

package_data = \
{'': ['*'], 'preql': ['modules/*']}

install_requires = \
['arrow',
 'dsnparse',
 'lark-parser>=0.11.3,<0.12.0',
 'prompt-toolkit',
 'pygments',
 'rich>=10.7.0,<11.0.0',
 'runtype>=0.1.16,<0.2.0']

extras_require = \
{'mysql': ['mysqlclient'], 'pgsql': ['psycopg2'], 'server': ['starlette']}

entry_points = \
{'console_scripts': ['preql = preql.__main__:main']}

setup_kwargs = {
    'name': 'preql',
    'version': '0.2.7',
    'description': 'An interpreted relational query language that compiles to SQL',
    'long_description': '![alt text](logo_small.png "Logo")\n\nPreql is an interpreted, relational programming language, that specializes in database queries.\n\nIt is designed for use by data engineers, analysts and data scientists.\n\nPreql\'s main objective is to provide an alternative to SQL, in the form of a high-level programming language, with first-class functions, modules, strict typing, and Python integration.\n\n**How does it work?**\n\nPreql code is interpreted and gets compiled to SQL at runtime. This way, Preql gains the performance and abilities of SQL, but can also operate as a normal scripting language.\n\nCurrently supported dialects are:\n* Postgres\n* MySQL\n* Sqlite\n* BigQuery\n* More... (planned)\n\nFor features that are database-specific, or aren\'t implemented in Preql, there is a `SQL()` function that provides a convenient escape hatch to write raw SQL code.\n\n**Main Features**\n\n* Modern syntax and semantics\n    - Interpreted, everything is an object\n    - Strong type system with gradual type validation and duck-typing\n* Compiles to SQL\n* Python and Pandas integration\n* Interactive shell (REPL) with auto-completion\n* Runs on Jupyter Notebook\n\n\n**Note: Preql is still work in progress, and isn\'t yet recommended for use in production.**\n\n[![tests](https://github.com/erezsh/preql/actions/workflows/tests.yml/badge.svg)](https://github.com/erezsh/Preql/actions/workflows/tests.yml)\n\n# Learn More\n\n- [**Read the documentation**](https://preql.readthedocs.io/en/latest/)\n\n- [Follow the tutorial](https://preql.readthedocs.io/en/latest/tutorial.html)\n\n- [Browse the examples](https://github.com/erezsh/Preql/tree/master/examples)\n\n- Check out the online demo: [![Binder](https://mybinder.org/badge_logo.svg)](https://mybinder.org/v2/gh/erezsh/preql-binder/master?filepath=tutorial_exploring_a_database.ipynb)\n\n# Get started\n\nSimply install via pip:\n\n```sh\n    pip install -U preql\n```\n\nThen just run the interpreter:\n\n```sh\n    preql\n```\n\nRequires Python 3.6+\n\n[Read more](https://preql.readthedocs.io/en/latest/getting-started.html)\n\n# Quick Example\n\n```javascript\n// Declare a new table\ntable Continent {\n    name: string\n    area: int       // km²\n    population: int\n}\n\n// Initialize the table, by inserting rows\nnew Continent("Africa", 30370000, 1287920000)\nnew Continent("Antarctica", 14000000, 4490)\nnew Continent("Asia", 44579000, 4545133000)\nnew Continent("Europe", 10180000, 742648000)\nnew Continent("North America", 24709000, 587615000)\nnew Continent("South America", 17840000, 428240000)\nnew Continent("Australia", 8600000, 41261000)\n\n// Print the continents, ordered by density\nprint Continent {\n    ...                         // Include existing fields\n    density: population / area  // Create new a field\n\n} order{^density}\n\n// Print the total land area\nprint "Total land area:", sum(Continent{area}), "km²"\n\n//  ========================= Output ==========================\n\n                              table  =7\n┏━━━━┳━━━━━━━━━━━━━━━┳━━━━━━━━━━┳━━━━━━━━━━━━┳━━━━━━━━━━━━━━━━━━━━━━━┓\n┃ id ┃ name          ┃     area ┃ population ┃               density ┃\n┡━━━━╇━━━━━━━━━━━━━━━╇━━━━━━━━━━╇━━━━━━━━━━━━╇━━━━━━━━━━━━━━━━━━━━━━━┩\n│  3 │ Asia          │ 44579000 │ 4545133000 │     101.9568182328002 │\n│  4 │ Europe        │ 10180000 │  742648000 │      72.9516699410609 │\n│  1 │ Africa        │ 30370000 │ 1287920000 │     42.40763911755021 │\n│  6 │ South America │ 17840000 │  428240000 │    24.004484304932735 │\n│  5 │ North America │ 24709000 │  587615000 │    23.781415678497712 │\n│  7 │ Australia     │  8600000 │   41261000 │     4.797790697674419 │\n│  2 │ Antarctica    │ 14000000 │       4490 │ 0.0003207142857142857 │\n└────┴───────────────┴──────────┴────────────┴───────────────────────┘\n\nTotal land area: 150278000 km²\n```\n\nIn the background, this table was generated by executing the following compiled SQL code (reformatted):\n\n```sql\n-- Continent {..., density: population / area} order{ ^density }\nWITH subq_1(id, name, area, population, density) AS (\n    SELECT id, name, area, population, (CAST(population AS float) / area) AS density\n    FROM Continent\n    ORDER BY density DESC)\nSELECT * FROM subq_1\n```\n\nSee more examples in [the examples folder](https://github.com/erezsh/Preql/tree/master/examples).\n\n# Interactive Environment\n\n![Screenshot.png](docs/screenshot.png)\n\n# License\n\nPreql uses an “Interface-Protection Clause” on top of the MIT license.\n\nSee: [LICENSE](LICENSE)\n\nIn simple words, it\'s free for personal use. Also, it can be used for any commercial or non-commercial purpose, as long as your product doesn\'t base its value on exposing the Preql language itself to your users. [Read more](https://preql.readthedocs.io/en/latest/faq.html#license)\n',
    'author': 'Erez Shin',
    'author_email': 'erezshin@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/erezsh/Preql',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'entry_points': entry_points,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
