﻿"""Define the functions required for parsing wikitext into spans."""
from functools import partial
from typing import Dict, List, Callable, Optional

from regex import VERBOSE, IGNORECASE
from regex import compile as regex_compile

from ._config import (
    _parsable_tag_extensions, regex_pattern, _unparsable_tag_extensions,
    _bare_external_link_schemes, _parser_functions, _HTML_TAG_NAME)


# According to https://www.mediawiki.org/wiki/Manual:$wgLegalTitleChars
# illegal title characters are: r'[]{}|#<>[\u0000-\u0020]'
VALID_TITLE_CHARS_PATTERN = rb'[^\x00-\x1f\|\{\}\[\]<>\n]++'
# Parameters
# Parser functions
# According to https://www.mediawiki.org/wiki/Help:Magic_words
# See also:
# https://translatewiki.net/wiki/MediaWiki:Sp-translate-data-MagicWords/fa
PARAMS_FINDITER = regex_compile(
    rb'\{\{\{(?>[^{}]*+|}(?!})|{(?!{))*+\}\}\}').finditer
PF_TL_FINDITER = regex_compile(
    rb'\{\{'
    rb'(?>'
    # parser function
    rb'\s*+'
    rb'(?>\#[^{}\s:]++|' + regex_pattern(_parser_functions).encode()[3:] +
    # end of generated part
    rb':(?>[^{}]*+|}(?!})|{(?!{))*+\}\}()'
    rb'|'
    # invalid template name
    rb'[\s_]*+'  # invalid name
    rb'(?:\|(?>[^{}]++|{(?!{)|}(?!}))*+)?+'  # args
    rb'\}\}()'
    rb'|'
    # template
    rb'\s*+'
    + VALID_TITLE_CHARS_PATTERN +  # template name
    rb'\s*+'
    rb'(?:\|(?>[^{}]++|{(?!{)|}(?!}))*+)?+'  # args
    rb'\}\}'
    rb')').finditer
# External links
INVALID_EXTLINK_CHARS = rb' \t\n<>\[\]"'
VALID_EXTLINK_CHARS = rb'[^' + INVALID_EXTLINK_CHARS + rb']++'
# See more info on literal IPv6 see:
# https://en.wikipedia.org/wiki/IPv6_address#Literal_IPv6_addresses_in_network_resource_identifiers
# The following pattern is part of EXT_LINK_ADDR constant in
# https://github.com/wikimedia/mediawiki/blob/master/includes/parser/Parser.php
LITERAL_IPV6_AND_TAIL = \
    rb'\[[0-9a-fA-F:.]++\][^' + INVALID_EXTLINK_CHARS + rb']*+'
# A \b is added to the beginning.
BARE_EXTERNAL_LINK_SCHEMES = (
    rb'\b' + regex_pattern(_bare_external_link_schemes).encode())
EXTERNAL_LINK_URL_TAIL = (
    rb'(?>' + LITERAL_IPV6_AND_TAIL + rb'|' + VALID_EXTLINK_CHARS + rb')')
BARE_EXTERNAL_LINK = (
    BARE_EXTERNAL_LINK_SCHEMES + EXTERNAL_LINK_URL_TAIL)
# Wikilinks
# https://www.mediawiki.org/wiki/Help:Links#Internal_links
WIKILINK_FINDITER = regex_compile(
    rb'''
    (?<!\[)\[\[
    (?!\ *+''' + BARE_EXTERNAL_LINK + rb')'
    + VALID_TITLE_CHARS_PATTERN + rb'''
    (?:
        \]\]
        |
        \| # Text of the wikilink
        (?> # Any character that is not the start of another wikilink
            [^\[\]]++
            |
            \[(?!\[) # optionally followed by a single closing bracket:
            [^\[\]]*+
            (?:\](?>(?!\])|(?=\]\])))?
            |
            \](?!\])
        )*+
        \]\]
    )
    ''',
    IGNORECASE | VERBOSE).finditer

# these characters iterfere with detection of (args|tls|wlinks|wlists)
blank_sensitive_chars = partial(regex_compile(br'[\|\{\}\n]').sub, br' ')
blank_brackets = partial(regex_compile(br'[\[\]]').sub, br' ')

# todo: make regex_pattern return bytes?
PARSABLE_TAG_EXTENSIONS_PATTERN = regex_pattern(
    _parsable_tag_extensions).encode()
UNPARSABLE_TAG_EXTENSIONS_PATTERN = regex_pattern(
    _unparsable_tag_extensions).encode()

# The idea of the following regex is to detect innermost HTML tags. From
# http://blog.stevenlevithan.com/archives/match-innermost-html-element
# But it's not bullet proof:
# https://stackoverflow.com/questions/3076219/
EXTENSION_TAGS_FINDITER = regex_compile(
    rb'< (' + UNPARSABLE_TAG_EXTENSIONS_PATTERN + rb'|(' +
    PARSABLE_TAG_EXTENSIONS_PATTERN + rb''')) \b [^>]*+
        (?:
            (?<=/)> # self-closing
            |>(?># contents
                # Either contains no other tags or
                [^<]++
                |
                # the nested-tag is something else or
                < (?! \1 \b [^>]*+ >)
                |
                # the nested tag closes itself.
                # Note that for extension tags whitespace
                # is not allowed between / and >.
                <\1\b[^>]*/>
            )*?
            # tag-end
            </\1\s*+>
        )''', IGNORECASE | VERBOSE).finditer
COMMENT_PATTERN = r'<!--[\s\S]*?-->'
COMMENT_FINDITER = regex_compile(COMMENT_PATTERN.encode()).finditer

# HTML tags
# Tags:
# https://infra.spec.whatwg.org/#ascii-whitespace
SPACE_CHARS = rb' \t\n\u000C\r'  # \s - \v
# http://stackoverflow.com/a/93029/2705757
# chrs = (chr(i) for i in range(sys.maxunicode))
# control_chars = ''.join(c for c in chrs if unicodedata.category(c) == 'Cc')
CONTROL_CHARS = rb'\x00-\x1f\x7f-\x9f'
# https://www.w3.org/TR/html5/syntax.html#syntax-attributes
ATTR_NAME = (
    rb'(?<attr_name>[^' + SPACE_CHARS + CONTROL_CHARS + rb'\u0000"\'>/=]++)')
WS_EQ_WS = rb'[' + SPACE_CHARS + rb']*+=[' + SPACE_CHARS + rb']*+'
UNQUOTED_ATTR_VAL = (
    rb'(?<attr_value>[^' + SPACE_CHARS + rb'"\'=<>`]++)')
QUOTED_ATTR_VAL = rb'(?<quote>[\'"])(?<attr_value>.+?)(?P=quote)'
# May include character references, but for now, ignore the fact that they
# cannot contain an ambiguous ampersand.
ATTR_VAL = (
    # If an empty attribute is to be followed by the optional
    # "/" character, then there must be a space character separating
    # the two. This rule is ignored here.
    rb'(?:'
    + WS_EQ_WS + UNQUOTED_ATTR_VAL + rb'[' + SPACE_CHARS + rb']*|'
    + WS_EQ_WS + QUOTED_ATTR_VAL + rb'[' + SPACE_CHARS + rb']*|'
    + rb'[' + SPACE_CHARS + rb']*+(?<attr_value>)'  # empty attribute
    + rb')')
# Ignore ambiguous ampersand for the sake of simplicity.
ATTR_PATTERN = (
    rb'(?<attr>[' + SPACE_CHARS + rb']++' + ATTR_NAME + ATTR_VAL + rb')')
ATTRS_MATCH = regex_compile(
    # Leading space is not required at the start of the attribute string.
    rb'(?<attr>[' + SPACE_CHARS + rb']*+' + ATTR_NAME + ATTR_VAL + rb')*+'
    rb'(?<attr_insert>)',
).match
# VOID_ELEMENTS = (
#     'area', 'base', 'br', 'col', 'embed', 'hr', 'img', 'input', 'keygen',
#     'link', 'meta', 'param', 'source', 'track', 'wbr'
# )
# RAW_TEXT_ELEMENTS = ('script', 'style')
# ESCAPABLE_RAW_TEXT_ELEMENTS = ('textarea', 'title')
# Detecting foreign elements in MathML and SVG namespaces is not implemented
# yet. See
# https://developer.mozilla.org/en/docs/Web/SVG/Namespaces_Crash_Course
# for an overview.
# note that end tags do not accept attributes, but MW currently cleans up and
# ignores such attributes
END_TAG_PATTERN = rb'(?<end_tag></{name}(?:>|[' + SPACE_CHARS + rb'][^>]*+>))'
START_TAG_PATTERN = (
    rb'(?<start_tag>'
    rb'<{name}(?:' + ATTR_PATTERN + rb')*'
    rb'[' + SPACE_CHARS + rb']*+'
    rb'(?:(?<self_closing>/[' + SPACE_CHARS + b']*+>)|>)'
    rb')')
HTML_START_TAG_FINDITER = regex_compile(
    START_TAG_PATTERN.replace(b'{name}', _HTML_TAG_NAME, 1)).finditer
HTML_END_TAG_FINDITER = regex_compile(
    END_TAG_PATTERN.replace(b'{name}', _HTML_TAG_NAME, 1)).finditer


def parse_to_spans(byte_array: bytearray) -> Dict[str, List[List[int]]]:
    """Calculate and set self._type_to_spans.

    The result is a dictionary containing lists of spans:
    {
        'Comment': comment_spans,
        'ExtTag': extension_tag_spans,
        'Parameter': parameter_spans,
        'ParserFunction': parser_function_spans,
        'Template': template_spans,
        'WikiLink': wikilink_spans,
    }
    """
    comment_spans = []  # type: List[List[int]]
    cms_append = comment_spans.append
    extension_tag_spans = []  # type: List[List[int]]
    ets_append = extension_tag_spans.append
    wikilink_spans = []  # type: List[List[int]]
    wls_append = wikilink_spans.append
    parameter_spans = []  # type: List[List[int]]
    pms_append = parameter_spans.append
    parser_function_spans = []  # type: List[List[int]]
    pfs_append = parser_function_spans.append
    template_spans = []  # type: List[List[int]]
    tls_append = template_spans.append
    # HTML <!-- comments -->
    for match in COMMENT_FINDITER(byte_array):
        ms, me = match.span()
        cms_append([ms, me])
        byte_array[ms:me] = b' ' * (me - ms)
    # <extension tags>
    for match in EXTENSION_TAGS_FINDITER(byte_array):
        ms, me = match.span()
        ets_append([ms, me])
        if match[2]:  # parsable tag extension group
            parse_pm_pf_tl(
                byte_array, ms, me,
                pms_append, pfs_append, tls_append, wls_append)
        byte_array[ms:me] = b'_' * (me - ms)
    parse_pm_pf_tl(
        byte_array, 0, None,
        pms_append, pfs_append,
        tls_append, wls_append)
    return {
        'Comment': sorted(comment_spans),
        'ExtensionTag': sorted(extension_tag_spans),
        'Parameter': sorted(parameter_spans),
        'ParserFunction': sorted(parser_function_spans),
        'Template': sorted(template_spans),
        'WikiLink': sorted(wikilink_spans)}


def parse_pm_pf_tl(
    byte_array: bytearray, start: int, end: Optional[int],
    params_append: Callable, pfs_append: Callable,
    tls_append: Callable, wikilinks_append: Callable,
) -> None:
    """Find the spans of parameters, parser functions, and templates.

    :byte_array: The byte_array or part of byte_array that is being parsed.
    :start: Add to every returned start.

    This is the innermost loop of the parse_to_spans function.
    If the byte_array passed to parse_to_spans contains n WikiLinks, then
    this function will be called n + 1 times. One time for the whole byte_array
    and n times for each of the n WikiLinks.
    """
    start_and_end_tags = list(
        HTML_START_TAG_FINDITER(byte_array, start, end)
    ) + list(HTML_END_TAG_FINDITER(byte_array, start, end))
    for match in start_and_end_tags:
        ms, me = match.span()
        byte_array[ms:me] = blank_brackets(byte_array[ms:me])
    while True:
        while True:
            match = None
            for match in PARAMS_FINDITER(byte_array, start, end):
                ms, me = match.span()
                params_append([ms, me])
                byte_array[ms:me] = \
                    b'PPP' + byte_array[ms + 3:me - 3].replace(b'|', b'P') \
                    + b'PPP'
            if match is None:
                break
        match = None
        for match in WIKILINK_FINDITER(byte_array, start, end):
            ms, me = match.span()
            wikilinks_append([ms, me])
            parse_pm_pf_tl(
                byte_array, ms + 2, me - 2,
                params_append, pfs_append, tls_append, wikilinks_append)
            byte_array[ms:me] = b'_' * (me - ms)
        for match in PF_TL_FINDITER(byte_array, start, end):
            ms, me = match.span()
            if match[1] is not None:
                pfs_append([ms, me])
                byte_array[ms:me] = b'X' * (me - ms)
            elif match[2] is not None:  # invalid template name
                byte_array[ms:me] = b'_' * (me - ms)
                byte_array[ms+1] = 123
                continue
            else:
                tls_append([ms, me])
                byte_array[ms:me] = b'X' * (me - ms)
        if match is None:
            break
    for match in start_and_end_tags:
        ms, me = match.span()
        byte_array[ms:me] = blank_sensitive_chars(byte_array[ms:me])
