# This file is part of pyGPCCA.
#
# Copyright (c) 2020 Bernhard Reuter.
# With contributions of Marius Lange and Michal Klein.
# Based on the original MATLAB GPCCA code authored by Bernhard Reuter, Susanna Roeblitz and Marcus Weber,
# Zuse Institute Berlin, Takustrasse 7, 14195 Berlin
# --------------------------------------------------
# If you use this code or parts of it, cite the following reference:
# ------------------------------------------------------------------
# Reuter, B., Weber, M., Fackeldey, K., Röblitz, S., & Garcia, M. E. (2018).
# Generalized Markov State Modeling Method for Nonequilibrium Biomolecular Dynamics:
# Exemplified on Amyloid β Conformational Dynamics Driven by an Oscillating Electric Field.
# Journal of Chemical Theory and Computation, 14(7), 3579–3594. https://doi.org/10.1021/acs.jctc.8b00079
# ----------------------------------------------------------------
# pyGPCCA is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from typing import Tuple
import warnings

import pytest

from scipy.sparse import diags
from numpy.testing import assert_allclose as assert_allclose_np
import numpy as np


def assert_allclose(actual, desired, rtol=1.0e-5, atol=1.0e-8, err_msg="", verbose=True):
    return assert_allclose_np(actual, desired, rtol=rtol, atol=atol, err_msg=err_msg, verbose=True)


def get_known_input(Tc: np.ndarray) -> Tuple[np.ndarray, np.ndarray]:
    assert not np.allclose(Tc, 0.0), "Tc doesn't seem to be a count matrix."
    assert Tc.dtype == np.float64, "Expected double precision"
    assert Tc.shape[0] == Tc.shape[1], "Expected a square matrix."

    row = np.sum(Tc, axis=1) > 0.0001
    P = Tc.copy()
    Ts = Tc[row, :]
    P[row, :] = np.diag(1.0 / np.sum(Ts, axis=1)) @ Ts

    sd = np.sum(Tc, axis=1)

    return P, sd / np.sum(sd)


def bdc(q: np.ndarray, p: np.ndarray, sparse: bool = True) -> Tuple[np.ndarray, np.ndarray]:
    """Birth death chain."""
    q, p = np.asarray(q), np.asarray(p)
    if q[0] != 0.0:
        raise ValueError("Probability q[0] must be zero.")
    if p[-1] != 0.0:
        raise ValueError("Probability p[-1] must be zero.")
    if not np.all(q + p <= 1.0):
        raise ValueError("Probabilities q+p can not exceed one.")
    r = 1 - q - p
    P = diags([q[1:], r, p[0:-1]], [-1, 0, 1])

    mu = np.zeros(r.shape[0])
    mu[0] = 1.0
    mu[1:] = np.cumprod(p[:-1] / q[1:])

    return (P if sparse else P.A), mu / np.sum(mu)


def mu(mu: int):
    return np.array(
        [
            [1000, 100, 100, 10, 0, 0, 0, 0, 0],
            [100, 1000, 100, 0, 0, 0, 0, 0, 0],
            [100, 100, 1000, 0, mu, 0, 0, 0, 0],
            [10, 0, 0, 1000, 100, 100, 10, 0, 0],
            [0, 0, mu, 100, 1000, 100, 0, 0, 0],
            [0, 0, 0, 100, 100, 1000, 0, mu, 0],
            [0, 0, 0, 10, 0, 0, 1000, 100, 100],
            [0, 0, 0, 0, 0, mu, 100, 1000, 100],
            [0, 0, 0, 0, 0, 0, 100, 100, 1000],
        ],
        dtype=np.float64,
    )


@pytest.fixture(scope="session", autouse=True)
def _session_setup():
    warnings.filterwarnings("once", category=DeprecationWarning)
    warnings.filterwarnings("once", category=PendingDeprecationWarning)


def _skip_if_no_petsc_slepc() -> bool:
    try:
        import mpi4py  # noqa: F401
        import petsc4py  # noqa: F401
        import slepc4py  # noqa: F401

        return False
    except ImportError:
        return True


skip_if_no_petsc_slepc = pytest.mark.skipif(_skip_if_no_petsc_slepc(), reason="No PETSc or SLEPc is installed.")


@pytest.fixture(scope="session")
def count_sd():
    return np.array(
        [
            2.1196820476928461e-03,
            5.4491826226066090e-03,
            2.3426486027095936e-02,
            5.8221266809978502e-02,
            7.2479128130780376e-02,
            5.4041893715942611e-02,
            2.8355746638004300e-02,
            1.4687796830475429e-02,
            9.4685797130430443e-03,
            5.8791181322801582e-03,
            5.1292306154076886e-03,
            4.1493775933609959e-03,
            3.6094585812128182e-03,
            3.7094435834624808e-03,
            4.4093385992101186e-03,
            5.5791631255311704e-03,
            7.3189021646752990e-03,
            1.3967904814277858e-02,
            3.0445433185022245e-02,
            5.9751037344398343e-02,
            8.7266909963505473e-02,
            8.8306753986901965e-02,
            6.0670899365095239e-02,
            2.9825526171074340e-02,
            1.3857921311803230e-02,
            7.7488376743488473e-03,
            5.3791931210318451e-03,
            3.9594060890866369e-03,
            4.0093985902114682e-03,
            3.2395140728890665e-03,
            3.7694345848122784e-03,
            5.3691946208068789e-03,
            5.8091286307053944e-03,
            7.6788481727740835e-03,
            1.4617807328900665e-02,
            2.7865820126980953e-02,
            5.3891916212568114e-02,
            7.4298855171724243e-02,
            5.8661200819877017e-02,
            2.2656601509773534e-02,
            5.9291106334049895e-03,
            2.9895515672649104e-03,
        ],
        dtype=np.float64,
    )


@pytest.fixture(scope="session")
def count_A():
    return np.array(
        [
            [2.8405568184913244e-01, 2.8755576853720538e-01, 4.2838854961366207e-01],
            [3.7905560917700676e-01, -3.3892409425382630e-01, -4.0131514923180477e-02],
            [-1.9965336957541363e-01, -2.7103148561320506e-01, 4.7068485518861869e-01],
        ],
        dtype=np.float64,
    )


@pytest.fixture(scope="session")
def count_A_sparse():
    return np.array(
        [
            [0.28406323, 0.28756332, 0.42837345],
            [-0.37903994, 0.33891752, 0.04012242],
            [0.1996216, 0.27103713, -0.47065873],
        ],
        dtype=np.float64,
    )


@pytest.fixture(scope="session")
def count_chi():
    return np.array(
        [
            [
                8.5043523873879956e-01,
                4.3886683668813712e-11,
                1.4956476121731360e-01,
            ],
            [
                8.8090007542006055e-01,
                2.5589115375342489e-02,
                9.3510809204596937e-02,
            ],
            [
                9.6495990190978509e-01,
                1.1558207813296294e-02,
                2.3481890276918493e-02,
            ],
            [
                9.7558793310460945e-01,
                1.5709140129765899e-02,
                8.7029267656243241e-03,
            ],
            [
                9.7963327053545934e-01,
                1.4060862546031219e-02,
                6.3058669185093305e-03,
            ],
            [
                9.8568851438933425e-01,
                1.4311485610665552e-02,
                2.8071248236694572e-17,
            ],
            [
                9.6791270535282792e-01,
                1.7471491809863252e-02,
                1.4615802837308636e-02,
            ],
            [
                9.4601281012257998e-01,
                1.7835098543344384e-02,
                3.6152091334075327e-02,
            ],
            [
                9.2448952433822940e-01,
                1.8479564079042093e-02,
                5.7030911582728390e-02,
            ],
            [
                8.8318556579259333e-01,
                2.4948552633467080e-02,
                9.1865881573939379e-02,
            ],
            [
                7.5874311920722926e-01,
                2.5407453476171762e-02,
                2.1584942731659873e-01,
            ],
            [
                6.8245807053420782e-01,
                5.4751908940178097e-02,
                2.6279002052561395e-01,
            ],
            [
                5.3234840569967190e-01,
                6.3068044907413703e-02,
                4.0458354939291430e-01,
            ],
            [
                3.1537515699639085e-01,
                5.2337908021706729e-02,
                6.3228693498190214e-01,
            ],
            [
                2.2250786517110696e-01,
                7.6139414102712424e-02,
                7.0135272072618049e-01,
            ],
            [
                1.2375934853700912e-01,
                5.4508417089850582e-02,
                8.2173223437314014e-01,
            ],
            [
                4.9653667316514778e-02,
                2.9603583297963372e-02,
                9.2074274938552181e-01,
            ],
            [
                2.5736159571559877e-02,
                1.2561644103436256e-02,
                9.6170219632500376e-01,
            ],
            [
                1.1894344993512570e-02,
                6.3139005313615033e-03,
                9.8179175447512568e-01,
            ],
            [
                4.8084968892439985e-03,
                3.6671524338928122e-03,
                9.9152435067686306e-01,
            ],
            [
                3.1536353492936597e-03,
                7.1221730872186764e-04,
                9.9613414734198436e-01,
            ],
            [
                -3.0507380398593992e-18,
                -6.5549521997357121e-18,
                9.9999999999999989e-01,
            ],
            [
                8.2726996336765744e-03,
                3.0940472642753200e-04,
                9.9141789563989580e-01,
            ],
            [
                1.2722881126783353e-02,
                5.9164159394744483e-03,
                9.8136070293374211e-01,
            ],
            [
                2.2122528189377291e-02,
                1.3902592390825667e-02,
                9.6397487941979698e-01,
            ],
            [
                4.0589834059942057e-02,
                3.7980881743777700e-02,
                9.2142928419628023e-01,
            ],
            [
                8.0588740622014210e-02,
                1.1344669076732210e-01,
                8.0596456861066357e-01,
            ],
            [
                1.0568119504467953e-01,
                2.0230326690849842e-01,
                6.9201553804682192e-01,
            ],
            [
                8.3281552679901921e-02,
                3.4042640424725878e-01,
                5.7629204307283921e-01,
            ],
            [
                5.8911161575873422e-02,
                4.8967335413071028e-01,
                4.5141548429341599e-01,
            ],
            [
                4.3770917595157881e-02,
                6.6193931626137592e-01,
                2.9428976614346619e-01,
            ],
            [
                2.9817138929281523e-02,
                8.0565415247850192e-01,
                1.6452870859221638e-01,
            ],
            [
                1.4136303417928717e-02,
                8.8989778149855780e-01,
                9.5965915083513417e-02,
            ],
            [
                1.3671381908787954e-02,
                9.4419514302533858e-01,
                4.2133475065873442e-02,
            ],
            [
                7.5795321139255963e-03,
                9.6824758368408681e-01,
                2.4172884201987460e-02,
            ],
            [
                1.0405260238833199e-02,
                9.8297752755905998e-01,
                6.6172122021064947e-03,
            ],
            [
                4.5138347223880404e-03,
                9.8998687175067734e-01,
                5.4992935269345851e-03,
            ],
            [
                8.4518297221162914e-03,
                9.9154817000192508e-01,
                2.7595845559331095e-10,
            ],
            [
                6.5843351926822497e-03,
                9.8840580521018928e-01,
                5.0098595971283938e-03,
            ],
            [
                9.2202981926351631e-03,
                9.7351230297269997e-01,
                1.7267398834664754e-02,
            ],
            [
                1.6299801690329144e-02,
                9.2119142752842731e-01,
                6.2508770781243561e-02,
            ],
            [4.5803069549102952e-10, 9.1035357717144483e-01, 8.9646422370524254e-02],
        ],
        dtype=np.float64,
    )


@pytest.fixture(scope="session")
def count_chi_sparse():
    return np.array(
        [
            [8.50407926e-01, 1.84345442e-05, 1.49573639e-01],
            [8.80868740e-01, 2.56083344e-02, 9.35229259e-02],
            [9.64921893e-01, 1.15791809e-02, 2.34989259e-02],
            [9.75548771e-01, 1.57303637e-02, 8.72086554e-03],
            [9.79593838e-01, 1.40821658e-02, 6.32399623e-03],
            [9.85648535e-01, 1.43329214e-02, 1.85435129e-05],
            [9.67874127e-01, 1.74925576e-02, 1.46333153e-02],
            [9.45976143e-01, 1.78556914e-02, 3.61681661e-02],
            [9.24454720e-01, 1.84996935e-02, 5.70455869e-02],
            [8.83154062e-01, 2.49678181e-02, 9.18781197e-02],
            [7.58722557e-01, 2.54240237e-02, 2.15853419e-01],
            [6.82442695e-01, 5.47669679e-02, 2.62790337e-01],
            [5.32345822e-01, 6.30798903e-02, 4.04574287e-01],
            [3.15392254e-01, 5.23449982e-02, 6.32262748e-01],
            [2.22531900e-01, 7.61446067e-02, 7.01323494e-01],
            [1.23793216e-01, 5.45113642e-02, 8.21695420e-01],
            [4.96953675e-02, 2.96048032e-02, 9.20699829e-01],
            [2.57808588e-02, 1.25622629e-02, 9.61656878e-01],
            [1.19405908e-02, 6.31418899e-03, 9.81745220e-01],
            [4.85550550e-03, 3.66727445e-03, 9.91477220e-01],
            [3.20094436e-03, 7.12289124e-04, 9.96086767e-01],
            [4.76241598e-05, 0.00000000e00, 9.99952376e-01],
            [8.31957865e-03, 3.09585551e-04, 9.91370836e-01],
            [1.27690747e-02, 5.91672043e-03, 9.81314205e-01],
            [2.21674757e-02, 1.39031394e-02, 9.63929385e-01],
            [4.06318947e-02, 3.79819458e-02, 9.21386160e-01],
            [8.06233287e-02, 1.13448988e-01, 8.05927683e-01],
            [1.05708924e-01, 2.02306539e-01, 6.91984537e-01],
            [8.33040292e-02, 3.40429861e-01, 5.76266110e-01],
            [5.89279778e-02, 4.89677006e-01, 4.51395016e-01],
            [4.37800559e-02, 6.61943476e-01, 2.94276468e-01],
            [2.98199884e-02, 8.05658707e-01, 1.64521305e-01],
            [1.41361275e-02, 8.89902405e-01, 9.59614678e-02],
            [1.36684064e-02, 9.44200020e-01, 4.21315740e-02],
            [7.57583509e-03, 9.68252445e-01, 2.41717200e-02],
            [1.04005436e-02, 9.82982521e-01, 6.61693492e-03],
            [4.50927055e-03, 9.89991772e-01, 5.49895751e-03],
            [8.44683688e-03, 9.91553163e-01, 1.44184623e-17],
            [6.57967129e-03, 9.88410743e-01, 5.00958610e-03],
            [9.21618117e-03, 9.73517225e-01, 1.72665936e-02],
            [1.62977980e-02, 9.21196249e-01, 6.25059525e-02],
            [0.00000000e00, 9.10357993e-01, 8.96420068e-02],
        ],
        dtype=np.float64,
    )


@pytest.fixture(scope="session")
def count_sd_weights():
    return np.array(
        [2.8405568184913244e-01, 2.8755576853720533e-01, 4.2838854961366213e-01],
        dtype=np.float64,
    )


@pytest.fixture(scope="session")
def count_Pc():
    return np.array(
        [
            [9.5104510750059479e-01, 1.8354056625839690e-02, 3.0600835873566056e-02],
            [1.8594617368171685e-02, 9.5127058926326480e-01, 3.0134793368563478e-02],
            [1.9985610707576221e-02, 2.0510702582310643e-02, 9.5950368671011388e-01],
        ],
        dtype=np.float64,
    )


@pytest.fixture(params=[10, 50, 100, 200, 500, 1000])
def example_matrix_mu(request):
    mu = request.param

    # TODO: check if mu=0 is irreducible
    return np.array(
        [
            [1000, 100, 100, 10, 0, 0, 0, 0, 0],
            [100, 1000, 100, 0, 0, 0, 0, 0, 0],
            [100, 100, 1000, 0, mu, 0, 0, 0, 0],
            [10, 0, 0, 1000, 100, 100, 10, 0, 0],
            [0, 0, mu, 100, 1000, 100, 0, 0, 0],
            [0, 0, 0, 100, 100, 1000, 0, mu, 0],
            [0, 0, 0, 10, 0, 0, 1000, 100, 100],
            [0, 0, 0, 0, 0, mu, 100, 1000, 100],
            [0, 0, 0, 0, 0, 0, 100, 100, 1000],
        ],
        dtype=np.float64,
    )


@pytest.fixture(scope="session")
def A_mu0_init():
    return np.array(
        [
            [
                3.3390904470272748e-01,
                3.3304547764863601e-01,
                3.3304547764863596e-01,
            ],
            [
                1.1824450993798861e-14,
                -4.0366140439775017e-01,
                4.0366140439773823e-01,
            ],
            [-4.5730890701570082e-01, 2.2865445350784039e-01, 2.2865445350786040e-01],
        ],
        dtype=np.float64,
    )


@pytest.fixture(scope="session")
def A_mu0():
    return np.array(
        [
            [
                3.3390904470272720e-01,
                3.3304547764863651e-01,
                3.3304547764863612e-01,
            ],
            [
                1.1934897514720417e-14,
                -4.0366140439774961e-01,
                4.0366140439773768e-01,
            ],
            [-4.5730890701570021e-01, 2.2865445350784008e-01, 2.2865445350786009e-01],
        ],
        dtype=np.float64,
    )


@pytest.fixture(scope="session")
def svecs_mu0():
    return np.array(
        [
            [
                1.0000000000000002e00,
                1.1983873824612949e00,
                6.6212084649127290e-01,
            ],
            [
                1.0000000000000002e00,
                1.2386618947282897e00,
                7.3016081598270310e-01,
            ],
            [
                1.0000000000000002e00,
                1.2386618947282877e00,
                7.3016081598270255e-01,
            ],
            [
                1.0000000000000002e00,
                2.9934085326177830e-14,
                -1.3208168935772115e00,
            ],
            [
                1.0000000000000002e00,
                3.8344472783066848e-14,
                -1.4565448979422635e00,
            ],
            [
                1.0000000000000002e00,
                3.8841087123289861e-14,
                -1.4565448979422631e00,
            ],
            [
                1.0000000000000002e00,
                -1.1983873824613247e00,
                6.6212084649121505e-01,
            ],
            [
                1.0000000000000002e00,
                -1.2386618947283270e00,
                7.3016081598263805e-01,
            ],
            [1.0000000000000002e00, -1.2386618947283281e00, 7.3016081598263893e-01],
        ],
        dtype=np.float64,
    )


@pytest.fixture(scope="session")
def A_mu0_krylov_sparse_init() -> np.ndarray:
    return np.array(
        [
            [0.33335121587060095871, 0.33332439206469977044, 0.33332439206469915982],
            [-0.00000000000000805242, -0.40387210091001207735, 0.40387210091002012646],
            [-0.45723108811365476001, 0.22861554405683459645, 0.22861554405682027458],
        ],
        dtype=np.float64,
    )


@pytest.fixture(scope="session")
def A_mu0_krylov_sparse() -> np.ndarray:
    return np.array(
        [
            [0.33335121587060068116, 0.33332439206469949289, 0.33332439206469965942],
            [-0.00000000000000804912, -0.40387210091001152223, 0.40387210091001957135],
            [-0.45723108811365426041, 0.22861554405683429114, 0.22861554405681996927],
        ],
        dtype=np.float64,
    )


@pytest.fixture(scope="session")
def svecs_mu0_krylov_sparse(A_mu0) -> np.ndarray:
    return np.array(
        [
            [1.00000000000000000000, -1.19776219433548658522, 0.66101352170956473309],
            [1.00000000000000000000, -1.23801569574471570512, 0.72906507133160869394],
            [1.00000000000000000000, -1.23801569574471770352, 0.72906507133160924905],
            [1.00000000000000000000, -0.00000000000002250283, -1.32226170642953344547],
            [1.00000000000000000000, -0.00000000000002489574, -1.45801281115795178245],
            [1.00000000000000000000, -0.00000000000002725720, -1.45801281115795178245],
            [1.00000000000000000000, 1.19776219433550989990, 0.66101352170952409892],
            [1.00000000000000000000, 1.23801569574473901980, 0.72906507133156417400],
            [1.00000000000000000000, 1.23801569574474501501, 0.72906507133156528422],
        ],
        dtype=np.float64,
    )


@pytest.fixture(scope="session")
def A_mu0_krylov_dense_init() -> np.ndarray:
    return np.array(
        [
            [0.33335121587060095871, 0.33332439206469949289, 0.33332439206469943738],
            [0.00000000000003168111, 0.40387210091000114165, -0.40387210091003289403],
            [-0.45723108811365464899, 0.22861554405685508007, 0.22861554405679970770],
        ],
        dtype=np.float64,
    )


@pytest.fixture(scope="session")
def A_mu0_krylov_dense() -> np.ndarray:
    return np.array(
        [
            [0.33335121587060101422, 0.33332439206469949289, 0.33332439206469943738],
            [0.00000000000003175238, 0.40387210091000097512, -0.40387210091003272749],
            [-0.45723108811365453796, 0.22861554405685496905, 0.22861554405679959667],
        ],
        dtype=np.float64,
    )


@pytest.fixture(scope="session")
def svecs_mu0_krylov_dense(A_mu0) -> np.ndarray:
    return np.array(
        [
            [1.00000000000000000000, -1.19776219433554897975, 0.66101352170946059417],
            [1.00000000000000000000, -1.23801569574477676738, 0.72906507133150055822],
            [1.00000000000000000000, -1.23801569574477854374, 0.72906507133150177946],
            [1.00000000000000000000, 0.00000000000009534040, -1.32226170642953522183],
            [1.00000000000000000000, 0.00000000000009983375, -1.45801281115795244858],
            [1.00000000000000000000, 0.00000000000010021521, -1.45801281115794933996],
            [1.00000000000000000000, 1.19776219433545350057, 0.66101352170963001420],
            [1.00000000000000000000, 1.23801569574467862367, 0.72906507133167375301],
            [1.00000000000000000000, 1.23801569574467729140, 0.72906507133167042234],
        ],
        dtype=np.float64,
    )


@pytest.fixture(scope="session")
def A_mu1000_init():
    return np.array(
        [
            [
                9.1894070188323573e-02,
                1.7530349447685195e-01,
                1.7530349447685301e-01,
                2.7874947042898623e-01,
                2.7874947042898501e-01,
            ],
            [
                -1.9968968873256783e-16,
                2.3004217991775602e-01,
                -2.3004217991775622e-01,
                2.4680580481257552e-01,
                -2.4680580481257508e-01,
            ],
            [
                8.2063944467677424e-02,
                -2.1685904674531978e-01,
                -2.1685904674531978e-01,
                1.7582707451147850e-01,
                1.7582707451148361e-01,
            ],
            [
                -2.6690050013010803e-16,
                -1.5813354667522189e-01,
                1.5813354667521951e-01,
                3.0646480920422386e-01,
                -3.0646480920422126e-01,
            ],
            [
                -2.6659946136620399e-01,
                -4.6750705493354333e-02,
                -4.6750705493354631e-02,
                1.8005043617645519e-01,
                1.8005043617645786e-01,
            ],
        ],
        dtype=np.float64,
    )


@pytest.fixture(scope="session")
def A_mu1000():
    return np.array(
        [
            [
                1.1624352928638476e-01,
                1.4701250216352987e-01,
                1.4701250216353140e-01,
                2.9486573319327480e-01,
                2.9486573319327908e-01,
            ],
            [
                -2.0948669444255076e-16,
                1.9291729793399881e-01,
                -1.9291729793399898e-01,
                2.0697555985554686e-01,
                -2.0697555985554650e-01,
            ],
            [
                6.8820224317862555e-02,
                -1.8186169747481468e-01,
                -1.8186169747481468e-01,
                1.4745158531588126e-01,
                1.4745158531588554e-01,
            ],
            [
                -1.8621039506004511e-16,
                -1.3261349091810185e-01,
                1.3261349091809985e-01,
                2.5700661906731453e-01,
                -2.5700661906731237e-01,
            ],
            [
                -2.2357485803610139e-01,
                -3.9205939465147326e-02,
                -3.9205939465147575e-02,
                1.5099336848319703e-01,
                1.5099336848319928e-01,
            ],
        ],
        dtype=np.float64,
    )


@pytest.fixture(scope="session")
def svecs_mu1000():
    return np.array(
        [
            [
                1.0000000000000002e00,
                1.3488263927262381e00,
                -1.3104358647426639e00,
                -1.0433110064103177e00,
                -6.0005035322489464e-01,
            ],
            [
                1.0000000000000002e00,
                1.3990235434236424e00,
                -1.4737789968291495e00,
                -1.1266779128181197e00,
                -1.0896513975018769e-01,
            ],
            [
                1.0000000000000002e00,
                8.8508490246081195e-01,
                4.3510374799711093e-01,
                9.8936037088282069e-01,
                6.5386362361676154e-01,
            ],
            [
                1.0000000000000002e00,
                -1.5699540956179371e-15,
                1.4466972854287901e00,
                -2.8905778341536754e-15,
                -2.9609371304557919e00,
            ],
            [
                1.0000000000000002e00,
                7.2188567221853306e-01,
                6.8838572855835944e-01,
                1.0501513257627397e00,
                5.5658671487431721e-01,
            ],
            [
                1.0000000000000002e00,
                -7.2188567221853406e-01,
                6.8838572855836977e-01,
                -1.0501513257627297e00,
                5.5658671487432432e-01,
            ],
            [
                1.0000000000000002e00,
                -1.3488263927262389e00,
                -1.3104358647426779e00,
                1.0433110064103133e00,
                -6.0005035322490197e-01,
            ],
            [
                1.0000000000000002e00,
                -8.8508490246081095e-01,
                4.3510374799711915e-01,
                -9.8936037088281159e-01,
                6.5386362361676742e-01,
            ],
            [
                1.0000000000000002e00,
                -1.3990235434236402e00,
                -1.4737789968291592e00,
                1.1266779128180928e00,
                -1.0896513975019061e-01,
            ],
        ],
        dtype=np.float64,
    )


@pytest.fixture(scope="session")
def chi_isa_mu0_n3():
    return np.array(
        [
            [
                3.1115284081507055e-02,
                6.9962414237666952e-04,
                9.6818509177611567e-01,
            ],
            [
                -2.2306101257299255e-17,
                2.7238701370796497e-16,
                9.9999999999999922e-01,
            ],
            [
                2.3155133771019864e-16,
                9.5036998707745997e-16,
                9.9999999999999833e-01,
            ],
            [
                9.3793037467239537e-01,
                3.1034812663803282e-02,
                3.1034812663800999e-02,
            ],
            [
                9.9999999999999956e-01,
                -8.3121127279257571e-16,
                9.4045844324124387e-16,
            ],
            [
                9.9999999999999933e-01,
                -8.9620175089935441e-16,
                1.2085348725220209e-15,
            ],
            [
                3.1115284081505195e-02,
                9.6818509177611567e-01,
                6.9962414237894190e-04,
            ],
            [
                4.1989789562932371e-16,
                9.9999999999999956e-01,
                -3.2491549094043100e-16,
            ],
            [1.3725993281327047e-17, 1.0000000000000002e00, -5.6591874961648636e-16],
        ],
        dtype=np.float64,
    )


@pytest.fixture(scope="session")
def chi_isa_mu100_n3():
    return np.array(
        [
            [
                -4.2580228676599614e-03,
                4.2801027432808406e-02,
                9.6145699543485141e-01,
            ],
            [
                -1.4772846838530374e-15,
                5.0378037267330542e-17,
                1.0000000000000013e00,
            ],
            [
                -2.7309451028471876e-02,
                2.5981853655796916e-01,
                7.6749091447050255e-01,
            ],
            [
                2.5476043241765107e-15,
                1.0000000000000000e00,
                -2.7488885789634078e-15,
            ],
            [
                -2.3685358779863556e-02,
                8.3488897155844455e-01,
                1.8879638722141870e-01,
            ],
            [
                1.8879638722141809e-01,
                8.3488897155844921e-01,
                -2.3685358779867570e-02,
            ],
            [
                9.6145699543484642e-01,
                4.2801027432807524e-02,
                -4.2580228676540859e-03,
            ],
            [
                7.6749091447050422e-01,
                2.5981853655796533e-01,
                -2.7309451028469895e-02,
            ],
            [9.9999999999999867e-01, 1.8729778932510503e-17, 1.1929305952124212e-15],
        ],
        dtype=np.float64,
    )


@pytest.fixture(scope="session")
def count():
    return np.array(
        [
            [
                5.7000000e001,
                0.0000000e000,
                0.0000000e000,
                1.0000000e001,
                0.0000000e000,
                1.0000000e001,
                1.9000000e001,
                2.8000000e001,
                3.8000000e001,
                0.0000000e000,
                1.9000000e001,
                1.0000000e001,
                0.0000000e000,
                0.0000000e000,
                1.0000000e001,
                0.0000000e000,
                1.0000000e001,
                0.0000000e000,
                0.0000000e000,
                0.0000000e000,
                0.0000000e000,
                0.0000000e000,
                0.0000000e000,
                0.0000000e000,
                0.0000000e000,
                1.0000000e000,
                0.0000000e000,
                0.0000000e000,
                0.0000000e000,
                0.0000000e000,
                0.0000000e000,
                0.0000000e000,
                0.0000000e000,
                0.0000000e000,
                0.0000000e000,
                0.0000000e000,
                0.0000000e000,
                0.0000000e000,
                0.0000000e000,
                0.0000000e000,
                0.0000000e000,
                0.0000000e000,
            ],
            [
                1.0000000e000,
                8.4000000e001,
                3.5000000e001,
                6.3000000e001,
                4.7000000e001,
                6.2000000e001,
                6.3000000e001,
                4.2000000e001,
                3.2000000e001,
                2.8000000e001,
                1.6000000e001,
                9.0000000e000,
                6.0000000e000,
                5.0000000e000,
                3.0000000e000,
                5.0000000e000,
                2.0000000e000,
                3.0000000e000,
                2.0000000e000,
                4.0000000e000,
                1.0000000e000,
                1.0000000e000,
                4.0000000e000,
                4.0000000e000,
                2.0000000e000,
                2.0000000e000,
                1.0000000e000,
                4.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
                2.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
                0.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
            ],
            [
                2.0000000e000,
                2.5000000e001,
                4.0200000e002,
                3.6400000e002,
                4.0600000e002,
                4.3100000e002,
                2.9000000e002,
                1.4600000e002,
                6.0000000e001,
                3.8000000e001,
                3.6000000e001,
                2.1000000e001,
                1.1000000e001,
                9.0000000e000,
                2.0000000e000,
                1.2000000e001,
                3.0000000e000,
                6.0000000e000,
                7.0000000e000,
                4.0000000e000,
                5.0000000e000,
                3.0000000e000,
                5.0000000e000,
                4.0000000e000,
                1.0000000e000,
                5.0000000e000,
                6.0000000e000,
                3.0000000e000,
                3.0000000e000,
                2.0000000e000,
                4.0000000e000,
                3.0000000e000,
                2.0000000e000,
                2.0000000e000,
                1.0000000e000,
                3.0000000e000,
                1.0000000e000,
                4.0000000e000,
                2.0000000e000,
                3.0000000e000,
                3.0000000e000,
                3.0000000e000,
            ],
            [
                1.5000000e001,
                5.0000000e001,
                3.4100000e002,
                1.4960000e003,
                1.5600000e003,
                1.1640000e003,
                5.1700000e002,
                1.9200000e002,
                1.0600000e002,
                5.1000000e001,
                3.0000000e001,
                2.2000000e001,
                1.5000000e001,
                1.0000000e001,
                1.2000000e001,
                1.5000000e001,
                1.3000000e001,
                1.4000000e001,
                1.1000000e001,
                4.0000000e000,
                9.0000000e000,
                5.0000000e000,
                1.5000000e001,
                8.0000000e000,
                7.0000000e000,
                6.0000000e000,
                9.0000000e000,
                8.0000000e000,
                9.0000000e000,
                4.0000000e000,
                1.1000000e001,
                1.2000000e001,
                6.0000000e000,
                1.2000000e001,
                6.0000000e000,
                8.0000000e000,
                3.0000000e000,
                1.2000000e001,
                8.0000000e000,
                1.1000000e001,
                1.1000000e001,
                5.0000000e000,
            ],
            [
                1.0000000e001,
                4.3000000e001,
                3.7200000e002,
                1.5460000e003,
                2.6530000e003,
                1.4030000e003,
                4.9600000e002,
                1.8800000e002,
                1.1600000e002,
                4.8000000e001,
                2.5000000e001,
                1.5000000e001,
                1.6000000e001,
                1.2000000e001,
                1.3000000e001,
                9.0000000e000,
                1.3000000e001,
                1.6000000e001,
                1.1000000e001,
                1.4000000e001,
                1.0000000e001,
                1.0000000e001,
                1.2000000e001,
                1.3000000e001,
                1.3000000e001,
                7.0000000e000,
                1.0000000e001,
                1.6000000e001,
                1.2000000e001,
                1.3000000e001,
                9.0000000e000,
                9.0000000e000,
                1.1000000e001,
                7.0000000e000,
                7.0000000e000,
                1.4000000e001,
                6.0000000e000,
                1.5000000e001,
                1.3000000e001,
                5.0000000e000,
                1.0000000e001,
                8.0000000e000,
            ],
            [
                5.0000000e000,
                4.7000000e001,
                4.0200000e002,
                1.1810000e003,
                1.4330000e003,
                1.3190000e003,
                4.6200000e002,
                1.8200000e002,
                8.1000000e001,
                4.6000000e001,
                2.0000000e001,
                2.0000000e001,
                1.1000000e001,
                8.0000000e000,
                6.0000000e000,
                9.0000000e000,
                4.0000000e000,
                6.0000000e000,
                2.0000000e000,
                5.0000000e000,
                4.0000000e000,
                3.0000000e000,
                1.0000000e001,
                1.0000000e001,
                8.0000000e000,
                8.0000000e000,
                7.0000000e000,
                4.0000000e000,
                4.0000000e000,
                6.0000000e000,
                8.0000000e000,
                1.0000000e001,
                9.0000000e000,
                6.0000000e000,
                5.0000000e000,
                7.0000000e000,
                7.0000000e000,
                6.0000000e000,
                7.0000000e000,
                8.0000000e000,
                7.0000000e000,
                1.2000000e001,
            ],
            [
                6.0000000e000,
                4.7000000e001,
                2.8400000e002,
                5.4700000e002,
                5.3200000e002,
                4.8500000e002,
                4.9900000e002,
                1.5000000e002,
                8.4000000e001,
                2.9000000e001,
                1.8000000e001,
                6.0000000e000,
                1.0000000e001,
                8.0000000e000,
                7.0000000e000,
                8.0000000e000,
                3.0000000e000,
                5.0000000e000,
                3.0000000e000,
                4.0000000e000,
                6.0000000e000,
                8.0000000e000,
                4.0000000e000,
                7.0000000e000,
                2.0000000e000,
                6.0000000e000,
                3.0000000e000,
                3.0000000e000,
                5.0000000e000,
                2.0000000e000,
                5.0000000e000,
                6.0000000e000,
                4.0000000e000,
                5.0000000e000,
                3.0000000e000,
                4.0000000e000,
                4.0000000e000,
                5.0000000e000,
                5.0000000e000,
                6.0000000e000,
                3.0000000e000,
                5.0000000e000,
            ],
            [
                5.0000000e000,
                3.4000000e001,
                1.5300000e002,
                2.1900000e002,
                2.1600000e002,
                2.1000000e002,
                1.6500000e002,
                2.2500000e002,
                8.1000000e001,
                3.2000000e001,
                1.8000000e001,
                1.3000000e001,
                1.0000000e001,
                5.0000000e000,
                6.0000000e000,
                5.0000000e000,
                3.0000000e000,
                4.0000000e000,
                4.0000000e000,
                3.0000000e000,
                5.0000000e000,
                2.0000000e000,
                2.0000000e000,
                5.0000000e000,
                1.0000000e000,
                5.0000000e000,
                3.0000000e000,
                4.0000000e000,
                3.0000000e000,
                2.0000000e000,
                2.0000000e000,
                3.0000000e000,
                2.0000000e000,
                3.0000000e000,
                3.0000000e000,
                2.0000000e000,
                2.0000000e000,
                2.0000000e000,
                2.0000000e000,
                2.0000000e000,
                1.0000000e000,
                2.0000000e000,
            ],
            [
                6.0000000e000,
                2.8000000e001,
                7.1000000e001,
                1.3600000e002,
                1.4300000e002,
                9.8000000e001,
                1.0400000e002,
                9.3000000e001,
                1.1800000e002,
                3.4000000e001,
                2.5000000e001,
                1.4000000e001,
                1.0000000e001,
                5.0000000e000,
                5.0000000e000,
                2.0000000e000,
                5.0000000e000,
                1.0000000e000,
                2.0000000e000,
                4.0000000e000,
                1.0000000e000,
                3.0000000e000,
                2.0000000e000,
                1.0000000e000,
                4.0000000e000,
                4.0000000e000,
                5.0000000e000,
                1.0000000e000,
                4.0000000e000,
                3.0000000e000,
                2.0000000e000,
                1.0000000e000,
                2.0000000e000,
                2.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
                0.0000000e000,
                1.0000000e000,
                1.0000000e000,
                2.0000000e000,
            ],
            [
                1.0000000e000,
                2.8000000e001,
                5.0000000e001,
                7.7000000e001,
                5.9000000e001,
                5.9000000e001,
                3.7000000e001,
                3.8000000e001,
                3.8000000e001,
                8.1000000e001,
                3.7000000e001,
                2.2000000e001,
                9.0000000e000,
                3.0000000e000,
                5.0000000e000,
                6.0000000e000,
                5.0000000e000,
                2.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
                0.0000000e000,
                2.0000000e000,
                3.0000000e000,
                3.0000000e000,
                0.0000000e000,
                4.0000000e000,
                1.0000000e000,
                2.0000000e000,
                1.0000000e000,
                2.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
                0.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
            ],
            [
                4.0000000e000,
                1.9000000e001,
                5.7000000e001,
                3.7000000e001,
                3.0000000e001,
                2.5000000e001,
                2.5000000e001,
                2.6000000e001,
                3.2000000e001,
                4.3000000e001,
                8.3000000e001,
                1.7000000e001,
                1.5000000e001,
                1.0000000e001,
                1.2000000e001,
                6.0000000e000,
                6.0000000e000,
                7.0000000e000,
                6.0000000e000,
                6.0000000e000,
                4.0000000e000,
                4.0000000e000,
                9.0000000e000,
                3.0000000e000,
                6.0000000e000,
                1.0000000e000,
                7.0000000e000,
                1.0000000e000,
                2.0000000e000,
                1.0000000e000,
                0.0000000e000,
                1.0000000e000,
                1.0000000e000,
                2.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
                0.0000000e000,
                0.0000000e000,
                0.0000000e000,
                1.0000000e000,
                1.0000000e000,
            ],
            [
                3.0000000e000,
                1.2000000e001,
                3.8000000e001,
                3.0000000e001,
                1.8000000e001,
                2.4000000e001,
                9.0000000e000,
                2.0000000e001,
                2.2000000e001,
                3.0000000e001,
                2.0000000e001,
                5.2000000e001,
                3.0000000e001,
                1.8000000e001,
                8.0000000e000,
                3.0000000e000,
                3.0000000e000,
                3.0000000e000,
                3.0000000e000,
                3.0000000e000,
                1.0000000e000,
                1.0000000e000,
                7.0000000e000,
                9.0000000e000,
                5.0000000e000,
                1.2000000e001,
                4.0000000e000,
                7.0000000e000,
                3.0000000e000,
                0.0000000e000,
                3.0000000e000,
                2.0000000e000,
                1.0000000e000,
                2.0000000e000,
                0.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
                3.0000000e000,
                1.0000000e000,
            ],
            [
                1.0000000e000,
                9.0000000e000,
                1.7000000e001,
                7.0000000e000,
                2.0000000e001,
                5.0000000e000,
                1.5000000e001,
                1.6000000e001,
                1.5000000e001,
                1.3000000e001,
                2.0000000e001,
                3.2000000e001,
                3.8000000e001,
                2.8000000e001,
                1.9000000e001,
                5.0000000e000,
                3.0000000e000,
                9.0000000e000,
                7.0000000e000,
                3.0000000e000,
                7.0000000e000,
                9.0000000e000,
                5.0000000e000,
                5.0000000e000,
                7.0000000e000,
                7.0000000e000,
                5.0000000e000,
                1.1000000e001,
                7.0000000e000,
                3.0000000e000,
                3.0000000e000,
                3.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
                0.0000000e000,
                1.0000000e000,
                0.0000000e000,
                0.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
            ],
            [
                1.0000000e000,
                6.0000000e000,
                8.0000000e000,
                8.0000000e000,
                4.0000000e000,
                4.0000000e000,
                9.0000000e000,
                4.0000000e000,
                6.0000000e000,
                4.0000000e000,
                1.2000000e001,
                1.7000000e001,
                2.4000000e001,
                5.5000000e001,
                3.2000000e001,
                1.5000000e001,
                8.0000000e000,
                1.2000000e001,
                1.2000000e001,
                9.0000000e000,
                1.0000000e001,
                8.0000000e000,
                1.7000000e001,
                1.1000000e001,
                1.5000000e001,
                1.9000000e001,
                1.2000000e001,
                1.2000000e001,
                8.0000000e000,
                1.0000000e000,
                1.0000000e000,
                2.0000000e000,
                0.0000000e000,
                2.0000000e000,
                0.0000000e000,
                2.0000000e000,
                1.0000000e000,
                0.0000000e000,
                0.0000000e000,
                0.0000000e000,
                0.0000000e000,
                0.0000000e000,
            ],
            [
                2.0000000e000,
                4.0000000e000,
                0.0000000e000,
                1.0000000e001,
                9.0000000e000,
                2.0000000e000,
                4.0000000e000,
                5.0000000e000,
                5.0000000e000,
                5.0000000e000,
                1.1000000e001,
                7.0000000e000,
                1.4000000e001,
                2.7000000e001,
                6.8000000e001,
                3.9000000e001,
                1.9000000e001,
                1.7000000e001,
                1.1000000e001,
                1.9000000e001,
                1.0000000e001,
                1.0000000e001,
                8.0000000e000,
                2.2000000e001,
                2.5000000e001,
                2.6000000e001,
                1.6000000e001,
                1.6000000e001,
                5.0000000e000,
                6.0000000e000,
                2.0000000e000,
                4.0000000e000,
                2.0000000e000,
                1.0000000e000,
                2.0000000e000,
                2.0000000e000,
                1.0000000e000,
                2.0000000e000,
                0.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
            ],
            [
                1.0000000e000,
                4.0000000e000,
                7.0000000e000,
                7.0000000e000,
                1.0000000e000,
                5.0000000e000,
                4.0000000e000,
                4.0000000e000,
                2.0000000e000,
                4.0000000e000,
                4.0000000e000,
                2.0000000e000,
                3.0000000e000,
                9.0000000e000,
                2.5000000e001,
                9.1000000e001,
                5.7000000e001,
                2.9000000e001,
                1.8000000e001,
                1.8000000e001,
                2.0000000e001,
                2.1000000e001,
                3.2000000e001,
                3.7000000e001,
                4.0000000e001,
                3.0000000e001,
                3.5000000e001,
                1.7000000e001,
                8.0000000e000,
                8.0000000e000,
                4.0000000e000,
                1.0000000e000,
                2.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
                0.0000000e000,
            ],
            [
                2.0000000e000,
                2.0000000e000,
                1.0000000e000,
                3.0000000e000,
                3.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
                3.0000000e000,
                3.0000000e000,
                3.0000000e000,
                2.0000000e000,
                2.0000000e000,
                4.0000000e000,
                9.0000000e000,
                4.2000000e001,
                9.1000000e001,
                5.0000000e001,
                4.0000000e001,
                3.4000000e001,
                4.4000000e001,
                4.9000000e001,
                6.9000000e001,
                7.2000000e001,
                7.5000000e001,
                5.9000000e001,
                3.2000000e001,
                9.0000000e000,
                6.0000000e000,
                3.0000000e000,
                3.0000000e000,
                3.0000000e000,
                2.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
            ],
            [
                2.0000000e000,
                2.0000000e000,
                3.0000000e000,
                4.0000000e000,
                3.0000000e000,
                3.0000000e000,
                3.0000000e000,
                3.0000000e000,
                3.0000000e000,
                1.0000000e000,
                5.0000000e000,
                4.0000000e000,
                4.0000000e000,
                5.0000000e000,
                8.0000000e000,
                1.6000000e001,
                3.5000000e001,
                2.0100000e002,
                1.2100000e002,
                1.0800000e002,
                1.2600000e002,
                1.4100000e002,
                1.5500000e002,
                1.7700000e002,
                1.4200000e002,
                5.8000000e001,
                1.9000000e001,
                7.0000000e000,
                7.0000000e000,
                5.0000000e000,
                3.0000000e000,
                2.0000000e000,
                2.0000000e000,
                3.0000000e000,
                3.0000000e000,
                1.0000000e000,
                2.0000000e000,
                2.0000000e000,
                3.0000000e000,
                1.0000000e000,
                2.0000000e000,
                2.0000000e000,
            ],
            [
                3.0000000e000,
                5.0000000e000,
                5.0000000e000,
                6.0000000e000,
                5.0000000e000,
                5.0000000e000,
                5.0000000e000,
                6.0000000e000,
                5.0000000e000,
                4.0000000e000,
                6.0000000e000,
                7.0000000e000,
                4.0000000e000,
                8.0000000e000,
                7.0000000e000,
                1.0000000e001,
                2.4000000e001,
                1.0100000e002,
                4.8700000e002,
                3.6000000e002,
                3.5600000e002,
                4.1000000e002,
                4.9400000e002,
                4.1700000e002,
                1.5200000e002,
                5.1000000e001,
                1.9000000e001,
                1.0000000e001,
                8.0000000e000,
                5.0000000e000,
                6.0000000e000,
                4.0000000e000,
                8.0000000e000,
                4.0000000e000,
                7.0000000e000,
                3.0000000e000,
                5.0000000e000,
                3.0000000e000,
                5.0000000e000,
                2.0000000e000,
                6.0000000e000,
                7.0000000e000,
            ],
            [
                1.2000000e001,
                8.0000000e000,
                1.4000000e001,
                1.0000000e000,
                7.0000000e000,
                5.0000000e000,
                9.0000000e000,
                1.1000000e001,
                1.1000000e001,
                1.0000000e001,
                8.0000000e000,
                6.0000000e000,
                8.0000000e000,
                9.0000000e000,
                1.7000000e001,
                1.0000000e001,
                2.4000000e001,
                8.8000000e001,
                3.3200000e002,
                1.3010000e003,
                1.0780000e003,
                1.1570000e003,
                1.1020000e003,
                4.2300000e002,
                1.0400000e002,
                4.4000000e001,
                1.9000000e001,
                1.5000000e001,
                1.6000000e001,
                9.0000000e000,
                1.2000000e001,
                8.0000000e000,
                1.2000000e001,
                6.0000000e000,
                9.0000000e000,
                1.0000000e001,
                8.0000000e000,
                6.0000000e000,
                1.0000000e001,
                1.5000000e001,
                8.0000000e000,
                1.4000000e001,
            ],
            [
                1.1000000e001,
                1.1000000e001,
                7.0000000e000,
                1.4000000e001,
                1.5000000e001,
                2.0000000e001,
                1.1000000e001,
                1.5000000e001,
                1.3000000e001,
                6.0000000e000,
                1.1000000e001,
                1.8000000e001,
                1.6000000e001,
                1.5000000e001,
                7.0000000e000,
                2.2000000e001,
                4.0000000e001,
                9.7000000e001,
                3.2400000e002,
                1.0460000e003,
                2.8300000e003,
                2.3920000e003,
                1.0870000e003,
                3.4300000e002,
                1.0000000e002,
                4.7000000e001,
                1.5000000e001,
                7.0000000e000,
                1.4000000e001,
                1.6000000e001,
                1.0000000e001,
                1.6000000e001,
                8.0000000e000,
                9.0000000e000,
                1.4000000e001,
                1.5000000e001,
                1.7000000e001,
                1.9000000e001,
                1.3000000e001,
                1.1000000e001,
                1.3000000e001,
                1.3000000e001,
            ],
            [
                9.0000000e000,
                1.1000000e001,
                1.3000000e001,
                6.0000000e000,
                9.0000000e000,
                3.0000000e000,
                1.8000000e001,
                6.0000000e000,
                9.0000000e000,
                1.3000000e001,
                1.3000000e001,
                1.5000000e001,
                1.8000000e001,
                1.7000000e001,
                1.8000000e001,
                1.3000000e001,
                3.6000000e001,
                1.0300000e002,
                3.6500000e002,
                1.1180000e003,
                2.4030000e003,
                2.9030000e003,
                1.0350000e003,
                3.1700000e002,
                9.9000000e001,
                4.6000000e001,
                2.1000000e001,
                1.7000000e001,
                9.0000000e000,
                1.0000000e001,
                1.3000000e001,
                1.6000000e001,
                1.4000000e001,
                1.5000000e001,
                8.0000000e000,
                9.0000000e000,
                6.0000000e000,
                1.3000000e001,
                1.5000000e001,
                1.0000000e001,
                1.8000000e001,
                2.2000000e001,
            ],
            [
                1.3000000e001,
                1.0000000e001,
                1.1000000e001,
                7.0000000e000,
                1.3000000e001,
                8.0000000e000,
                1.1000000e001,
                7.0000000e000,
                1.3000000e001,
                1.0000000e001,
                1.2000000e001,
                9.0000000e000,
                9.0000000e000,
                1.0000000e001,
                9.0000000e000,
                2.1000000e001,
                4.1000000e001,
                1.2400000e002,
                4.5200000e002,
                1.1080000e003,
                1.1210000e003,
                1.0660000e003,
                1.3730000e003,
                3.3900000e002,
                8.6000000e001,
                3.4000000e001,
                1.5000000e001,
                1.7000000e001,
                1.2000000e001,
                5.0000000e000,
                9.0000000e000,
                9.0000000e000,
                8.0000000e000,
                9.0000000e000,
                8.0000000e000,
                6.0000000e000,
                1.3000000e001,
                8.0000000e000,
                8.0000000e000,
                9.0000000e000,
                1.0000000e001,
                5.0000000e000,
            ],
            [
                6.0000000e000,
                6.0000000e000,
                5.0000000e000,
                6.0000000e000,
                4.0000000e000,
                5.0000000e000,
                7.0000000e000,
                5.0000000e000,
                3.0000000e000,
                4.0000000e000,
                6.0000000e000,
                5.0000000e000,
                6.0000000e000,
                4.0000000e000,
                1.1000000e001,
                1.7000000e001,
                4.4000000e001,
                1.4500000e002,
                4.1800000e002,
                4.6100000e002,
                3.8800000e002,
                3.4900000e002,
                3.6600000e002,
                4.9800000e002,
                1.0300000e002,
                2.4000000e001,
                8.0000000e000,
                9.0000000e000,
                9.0000000e000,
                3.0000000e000,
                5.0000000e000,
                4.0000000e000,
                4.0000000e000,
                5.0000000e000,
                5.0000000e000,
                4.0000000e000,
                4.0000000e000,
                6.0000000e000,
                6.0000000e000,
                7.0000000e000,
                3.0000000e000,
                5.0000000e000,
            ],
            [
                2.0000000e000,
                2.0000000e000,
                2.0000000e000,
                2.0000000e000,
                3.0000000e000,
                3.0000000e000,
                3.0000000e000,
                1.0000000e000,
                3.0000000e000,
                3.0000000e000,
                3.0000000e000,
                3.0000000e000,
                3.0000000e000,
                6.0000000e000,
                1.0000000e001,
                2.1000000e001,
                5.4000000e001,
                1.4400000e002,
                1.8600000e002,
                1.3800000e002,
                1.3200000e002,
                1.2900000e002,
                1.1200000e002,
                1.2700000e002,
                2.0000000e002,
                3.4000000e001,
                1.5000000e001,
                8.0000000e000,
                5.0000000e000,
                2.0000000e000,
                3.0000000e000,
                4.0000000e000,
                2.0000000e000,
                2.0000000e000,
                3.0000000e000,
                1.0000000e000,
                3.0000000e000,
                2.0000000e000,
                3.0000000e000,
                2.0000000e000,
                3.0000000e000,
                2.0000000e000,
            ],
            [
                1.0000000e000,
                2.0000000e000,
                2.0000000e000,
                2.0000000e000,
                1.0000000e000,
                0.0000000e000,
                2.0000000e000,
                2.0000000e000,
                2.0000000e000,
                1.0000000e000,
                1.0000000e000,
                5.0000000e000,
                3.0000000e000,
                8.0000000e000,
                1.3000000e001,
                2.2000000e001,
                5.7000000e001,
                8.1000000e001,
                8.2000000e001,
                6.8000000e001,
                6.0000000e001,
                5.4000000e001,
                4.7000000e001,
                3.8000000e001,
                4.8000000e001,
                1.0200000e002,
                2.5000000e001,
                1.1000000e001,
                6.0000000e000,
                4.0000000e000,
                4.0000000e000,
                2.0000000e000,
                2.0000000e000,
                1.0000000e000,
                3.0000000e000,
                1.0000000e000,
                3.0000000e000,
                3.0000000e000,
                2.0000000e000,
                1.0000000e000,
                2.0000000e000,
                1.0000000e000,
            ],
            [
                1.0000000e000,
                1.0000000e000,
                4.0000000e000,
                1.0000000e000,
                2.0000000e000,
                1.0000000e000,
                1.0000000e000,
                2.0000000e000,
                4.0000000e000,
                3.0000000e000,
                5.0000000e000,
                2.0000000e000,
                3.0000000e000,
                7.0000000e000,
                1.1000000e001,
                3.6000000e001,
                4.5000000e001,
                3.8000000e001,
                4.0000000e001,
                3.6000000e001,
                1.9000000e001,
                2.1000000e001,
                1.8000000e001,
                1.8000000e001,
                2.5000000e001,
                3.5000000e001,
                8.3000000e001,
                1.8000000e001,
                1.0000000e001,
                3.0000000e000,
                3.0000000e000,
                2.0000000e000,
                5.0000000e000,
                2.0000000e000,
                3.0000000e000,
                4.0000000e000,
                3.0000000e000,
                4.0000000e000,
                9.0000000e000,
                2.0000000e000,
                6.0000000e000,
                2.0000000e000,
            ],
            [
                0.0000000e000,
                4.0000000e000,
                0.0000000e000,
                2.0000000e000,
                1.0000000e000,
                4.0000000e000,
                1.0000000e000,
                2.0000000e000,
                1.0000000e000,
                1.0000000e000,
                0.0000000e000,
                5.0000000e000,
                9.0000000e000,
                1.1000000e001,
                1.7000000e001,
                2.7000000e001,
                1.7000000e001,
                2.2000000e001,
                2.1000000e001,
                1.3000000e001,
                9.0000000e000,
                4.0000000e000,
                1.6000000e001,
                1.2000000e001,
                1.9000000e001,
                2.3000000e001,
                2.8000000e001,
                4.0000000e001,
                1.5000000e001,
                1.2000000e001,
                7.0000000e000,
                1.3000000e001,
                4.0000000e000,
                2.0000000e000,
                4.0000000e000,
                5.0000000e000,
                4.0000000e000,
                9.0000000e000,
                3.0000000e000,
                4.0000000e000,
                4.0000000e000,
                1.0000000e000,
            ],
            [
                1.0000000e000,
                1.0000000e000,
                0.0000000e000,
                1.0000000e000,
                2.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
                4.0000000e000,
                3.0000000e000,
                2.0000000e000,
                2.0000000e000,
                6.0000000e000,
                7.0000000e000,
                6.0000000e000,
                1.4000000e001,
                1.3000000e001,
                1.5000000e001,
                1.3000000e001,
                2.0000000e001,
                1.6000000e001,
                4.0000000e000,
                9.0000000e000,
                1.1000000e001,
                1.3000000e001,
                1.3000000e001,
                1.6000000e001,
                1.6000000e001,
                5.7000000e001,
                2.9000000e001,
                1.6000000e001,
                2.0000000e001,
                6.0000000e000,
                0.0000000e000,
                1.1000000e001,
                4.0000000e000,
                1.0000000e001,
                6.0000000e000,
                2.0000000e001,
                6.0000000e000,
                4.0000000e000,
                1.0000000e000,
            ],
            [
                0.0000000e000,
                0.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
                0.0000000e000,
                0.0000000e000,
                0.0000000e000,
                3.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
                2.0000000e000,
                1.0000000e000,
                9.0000000e000,
                1.5000000e001,
                5.0000000e000,
                9.0000000e000,
                1.1000000e001,
                5.0000000e000,
                5.0000000e000,
                5.0000000e000,
                3.0000000e000,
                9.0000000e000,
                3.0000000e000,
                7.0000000e000,
                4.0000000e000,
                1.5000000e001,
                3.5000000e001,
                5.1000000e001,
                1.5000000e001,
                7.0000000e000,
                5.0000000e000,
                1.1000000e001,
                5.0000000e000,
                1.1000000e001,
                1.1000000e001,
                2.0000000e001,
                9.0000000e000,
                1.5000000e001,
                9.0000000e000,
                3.0000000e000,
            ],
            [
                1.0000000e000,
                1.0000000e000,
                0.0000000e000,
                1.0000000e000,
                1.0000000e000,
                0.0000000e000,
                1.0000000e000,
                0.0000000e000,
                2.0000000e000,
                2.0000000e000,
                0.0000000e000,
                2.0000000e000,
                3.0000000e000,
                1.0000000e000,
                3.0000000e000,
                6.0000000e000,
                6.0000000e000,
                5.0000000e000,
                6.0000000e000,
                2.0000000e000,
                6.0000000e000,
                4.0000000e000,
                5.0000000e000,
                5.0000000e000,
                8.0000000e000,
                8.0000000e000,
                4.0000000e000,
                8.0000000e000,
                1.8000000e001,
                1.4000000e001,
                5.8000000e001,
                3.1000000e001,
                1.6000000e001,
                1.2000000e001,
                1.7000000e001,
                1.4000000e001,
                2.2000000e001,
                2.3000000e001,
                2.7000000e001,
                2.6000000e001,
                8.0000000e000,
                0.0000000e000,
            ],
            [
                0.0000000e000,
                2.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
                0.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
                0.0000000e000,
                1.0000000e000,
                3.0000000e000,
                2.0000000e000,
                2.0000000e000,
                5.0000000e000,
                1.0000000e000,
                6.0000000e000,
                1.0000000e000,
                3.0000000e000,
                5.0000000e000,
                0.0000000e000,
                2.0000000e000,
                2.0000000e000,
                1.0000000e000,
                8.0000000e000,
                3.0000000e000,
                3.0000000e000,
                1.5000000e001,
                2.1000000e001,
                6.0000000e000,
                3.0000000e001,
                9.6000000e001,
                3.3000000e001,
                3.3000000e001,
                2.8000000e001,
                3.3000000e001,
                3.9000000e001,
                3.4000000e001,
                4.1000000e001,
                4.5000000e001,
                2.1000000e001,
                6.0000000e000,
            ],
            [
                1.0000000e000,
                1.0000000e000,
                0.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
                0.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
                2.0000000e000,
                2.0000000e000,
                4.0000000e000,
                2.0000000e000,
                4.0000000e000,
                5.0000000e000,
                1.0000000e000,
                3.0000000e000,
                1.0000000e000,
                2.0000000e000,
                5.0000000e000,
                4.0000000e000,
                6.0000000e000,
                4.0000000e000,
                5.0000000e000,
                3.0000000e000,
                1.2000000e001,
                2.6000000e001,
                9.4000000e001,
                2.9000000e001,
                3.7000000e001,
                5.5000000e001,
                6.0000000e001,
                5.9000000e001,
                6.7000000e001,
                5.0000000e001,
                2.5000000e001,
                2.0000000e000,
            ],
            [
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
                2.0000000e000,
                2.0000000e000,
                3.0000000e000,
                1.0000000e000,
                2.0000000e000,
                1.0000000e000,
                2.0000000e000,
                1.0000000e000,
                3.0000000e000,
                3.0000000e000,
                0.0000000e000,
                5.0000000e000,
                1.0000000e000,
                3.0000000e000,
                1.0000000e000,
                2.0000000e000,
                1.0000000e000,
                4.0000000e000,
                2.0000000e000,
                2.0000000e000,
                8.0000000e000,
                8.0000000e000,
                2.3000000e001,
                2.5000000e001,
                8.3000000e001,
                5.1000000e001,
                8.2000000e001,
                8.8000000e001,
                1.1500000e002,
                1.1000000e002,
                9.3000000e001,
                3.0000000e001,
                2.0000000e000,
            ],
            [
                2.0000000e000,
                1.0000000e000,
                3.0000000e000,
                1.0000000e000,
                2.0000000e000,
                1.0000000e000,
                2.0000000e000,
                2.0000000e000,
                2.0000000e000,
                2.0000000e000,
                2.0000000e000,
                2.0000000e000,
                3.0000000e000,
                3.0000000e000,
                3.0000000e000,
                2.0000000e000,
                2.0000000e000,
                2.0000000e000,
                2.0000000e000,
                5.0000000e000,
                2.0000000e000,
                5.0000000e000,
                2.0000000e000,
                2.0000000e000,
                3.0000000e000,
                5.0000000e000,
                4.0000000e000,
                4.0000000e000,
                8.0000000e000,
                5.0000000e000,
                1.2000000e001,
                1.9000000e001,
                3.1000000e001,
                4.6000000e001,
                2.2800000e002,
                1.5100000e002,
                2.0600000e002,
                2.5100000e002,
                2.3500000e002,
                1.4900000e002,
                4.5000000e001,
                5.0000000e000,
            ],
            [
                4.0000000e000,
                3.0000000e000,
                4.0000000e000,
                5.0000000e000,
                4.0000000e000,
                5.0000000e000,
                6.0000000e000,
                3.0000000e000,
                3.0000000e000,
                4.0000000e000,
                5.0000000e000,
                7.0000000e000,
                5.0000000e000,
                6.0000000e000,
                6.0000000e000,
                4.0000000e000,
                4.0000000e000,
                2.0000000e000,
                4.0000000e000,
                0.0000000e000,
                3.0000000e000,
                6.0000000e000,
                0.0000000e000,
                4.0000000e000,
                6.0000000e000,
                5.0000000e000,
                6.0000000e000,
                7.0000000e000,
                6.0000000e000,
                9.0000000e000,
                1.0000000e001,
                2.1000000e001,
                4.1000000e001,
                6.7000000e001,
                1.3900000e002,
                4.8100000e002,
                4.4000000e002,
                5.8300000e002,
                5.3400000e002,
                2.8300000e002,
                4.6000000e001,
                6.0000000e000,
            ],
            [
                5.0000000e000,
                5.0000000e000,
                4.0000000e000,
                7.0000000e000,
                7.0000000e000,
                5.0000000e000,
                1.1000000e001,
                6.0000000e000,
                9.0000000e000,
                6.0000000e000,
                6.0000000e000,
                3.0000000e000,
                1.0000000e001,
                9.0000000e000,
                8.0000000e000,
                8.0000000e000,
                9.0000000e000,
                6.0000000e000,
                6.0000000e000,
                1.0000000e001,
                7.0000000e000,
                1.2000000e001,
                8.0000000e000,
                7.0000000e000,
                1.3000000e001,
                1.2000000e001,
                1.2000000e001,
                8.0000000e000,
                1.0000000e001,
                1.6000000e001,
                1.7000000e001,
                2.8000000e001,
                4.8000000e001,
                7.3000000e001,
                1.8200000e002,
                4.2000000e002,
                1.3210000e003,
                1.4520000e003,
                1.2010000e003,
                3.4400000e002,
                4.7000000e001,
                1.2000000e001,
            ],
            [
                6.0000000e000,
                1.2000000e001,
                1.2000000e001,
                7.0000000e000,
                1.6000000e001,
                1.6000000e001,
                8.0000000e000,
                1.6000000e001,
                1.0000000e001,
                9.0000000e000,
                1.5000000e001,
                1.3000000e001,
                5.0000000e000,
                9.0000000e000,
                1.3000000e001,
                7.0000000e000,
                3.0000000e000,
                9.0000000e000,
                9.0000000e000,
                1.2000000e001,
                1.0000000e001,
                1.1000000e001,
                1.7000000e001,
                5.0000000e000,
                1.5000000e001,
                7.0000000e000,
                1.5000000e001,
                1.9000000e001,
                1.8000000e001,
                1.4000000e001,
                1.5000000e001,
                2.7000000e001,
                4.6000000e001,
                9.5000000e001,
                2.2800000e002,
                5.5400000e002,
                1.4250000e003,
                2.7170000e003,
                1.5460000e003,
                3.8600000e002,
                4.0000000e001,
                1.4000000e001,
            ],
            [
                1.0000000e001,
                8.0000000e000,
                7.0000000e000,
                7.0000000e000,
                1.1000000e001,
                5.0000000e000,
                8.0000000e000,
                8.0000000e000,
                5.0000000e000,
                9.0000000e000,
                8.0000000e000,
                1.0000000e001,
                9.0000000e000,
                8.0000000e000,
                9.0000000e000,
                5.0000000e000,
                1.2000000e001,
                1.2000000e001,
                1.3000000e001,
                7.0000000e000,
                6.0000000e000,
                1.0000000e001,
                8.0000000e000,
                6.0000000e000,
                1.0000000e001,
                8.0000000e000,
                1.3000000e001,
                1.1000000e001,
                2.0000000e001,
                1.2000000e001,
                2.4000000e001,
                2.8000000e001,
                4.9000000e001,
                9.2000000e001,
                2.1000000e002,
                5.1100000e002,
                1.1920000e003,
                1.5660000e003,
                1.5340000e003,
                3.3000000e002,
                4.4000000e001,
                1.2000000e001,
            ],
            [
                3.0000000e000,
                3.0000000e000,
                4.0000000e000,
                1.0000000e000,
                4.0000000e000,
                4.0000000e000,
                4.0000000e000,
                6.0000000e000,
                2.0000000e000,
                3.0000000e000,
                2.0000000e000,
                4.0000000e000,
                3.0000000e000,
                4.0000000e000,
                4.0000000e000,
                4.0000000e000,
                2.0000000e000,
                3.0000000e000,
                2.0000000e000,
                7.0000000e000,
                2.0000000e000,
                5.0000000e000,
                5.0000000e000,
                1.0000000e001,
                5.0000000e000,
                4.0000000e000,
                3.0000000e000,
                6.0000000e000,
                6.0000000e000,
                9.0000000e000,
                1.5000000e001,
                2.7000000e001,
                3.9000000e001,
                7.7000000e001,
                1.3900000e002,
                2.9000000e002,
                3.7100000e002,
                4.2000000e002,
                3.5000000e002,
                3.7800000e002,
                3.1000000e001,
                5.0000000e000,
            ],
            [
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
                2.0000000e000,
                1.0000000e000,
                0.0000000e000,
                1.0000000e000,
                1.0000000e000,
                0.0000000e000,
                2.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
                1.0000000e000,
                2.0000000e000,
                2.0000000e000,
                1.0000000e000,
                2.0000000e000,
                3.0000000e000,
                0.0000000e000,
                0.0000000e000,
                2.0000000e000,
                3.0000000e000,
                8.0000000e000,
                4.0000000e000,
                3.0000000e000,
                7.0000000e000,
                7.0000000e000,
                1.7000000e001,
                2.5000000e001,
                3.4000000e001,
                6.0000000e001,
                6.3000000e001,
                6.2000000e001,
                5.1000000e001,
                6.1000000e001,
                4.4000000e001,
                1.1300000e002,
                2.0000000e000,
            ],
            [
                0.0000000e000,
                0.0000000e000,
                1.0000000e000,
                1.0000000e000,
                0.0000000e000,
                0.0000000e000,
                0.0000000e000,
                0.0000000e000,
                0.0000000e000,
                0.0000000e000,
                0.0000000e000,
                0.0000000e000,
                0.0000000e000,
                0.0000000e000,
                1.0000000e000,
                0.0000000e000,
                1.0000000e000,
                0.0000000e000,
                0.0000000e000,
                0.0000000e000,
                0.0000000e000,
                0.0000000e000,
                0.0000000e000,
                0.0000000e000,
                1.0000000e000,
                1.0000000e000,
                9.0000000e000,
                1.0000000e000,
                1.0000000e000,
                9.0000000e000,
                1.0000000e000,
                2.5000000e001,
                9.0000000e000,
                9.0000000e000,
                2.5000000e001,
                9.0000000e000,
                4.2000000e001,
                9.0000000e000,
                9.0000000e000,
                9.0000000e000,
                9.0000000e000,
                1.1700000e002,
            ],
        ],
        dtype=np.float64,
    )


@pytest.fixture(scope="session")
def P(count: np.ndarray) -> np.ndarray:
    P, _ = get_known_input(count)
    return P


@pytest.fixture(scope="session")
def sd(count: np.ndarray) -> np.ndarray:
    _, sd = get_known_input(count)
    return sd


@pytest.fixture(scope="session")
def P_mu(P: np.ndarray) -> np.ndarray:
    return np.array(
        [
            0.002187571738471056,
            0.005423036997058904,
            0.023420448543298972,
            0.058285779596956404,
            0.07249429129236545,
            0.05405254454504242,
            0.02835011453613025,
            0.014649669543955093,
            0.009497264358463316,
            0.005873112283083328,
            0.005118819796296368,
            0.004124483560128298,
            0.003571750407440678,
            0.0036920622896702243,
            0.004407199335201476,
            0.0055744492572986064,
            0.007342640276733729,
            0.013993296983640196,
            0.030469625501623968,
            0.05972086356741919,
            0.08726492361388036,
            0.08835621020173381,
            0.06072468748794482,
            0.02987120408479474,
            0.01386374881950853,
            0.00776921665647811,
            0.005324104870327323,
            0.003955418928941411,
            0.004027280031456168,
            0.003218799581014265,
            0.0037458587500202715,
            0.005387966522927479,
            0.005832573831546229,
            0.007677029504691161,
            0.014593685370313842,
            0.027832664619951247,
            0.05385416286668125,
            0.07425019666813401,
            0.0585554468893776,
            0.022629226540465043,
            0.005929305078818829,
            0.003087264670715396,
        ],
        dtype=np.float64,
    )


# this is an irreducible matrix
@pytest.fixture(scope="session")
def P_2():
    return np.array(
        [
            [0.0, 0.8, 0.2, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],
            [0.2, 0.0, 0.6, 0.2, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],
            [0.6, 0.2, 0.0, 0.2, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],
            [0.0, 0.05, 0.05, 0.0, 0.6, 0.3, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 0.0, 0.25, 0.0, 0.25, 0.4, 0.0, 0.0, 0.1, 0.0, 0.0],
            [0.0, 0.0, 0.0, 0.25, 0.25, 0.0, 0.1, 0.0, 0.0, 0.4, 0.0, 0.0],
            [0.0, 0.0, 0.0, 0.0, 0.05, 0.05, 0.0, 0.7, 0.2, 0.0, 0.0, 0.0],
            [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.2, 0.0, 0.8, 0.0, 0.0, 0.0],
            [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.8, 0.2, 0.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 0.0, 0.0, 0.05, 0.05, 0.0, 0.0, 0.0, 0.0, 0.7, 0.2],
            [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.2, 0.0, 0.8],
            [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.8, 0.2, 0.0],
        ],
        dtype=np.float64,
    )


@pytest.fixture(scope="session")
# this is a reducible matrix with one recurrent and one transient class
def P_3():
    return np.array(
        [
            [0.0, 0.8, 0.2, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],
            [0.2, 0.0, 0.6, 0.2, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],
            [0.6, 0.2, 0.0, 0.2, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],
            [0.0, 0.05, 0.05, 0.0, 0.6, 0.3, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 0.0, 0.25, 0.0, 0.25, 0.4, 0.0, 0.0, 0.1, 0.0, 0.0],
            [0.0, 0.0, 0.0, 0.25, 0.25, 0.0, 0.1, 0.0, 0.0, 0.4, 0.0, 0.0],
            [0.0, 0.0, 0.0, 0.0, 0.05, 0.05, 0.0, 0.7, 0.2, 0.0, 0.0, 0.0],
            [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.2, 0.0, 0.8, 0.0, 0.0, 0.0],
            [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.8, 0.2, 0.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.7, 0.3],
            [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.2, 0.0, 0.8],
            [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.8, 0.2, 0.0],
        ],
        dtype=np.float64,
    )


@pytest.fixture(scope="session")
# this is a reducible matrix with two recurrent and one transient class
def P_4():
    return np.array(
        [
            [0.0, 0.8, 0.2, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],
            [0.2, 0.0, 0.6, 0.2, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],
            [0.6, 0.2, 0.0, 0.2, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],
            [0.0, 0.05, 0.05, 0.0, 0.6, 0.3, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 0.0, 0.25, 0.0, 0.25, 0.4, 0.0, 0.0, 0.1, 0.0, 0.0],
            [0.0, 0.0, 0.0, 0.25, 0.25, 0.0, 0.1, 0.0, 0.0, 0.4, 0.0, 0.0],
            [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.7, 0.3, 0.0, 0.0, 0.0],
            [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.2, 0.0, 0.8, 0.0, 0.0, 0.0],
            [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.8, 0.2, 0.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.7, 0.3],
            [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.2, 0.0, 0.8],
            [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.8, 0.2, 0.0],
        ],
        dtype=np.float64,
    )


@pytest.fixture(scope="session")
# this is a reducible matrix with one absorbing state
def P_5():
    return np.array(
        [
            [0.0, 0.8, 0.2, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],
            [0.2, 0.0, 0.6, 0.2, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],
            [0.6, 0.2, 0.0, 0.2, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],
            [0.0, 0.05, 0.05, 0.0, 0.6, 0.3, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 0.0, 0.25, 0.0, 0.25, 0.4, 0.0, 0.0, 0.1, 0.0, 0.0],
            [0.0, 0.0, 0.0, 0.25, 0.25, 0.0, 0.1, 0.0, 0.0, 0.4, 0.0, 0.0],
            [0.0, 0.0, 0.0, 0.0, 0.05, 0.05, 0.0, 0.7, 0.2, 0.0, 0.0, 0.0],
            [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.2, 0.0, 0.8, 0.0, 0.0, 0.0],
            [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.8, 0.2, 0.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 0.0, 0.0, 0.05, 0.05, 0.0, 0.0, 0.0, 0.0, 0.7, 0.2],
            [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.2, 0.0, 0.8],
            [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.0],
        ],
        dtype=np.float64,
    )


@pytest.fixture(scope="session")
# this is a reducible matrix with two absorbing states
def P_6():
    return np.array(
        [
            [0.0, 0.8, 0.2, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],
            [0.2, 0.0, 0.6, 0.2, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],
            [0.6, 0.2, 0.0, 0.2, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],
            [0.0, 0.05, 0.05, 0.0, 0.6, 0.3, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 0.0, 0.25, 0.0, 0.25, 0.4, 0.0, 0.0, 0.1, 0.0, 0.0],
            [0.0, 0.0, 0.0, 0.25, 0.25, 0.0, 0.1, 0.0, 0.0, 0.4, 0.0, 0.0],
            [0.0, 0.0, 0.0, 0.0, 0.05, 0.05, 0.0, 0.7, 0.2, 0.0, 0.0, 0.0],
            [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.2, 0.0, 0.8, 0.0, 0.0, 0.0],
            [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 0.0, 0.0, 0.05, 0.05, 0.0, 0.0, 0.0, 0.0, 0.7, 0.2],
            [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.2, 0.0, 0.8],
            [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.0],
        ],
        dtype=np.float64,
    )


@pytest.fixture(scope="session")
# this is a reducible matrix with 1 absorbing state, one recurrent and one transient class
def P_7():
    return np.array(
        [
            [0.0, 0.8, 0.2, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],
            [0.2, 0.0, 0.6, 0.2, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],
            [0.6, 0.2, 0.0, 0.2, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],
            [0.0, 0.05, 0.05, 0.0, 0.6, 0.3, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 0.0, 0.25, 0.0, 0.25, 0.4, 0.0, 0.0, 0.1, 0.0, 0.0],
            [0.0, 0.0, 0.0, 0.25, 0.25, 0.0, 0.1, 0.0, 0.0, 0.4, 0.0, 0.0],
            [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.7, 0.3, 0.0, 0.0, 0.0],
            [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.2, 0.0, 0.8, 0.0, 0.0, 0.0],
            [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.8, 0.2, 0.0, 0.0, 0.0, 0.0],
            [0.0, 0.0, 0.0, 0.0, 0.05, 0.05, 0.0, 0.0, 0.0, 0.0, 0.7, 0.2],
            [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.2, 0.0, 0.8],
            [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.0],
        ],
        dtype=np.float64,
    )


# "meta"-fixture
@pytest.fixture(params=[f"P_{i}" for i in range(3, 8)])
def P_i(
    request,
    P_2: np.ndarray,
    P_3: np.ndarray,
    P_4: np.ndarray,
    P_5: np.ndarray,
    P_6: np.ndarray,
    P_7: np.ndarray,
) -> np.ndarray:
    return locals()[request.param]


@pytest.fixture(scope="session")
def minChi_P_2_LM() -> np.ndarray:
    return np.array(
        [
            -1.6653345369377348e-16,
            -2.255418698866725e-16,
            -0.9923508699724691,
            -0.9972757370249341,
            -0.926802576904497,
            -0.2705117206956666,
            -0.3360447945215935,
            -0.2973036186306221,
            -0.29104047575515346,
            -0.42902208201892694,
            -3.5809019888001215e-16,
        ],
        dtype=np.float64,
    )


@pytest.fixture(scope="session")
def crispness_values_P_2_LM() -> np.ndarray:
    return np.array(
        [
            0.741514717000221,
            0.815128047451789,
            0.0,
            0.385871540187622,
            0.0,
            0.416280490635965,
            0.417889634858996,
            0.0,
            0.555131505008172,
            0.537583662460791,
            0.999999999999994,
        ],
        dtype=np.float64,
    )


@pytest.fixture(scope="session")
def optimal_crispness_P_2_LM() -> np.float64:
    return 0.8151280474517894


@pytest.fixture(scope="session")
def n_m_P_2_LM() -> np.int64:
    return 3


@pytest.fixture(scope="session")
def top_eigenvalues_P_2_LM() -> np.ndarray:
    return np.array(
        [
            1.0 + 0.0j,
            0.965542929702337 + 0.0j,
            0.884042792993727 + 0.0j,
            -0.482771464851169 + 0.489085740769071j,
            -0.482771464851169 - 0.489085740769071j,
            -0.49366904637718 + 0.473927878052129j,
            -0.49366904637718 - 0.473927878052129j,
            0.588536558884235 + 0.0j,
            -0.431989621331992 + 0.390304675376906j,
            -0.431989621331992 - 0.390304675376906j,
            -0.371262016459619 + 0.0j,
            -0.25 + 0.0j,
        ],
        dtype=np.complex128,
    )


@pytest.fixture(scope="session")
def minChi_P_2_LR() -> np.ndarray:
    return np.array(
        [
            -1.6653345369377348e-16,
            -2.255418698866725e-16,
            -0.050736153733740706,
            -0.09885481857040812,
            -0.10891702506635076,
            -0.22400607297199995,
            -0.04715799116142339,
            -0.8673086537165552,
            -0.992031454915766,
            -0.9881639104412867,
            -4.996003610813204e-16,
        ],
        dtype=np.float64,
    )


@pytest.fixture(scope="session")
def crispness_values_P_2_LR() -> np.ndarray:
    return np.array(
        [
            0.741514717000221,
            0.815128047451789,
            0.81196437708066,
            0.714517489751501,
            0.669180515603114,
            0.0,
            0.834142688699839,
            0.0,
            0.315166752037852,
            0.0,
            0.999999999999994,
        ],
        dtype=np.float64,
    )


@pytest.fixture(scope="session")
def optimal_crispness_P_2_LR() -> np.float64:
    return 0.8341426886998387


@pytest.fixture(scope="session")
def n_m_P_2_LR() -> np.int64:
    return 8


@pytest.fixture(scope="session")
def top_eigenvalues_P_2_LR() -> np.ndarray:
    return np.array(
        [
            1.0 + 0.0j,
            0.965542929702337 + 0.0j,
            0.884042792993727 + 0.0j,
            0.588536558884235 + 0.0j,
            -0.25 + 0.0j,
            -0.371262016459619 + 0.0j,
            -0.431989621331992 + 0.390304675376906j,
            -0.431989621331992 - 0.390304675376906j,
            -0.482771464851169 + 0.489085740769071j,
            -0.482771464851169 - 0.489085740769071j,
            -0.49366904637718 + 0.473927878052129j,
            -0.49366904637718 - 0.473927878052129j,
        ],
        dtype=np.complex128,
    )


@pytest.fixture(scope="session")
def M_1() -> np.ndarray:
    return np.array(
        [
            [2, -87, -20000, 10000],
            [5, 2, -20000, -10000],
            [0, 0, 1, -11],
            [0, 0, 37, 1],
        ],
        dtype=np.float64,
    )


@pytest.fixture(scope="session")
def M_2() -> np.ndarray:
    return np.array(
        [
            [1, -3, 3576, 4888],
            [1, 1, -88, -1440],
            [0, 0, 1.001, -3],
            [0, 0, 1.001, 1.001],
        ],
        dtype=np.float64,
    )


@pytest.fixture(scope="session")
def M_3() -> np.ndarray:
    return np.array(
        [
            [1, -100, 400, -1000],
            [0.01, 1, 1200, -10],
            [0, 0, 1.001, -0.01],
            [0, 0, 100, 1.001],
        ],
        dtype=np.float64,
    )


@pytest.fixture(scope="session")
def M_4() -> np.ndarray:
    return np.array(
        [
            [1, -1e4, 8812, 4566],
            [1e-4, 1, -9, 1200],
            [0, 0, 1e-5 + 1, -1e-4],
            [0, 0, 1e4, 1e-5 + 1],
        ],
        dtype=np.float64,
    )


@pytest.fixture(scope="session")
def M_5() -> np.ndarray:
    return np.array(
        [
            [1, -11, -20000, 10000],
            [37, 1, -20000, -10000],
            [0, 0, 2, -87],
            [0, 0, 5, 2],
        ],
        dtype=np.float64,
    )


# "meta"-fixture
@pytest.fixture(params=[f"M_{i}" for i in range(1, 6)])
def R_i(
    request,
    M_1: np.ndarray,
    M_2: np.ndarray,
    M_3: np.ndarray,
    M_4: np.ndarray,
    M_5: np.ndarray,
) -> np.ndarray:
    return locals()[request.param]
