"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.formatAsPythonModule = exports.sorted = exports.dedupArray = exports.deepMerge = exports.isObject = exports.isTruthy = exports.decamelizeKeysRecursively = exports.writeFile = exports.getFilePermissions = exports.execOrUndefined = exports.execCapture = exports.exec = void 0;
const child_process = require("child_process");
const path = require("path");
const fs = require("fs-extra");
const logging = require("./logging");
// eslint-disable-next-line @typescript-eslint/no-require-imports
const decamelize = require('decamelize');
function exec(command, options) {
    logging.verbose(command);
    return child_process.execSync(command, {
        stdio: ['inherit', process.stderr, 'pipe'],
        ...options,
    });
}
exports.exec = exec;
/**
 * Executes command and returns STDOUT. If the command fails (non-zero), throws an error.
 */
function execCapture(command, options) {
    logging.verbose(command);
    return child_process.execSync(command, {
        stdio: ['inherit', 'pipe', 'inherit'],
        ...options,
    });
}
exports.execCapture = execCapture;
/**
 * Executes `command` and returns its value or undefined if the command failed.
 */
function execOrUndefined(command, options) {
    try {
        const value = child_process.execSync(command, { stdio: ['inherit', 'pipe', 'ignore'], ...options }).toString('utf-8').trim();
        if (!value) {
            return undefined;
        } // an empty string is the same as undefined
        return value;
    }
    catch (_a) {
        return undefined;
    }
}
exports.execOrUndefined = execOrUndefined;
function getFilePermissions(options) {
    var _a, _b;
    const readonly = (_a = options.readonly) !== null && _a !== void 0 ? _a : false;
    const executable = (_b = options.executable) !== null && _b !== void 0 ? _b : false;
    if (readonly && executable) {
        return '500';
    }
    else if (readonly) {
        return '400';
    }
    else if (executable) {
        return '755';
    }
    else {
        return '644';
    }
}
exports.getFilePermissions = getFilePermissions;
function writeFile(filePath, data, options = {}) {
    if (fs.existsSync(filePath)) {
        fs.chmodSync(filePath, '600');
    }
    fs.mkdirpSync(path.dirname(filePath));
    fs.writeFileSync(filePath, data);
    fs.chmodSync(filePath, getFilePermissions(options));
}
exports.writeFile = writeFile;
function decamelizeKeysRecursively(input, opt) {
    var _a, _b, _c, _d;
    const shouldAlwaysDecamelize = () => true;
    const shouldDecamelize = (_a = opt === null || opt === void 0 ? void 0 : opt.shouldDecamelize) !== null && _a !== void 0 ? _a : shouldAlwaysDecamelize;
    const separator = (_b = opt === null || opt === void 0 ? void 0 : opt.separator) !== null && _b !== void 0 ? _b : '_';
    const path_ = (_c = opt === null || opt === void 0 ? void 0 : opt.path) !== null && _c !== void 0 ? _c : [];
    const maxDepth = (_d = opt === null || opt === void 0 ? void 0 : opt.maxDepth) !== null && _d !== void 0 ? _d : 10;
    if (path_.length > maxDepth) {
        throw new Error('Decamelled too deeply - check that the input has no circular references');
    }
    if (Array.isArray(input)) {
        return input.map((k, i) => decamelizeKeysRecursively(k, {
            ...opt,
            path: [...path_, i.toString()],
        }));
    }
    if (typeof input === 'object' && input !== null) {
        const mappedObject = {};
        for (const [key, value] of Object.entries(input)) {
            const transformedKey = shouldDecamelize([...path_, key], value)
                ? decamelize(key, separator)
                : key;
            mappedObject[transformedKey] = decamelizeKeysRecursively(value, {
                ...opt,
                path: [...path_, key],
            });
        }
        return mappedObject;
    }
    return input;
}
exports.decamelizeKeysRecursively = decamelizeKeysRecursively;
/**
 * Returns false if value is unset or a falsey value, and true otherwise.
 * @param value an environment variable
 */
function isTruthy(value) {
    return !(value === undefined || ['null', 'undefined', '0', 'false', ''].includes(value.toLocaleLowerCase()));
}
exports.isTruthy = isTruthy;
/**
 * Return whether the given value is an object
 *
 * Even though arrays and instances of classes technically are objects, we
 * usually want to treat them differently, so we return false in those cases.
 */
function isObject(x) {
    return x !== null && typeof x === 'object' && !Array.isArray(x)
        && x.constructor.name === 'Object';
}
exports.isObject = isObject;
/**
 * Recursively merge objects together
 *
 * The leftmost object is mutated and returned. Arrays are not merged
 * but overwritten just like scalars.
 *
 * If an object is merged into a non-object, the non-object is lost.
 *
 * `undefined`s will cause a value to be deleted if destructive is enabled.
 */
function deepMerge(objects, destructive = false) {
    function mergeOne(target, source) {
        for (const key of Object.keys(source)) {
            const value = source[key];
            if (isObject(value)) {
                // if the value at the target is not an object, override it with an
                // object so we can continue the recursion
                if (!isObject(target[key])) {
                    target[key] = value;
                }
                mergeOne(target[key], value);
                // if the result of the merge is an empty object, it's because the
                // eventual value we assigned is `undefined`, and there are no
                // sibling concrete values alongside, so we can delete this tree.
                const output = target[key];
                if (isObject(output) && Object.keys(output).length === 0 && destructive) {
                    delete target[key];
                }
            }
            else if (value === undefined && destructive) {
                delete target[key];
            }
            else if (typeof value !== 'undefined') {
                target[key] = value;
            }
        }
    }
    const others = objects.filter(x => x != null);
    if (others.length === 0) {
        return {};
    }
    const into = others.splice(0, 1)[0];
    others.forEach(other => mergeOne(into, other));
    return into;
}
exports.deepMerge = deepMerge;
/*
 * Deduplicate values in a list, returning a new array.
 * @param array list of values
 */
function dedupArray(array) {
    return array.filter((val, idx) => array.indexOf(val) === idx);
}
exports.dedupArray = dedupArray;
/**
 * Returns a sorted version of `x` or `undefined` if it is an empty array or object.
 */
function sorted(x) {
    if (x == null) {
        return undefined;
    }
    if (Array.isArray(x)) {
        if (x.length === 0) {
            return undefined;
        }
        return x.sort();
    }
    else if (typeof x === 'object') {
        if (Object.keys(x).length === 0) {
            return undefined;
        }
        const result = {};
        for (const [key, value] of Object.entries(x).sort(([l], [r]) => l.localeCompare(r))) {
            result[key] = value;
        }
        return result;
    }
    else {
        return x;
    }
}
exports.sorted = sorted;
function formatAsPythonModule(name) {
    return name.replace(/-/g, '_').replace(/\./g, '_');
}
exports.formatAsPythonModule = formatAsPythonModule;
//# sourceMappingURL=data:application/json;base64,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