"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Release = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const component_1 = require("../component");
const github_1 = require("../github");
const publisher_1 = require("../publisher");
const version_1 = require("../version");
const BUILD_JOBID = 'release';
/**
 * (experimental) Manages releases (currently through GitHub workflows).
 *
 * By default, no branches are released. To add branches, call `addBranch()`.
 *
 * @experimental
 */
class Release extends component_1.Component {
    /**
     * @experimental
     */
    constructor(project, options) {
        var _b, _c, _d, _e, _f, _g, _h;
        super(project);
        this.branches = new Array();
        this.jobs = {};
        if (Array.isArray(options.releaseBranches)) {
            throw new Error('"releaseBranches" is no longer an array. See type annotations');
        }
        this.task = options.task;
        this.preBuildSteps = (_b = options.releaseWorkflowSetupSteps) !== null && _b !== void 0 ? _b : [];
        this.postBuildSteps = (_c = options.postBuildSteps) !== null && _c !== void 0 ? _c : [];
        this.antitamper = (_d = options.antitamper) !== null && _d !== void 0 ? _d : true;
        this.artifactDirectory = (_e = options.artifactsDirectory) !== null && _e !== void 0 ? _e : 'dist';
        this.versionFile = options.versionFile;
        this.releaseSchedule = options.releaseSchedule;
        this.releaseEveryCommit = (_f = options.releaseEveryCommit) !== null && _f !== void 0 ? _f : true;
        this.containerImage = options.workflowContainerImage;
        this.version = new version_1.Version(project, {
            versionFile: options.versionFile,
        });
        this.publisher = new publisher_1.Publisher(project, {
            artifactName: this.artifactDirectory,
            buildJobId: BUILD_JOBID,
            jsiiReleaseVersion: options.jsiiReleaseVersion,
        });
        // add the default branch
        this.defaultBranch = {
            name: options.branch,
            prerelease: options.prerelease,
            majorVersion: options.majorVersion,
            workflowName: (_g = options.releaseWorkflowName) !== null && _g !== void 0 ? _g : 'Release',
        };
        this.branches.push(this.defaultBranch);
        for (const [name, opts] of Object.entries((_h = options.releaseBranches) !== null && _h !== void 0 ? _h : {})) {
            this.addBranch(name, opts);
        }
    }
    /**
     * (experimental) Adds a release branch.
     *
     * It is a git branch from which releases are published. If a project has more than one release
     * branch, we require that `majorVersion` is also specified for the primary branch in order to
     * ensure branches always release the correct version.
     *
     * @param branch The branch to monitor (e.g. `main`, `v2.x`).
     * @param options Branch definition.
     * @experimental
     */
    addBranch(branch, options) {
        if (this.branches.find(b => b.name === branch)) {
            throw new Error(`The release branch ${branch} is already defined`);
        }
        // if we add a branch, we require that the default branch will also define a
        // major version.
        if (this.defaultBranch.majorVersion === undefined) {
            throw new Error('you must specify "majorVersion" for the default branch when adding multiple release branches');
        }
        this.branches.push({
            name: branch,
            ...options,
        });
    }
    /**
     * (experimental) Adds jobs to all release workflows.
     *
     * @param jobs The jobs to add (name => job).
     * @experimental
     */
    addJobs(jobs) {
        for (const [name, job] of Object.entries(jobs)) {
            this.jobs[name] = job;
        }
    }
    // render a workflow per branch and all the jobs to it
    /**
     * (experimental) Called before synthesis.
     *
     * @experimental
     */
    preSynthesize() {
        for (const branch of this.branches) {
            const workflow = this.createWorkflow(branch);
            workflow.addJobs(this.publisher.render());
            workflow.addJobs(this.jobs);
        }
    }
    createWorkflow(branch) {
        var _b, _c;
        const github = this.project.github;
        if (!github) {
            throw new Error('no github support');
        }
        const workflowName = (_b = branch.workflowName) !== null && _b !== void 0 ? _b : `release-${branch.name}`;
        // to avoid race conditions between two commits trying to release the same
        // version, we check if the head sha is identical to the remote sha. if
        // not, we will skip the release and just finish the build.
        const gitRemoteStep = 'git_remote';
        const latestCommitOutput = 'latest_commit';
        const noNewCommits = `\${{ steps.${gitRemoteStep}.outputs.${latestCommitOutput} == github.sha }}`;
        const steps = new Array();
        // check out sources.
        steps.push({
            name: 'Checkout',
            uses: 'actions/checkout@v2',
            with: {
                // we must use 'fetch-depth=0' in order to fetch all tags
                // otherwise tags are not checked out
                'fetch-depth': 0,
            },
        });
        // sets git identity so we can push later
        steps.push({
            name: 'Set git identity',
            run: [
                'git config user.name "Automation"',
                'git config user.email "github-actions@github.com"',
            ].join('\n'),
        });
        steps.push(...(_c = this.preBuildSteps) !== null && _c !== void 0 ? _c : []);
        const env = {};
        if (branch.majorVersion !== undefined) {
            env.MAJOR = branch.majorVersion.toString();
        }
        if (branch.prerelease) {
            env.PRERELEASE = branch.prerelease;
        }
        steps.push({
            name: 'Bump to next version',
            run: this.project.runTaskCommand(this.version.bumpTask),
            env: Object.keys(env).length ? env : undefined,
        });
        // run the main build task
        steps.push({
            name: this.task.name,
            run: this.project.runTaskCommand(this.task),
        });
        // run post-build steps
        steps.push(...this.postBuildSteps);
        // create a backup of the version JSON file (e.g. package.json) because we
        // are going to revert the bump and we need the version number in order to
        // create the github release.
        const versionJsonBackup = `${this.versionFile}.bak.json`;
        steps.push({
            name: 'Backup version file',
            run: `cp -f ${this.versionFile} ${versionJsonBackup}`,
        });
        // revert the bump so anti-tamper will not fail
        steps.push({
            name: 'Unbump',
            run: this.project.runTaskCommand(this.version.unbumpTask),
        });
        // anti-tamper check (fails if there were changes to committed files)
        // this will identify any non-committed files generated during build (e.g. test snapshots)
        if (this.antitamper) {
            steps.push({
                name: 'Anti-tamper check',
                run: 'git diff --ignore-space-at-eol --exit-code',
            });
        }
        // check if new commits were pushed to the repo while we were building.
        // if new commits have been pushed, we will cancel this release
        steps.push({
            name: 'Check for new commits',
            id: gitRemoteStep,
            run: `echo ::set-output name=${latestCommitOutput}::"$(git ls-remote origin -h \${{ github.ref }} | cut -f1)"`,
        });
        // create a github release
        const getVersion = `v$(node -p \"require(\'./${versionJsonBackup}\').version\")`;
        steps.push({
            name: 'Create release',
            if: noNewCommits,
            run: [
                `gh release create ${getVersion}`,
                `-F ${this.version.changelogFile}`,
                `-t ${getVersion}`,
            ].join(' '),
            env: {
                GITHUB_TOKEN: '${{ secrets.GITHUB_TOKEN }}',
            },
        });
        steps.push({
            name: 'Upload artifact',
            if: noNewCommits,
            uses: 'actions/upload-artifact@v2.1.1',
            with: {
                name: this.artifactDirectory,
                path: this.artifactDirectory,
            },
        });
        const workflow = github.addWorkflow(workflowName);
        // determine release triggers
        workflow.on({
            schedule: this.releaseSchedule ? [{ cron: this.releaseSchedule }] : undefined,
            push: (this.releaseEveryCommit) ? { branches: [branch.name] } : undefined,
            workflowDispatch: {},
        });
        // add main build job
        workflow.addJobs({
            [BUILD_JOBID]: {
                runsOn: 'ubuntu-latest',
                container: this.containerImage ? { image: this.containerImage } : undefined,
                env: {
                    CI: 'true',
                    RELEASE: 'true',
                },
                permissions: {
                    contents: github_1.workflows.JobPermission.WRITE,
                },
                steps: steps,
            },
        });
        return workflow;
    }
}
exports.Release = Release;
_a = JSII_RTTI_SYMBOL_1;
Release[_a] = { fqn: "projen.release.Release", version: "0.23.2" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicmVsZWFzZS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uL3NyYy9yZWxlYXNlL3JlbGVhc2UudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSw0Q0FBeUM7QUFDekMsc0NBQXNEO0FBRXRELDRDQUF5QztBQUV6Qyx3Q0FBcUM7QUFFckMsTUFBTSxXQUFXLEdBQUcsU0FBUyxDQUFDOzs7Ozs7OztBQXNEOUIsTUFBYSxPQUFRLFNBQVEscUJBQVM7Ozs7SUFrQnBDLFlBQVksT0FBZ0IsRUFBRSxPQUF1Qjs7UUFDbkQsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBTEEsYUFBUSxHQUFHLElBQUksS0FBSyxFQUFpQixDQUFDO1FBQ3RDLFNBQUksR0FBa0MsRUFBRSxDQUFDO1FBTXhELElBQUksS0FBSyxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsZUFBZSxDQUFDLEVBQUU7WUFDMUMsTUFBTSxJQUFJLEtBQUssQ0FBQywrREFBK0QsQ0FBQyxDQUFDO1NBQ2xGO1FBRUQsSUFBSSxDQUFDLElBQUksR0FBRyxPQUFPLENBQUMsSUFBSSxDQUFDO1FBQ3pCLElBQUksQ0FBQyxhQUFhLFNBQUcsT0FBTyxDQUFDLHlCQUF5QixtQ0FBSSxFQUFFLENBQUM7UUFDN0QsSUFBSSxDQUFDLGNBQWMsU0FBRyxPQUFPLENBQUMsY0FBYyxtQ0FBSSxFQUFFLENBQUM7UUFDbkQsSUFBSSxDQUFDLFVBQVUsU0FBRyxPQUFPLENBQUMsVUFBVSxtQ0FBSSxJQUFJLENBQUM7UUFDN0MsSUFBSSxDQUFDLGlCQUFpQixTQUFHLE9BQU8sQ0FBQyxrQkFBa0IsbUNBQUksTUFBTSxDQUFDO1FBQzlELElBQUksQ0FBQyxXQUFXLEdBQUcsT0FBTyxDQUFDLFdBQVcsQ0FBQztRQUN2QyxJQUFJLENBQUMsZUFBZSxHQUFHLE9BQU8sQ0FBQyxlQUFlLENBQUM7UUFDL0MsSUFBSSxDQUFDLGtCQUFrQixTQUFHLE9BQU8sQ0FBQyxrQkFBa0IsbUNBQUksSUFBSSxDQUFDO1FBQzdELElBQUksQ0FBQyxjQUFjLEdBQUcsT0FBTyxDQUFDLHNCQUFzQixDQUFDO1FBRXJELElBQUksQ0FBQyxPQUFPLEdBQUcsSUFBSSxpQkFBTyxDQUFDLE9BQU8sRUFBRTtZQUNsQyxXQUFXLEVBQUUsT0FBTyxDQUFDLFdBQVc7U0FDakMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLFNBQVMsR0FBRyxJQUFJLHFCQUFTLENBQUMsT0FBTyxFQUFFO1lBQ3RDLFlBQVksRUFBRSxJQUFJLENBQUMsaUJBQWlCO1lBQ3BDLFVBQVUsRUFBRSxXQUFXO1lBQ3ZCLGtCQUFrQixFQUFFLE9BQU8sQ0FBQyxrQkFBa0I7U0FDL0MsQ0FBQyxDQUFDO1FBRUgseUJBQXlCO1FBQ3pCLElBQUksQ0FBQyxhQUFhLEdBQUc7WUFDbkIsSUFBSSxFQUFFLE9BQU8sQ0FBQyxNQUFNO1lBQ3BCLFVBQVUsRUFBRSxPQUFPLENBQUMsVUFBVTtZQUM5QixZQUFZLEVBQUUsT0FBTyxDQUFDLFlBQVk7WUFDbEMsWUFBWSxRQUFFLE9BQU8sQ0FBQyxtQkFBbUIsbUNBQUksU0FBUztTQUN2RCxDQUFDO1FBRUYsSUFBSSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxDQUFDO1FBRXZDLEtBQUssTUFBTSxDQUFDLElBQUksRUFBRSxJQUFJLENBQUMsSUFBSSxNQUFNLENBQUMsT0FBTyxPQUFDLE9BQU8sQ0FBQyxlQUFlLG1DQUFJLEVBQUUsQ0FBQyxFQUFFO1lBQ3hFLElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxFQUFFLElBQUksQ0FBQyxDQUFDO1NBQzVCO0lBQ0gsQ0FBQzs7Ozs7Ozs7Ozs7O0lBR00sU0FBUyxDQUFDLE1BQWMsRUFBRSxPQUFzQjtRQUNyRCxJQUFJLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLElBQUksS0FBSyxNQUFNLENBQUMsRUFBRTtZQUM5QyxNQUFNLElBQUksS0FBSyxDQUFDLHNCQUFzQixNQUFNLHFCQUFxQixDQUFDLENBQUM7U0FDcEU7UUFFRCw0RUFBNEU7UUFDNUUsaUJBQWlCO1FBQ2pCLElBQUksSUFBSSxDQUFDLGFBQWEsQ0FBQyxZQUFZLEtBQUssU0FBUyxFQUFFO1lBQ2pELE1BQU0sSUFBSSxLQUFLLENBQUMsOEZBQThGLENBQUMsQ0FBQztTQUNqSDtRQUVELElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDO1lBQ2pCLElBQUksRUFBRSxNQUFNO1lBQ1osR0FBRyxPQUFPO1NBQ1gsQ0FBQyxDQUFDO0lBQ0wsQ0FBQzs7Ozs7OztJQUdNLE9BQU8sQ0FBQyxJQUFtQztRQUNoRCxLQUFLLE1BQU0sQ0FBQyxJQUFJLEVBQUUsR0FBRyxDQUFDLElBQUksTUFBTSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsRUFBRTtZQUM5QyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxHQUFHLEdBQUcsQ0FBQztTQUN2QjtJQUNILENBQUM7SUFFRCxzREFBc0Q7Ozs7OztJQUMvQyxhQUFhO1FBQ2xCLEtBQUssTUFBTSxNQUFNLElBQUksSUFBSSxDQUFDLFFBQVEsRUFBRTtZQUNsQyxNQUFNLFFBQVEsR0FBRyxJQUFJLENBQUMsY0FBYyxDQUFDLE1BQU0sQ0FBQyxDQUFDO1lBQzdDLFFBQVEsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDO1lBQzFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1NBQzdCO0lBQ0gsQ0FBQztJQUVPLGNBQWMsQ0FBQyxNQUFxQjs7UUFDMUMsTUFBTSxNQUFNLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUM7UUFDbkMsSUFBSSxDQUFDLE1BQU0sRUFBRTtZQUFFLE1BQU0sSUFBSSxLQUFLLENBQUMsbUJBQW1CLENBQUMsQ0FBQztTQUFFO1FBRXRELE1BQU0sWUFBWSxTQUFHLE1BQU0sQ0FBQyxZQUFZLG1DQUFJLFdBQVcsTUFBTSxDQUFDLElBQUksRUFBRSxDQUFDO1FBRXJFLDBFQUEwRTtRQUMxRSx1RUFBdUU7UUFDdkUsMkRBQTJEO1FBQzNELE1BQU0sYUFBYSxHQUFHLFlBQVksQ0FBQztRQUNuQyxNQUFNLGtCQUFrQixHQUFHLGVBQWUsQ0FBQztRQUMzQyxNQUFNLFlBQVksR0FBRyxjQUFjLGFBQWEsWUFBWSxrQkFBa0IsbUJBQW1CLENBQUM7UUFFbEcsTUFBTSxLQUFLLEdBQUcsSUFBSSxLQUFLLEVBQXFCLENBQUM7UUFFN0MscUJBQXFCO1FBQ3JCLEtBQUssQ0FBQyxJQUFJLENBQUM7WUFDVCxJQUFJLEVBQUUsVUFBVTtZQUNoQixJQUFJLEVBQUUscUJBQXFCO1lBQzNCLElBQUksRUFBRTtnQkFDSix5REFBeUQ7Z0JBQ3pELHFDQUFxQztnQkFDckMsYUFBYSxFQUFFLENBQUM7YUFDakI7U0FDRixDQUFDLENBQUM7UUFFSCx5Q0FBeUM7UUFDekMsS0FBSyxDQUFDLElBQUksQ0FBQztZQUNULElBQUksRUFBRSxrQkFBa0I7WUFDeEIsR0FBRyxFQUFFO2dCQUNILG1DQUFtQztnQkFDbkMsbURBQW1EO2FBQ3BELENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQztTQUNiLENBQUMsQ0FBQztRQUVILEtBQUssQ0FBQyxJQUFJLENBQUMsU0FBRyxJQUFJLENBQUMsYUFBYSxtQ0FBSSxFQUFFLENBQUMsQ0FBQztRQUV4QyxNQUFNLEdBQUcsR0FBMkIsRUFBRSxDQUFDO1FBRXZDLElBQUksTUFBTSxDQUFDLFlBQVksS0FBSyxTQUFTLEVBQUU7WUFDckMsR0FBRyxDQUFDLEtBQUssR0FBRyxNQUFNLENBQUMsWUFBWSxDQUFDLFFBQVEsRUFBRSxDQUFDO1NBQzVDO1FBRUQsSUFBSSxNQUFNLENBQUMsVUFBVSxFQUFFO1lBQ3JCLEdBQUcsQ0FBQyxVQUFVLEdBQUcsTUFBTSxDQUFDLFVBQVUsQ0FBQztTQUNwQztRQUVELEtBQUssQ0FBQyxJQUFJLENBQUM7WUFDVCxJQUFJLEVBQUUsc0JBQXNCO1lBQzVCLEdBQUcsRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLGNBQWMsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQztZQUN2RCxHQUFHLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsU0FBUztTQUMvQyxDQUFDLENBQUM7UUFFSCwwQkFBMEI7UUFDMUIsS0FBSyxDQUFDLElBQUksQ0FBQztZQUNULElBQUksRUFBRSxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUk7WUFDcEIsR0FBRyxFQUFFLElBQUksQ0FBQyxPQUFPLENBQUMsY0FBYyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUM7U0FDNUMsQ0FBQyxDQUFDO1FBRUgsdUJBQXVCO1FBQ3ZCLEtBQUssQ0FBQyxJQUFJLENBQUMsR0FBRyxJQUFJLENBQUMsY0FBYyxDQUFDLENBQUM7UUFFbkMsMEVBQTBFO1FBQzFFLDBFQUEwRTtRQUMxRSw2QkFBNkI7UUFDN0IsTUFBTSxpQkFBaUIsR0FBRyxHQUFHLElBQUksQ0FBQyxXQUFXLFdBQVcsQ0FBQztRQUN6RCxLQUFLLENBQUMsSUFBSSxDQUFDO1lBQ1QsSUFBSSxFQUFFLHFCQUFxQjtZQUMzQixHQUFHLEVBQUUsU0FBUyxJQUFJLENBQUMsV0FBVyxJQUFJLGlCQUFpQixFQUFFO1NBQ3RELENBQUMsQ0FBQztRQUVILCtDQUErQztRQUMvQyxLQUFLLENBQUMsSUFBSSxDQUFDO1lBQ1QsSUFBSSxFQUFFLFFBQVE7WUFDZCxHQUFHLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxjQUFjLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxVQUFVLENBQUM7U0FDMUQsQ0FBQyxDQUFDO1FBRUgscUVBQXFFO1FBQ3JFLDBGQUEwRjtRQUMxRixJQUFJLElBQUksQ0FBQyxVQUFVLEVBQUU7WUFDbkIsS0FBSyxDQUFDLElBQUksQ0FBQztnQkFDVCxJQUFJLEVBQUUsbUJBQW1CO2dCQUN6QixHQUFHLEVBQUUsNENBQTRDO2FBQ2xELENBQUMsQ0FBQztTQUNKO1FBRUQsdUVBQXVFO1FBQ3ZFLCtEQUErRDtRQUMvRCxLQUFLLENBQUMsSUFBSSxDQUFDO1lBQ1QsSUFBSSxFQUFFLHVCQUF1QjtZQUM3QixFQUFFLEVBQUUsYUFBYTtZQUNqQixHQUFHLEVBQUUsMEJBQTBCLGtCQUFrQiw2REFBNkQ7U0FDL0csQ0FBQyxDQUFDO1FBRUgsMEJBQTBCO1FBQzFCLE1BQU0sVUFBVSxHQUFHLDRCQUE0QixpQkFBaUIsZ0JBQWdCLENBQUM7UUFDakYsS0FBSyxDQUFDLElBQUksQ0FBQztZQUNULElBQUksRUFBRSxnQkFBZ0I7WUFDdEIsRUFBRSxFQUFFLFlBQVk7WUFDaEIsR0FBRyxFQUFFO2dCQUNILHFCQUFxQixVQUFVLEVBQUU7Z0JBQ2pDLE1BQU0sSUFBSSxDQUFDLE9BQU8sQ0FBQyxhQUFhLEVBQUU7Z0JBQ2xDLE1BQU0sVUFBVSxFQUFFO2FBQ25CLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQztZQUNYLEdBQUcsRUFBRTtnQkFDSCxZQUFZLEVBQUUsNkJBQTZCO2FBQzVDO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsS0FBSyxDQUFDLElBQUksQ0FBQztZQUNULElBQUksRUFBRSxpQkFBaUI7WUFDdkIsRUFBRSxFQUFFLFlBQVk7WUFDaEIsSUFBSSxFQUFFLGdDQUFnQztZQUN0QyxJQUFJLEVBQUU7Z0JBQ0osSUFBSSxFQUFFLElBQUksQ0FBQyxpQkFBaUI7Z0JBQzVCLElBQUksRUFBRSxJQUFJLENBQUMsaUJBQWlCO2FBQzdCO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsTUFBTSxRQUFRLEdBQUcsTUFBTSxDQUFDLFdBQVcsQ0FBQyxZQUFZLENBQUMsQ0FBQztRQUVsRCw2QkFBNkI7UUFDN0IsUUFBUSxDQUFDLEVBQUUsQ0FBQztZQUNWLFFBQVEsRUFBRSxJQUFJLENBQUMsZUFBZSxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUUsSUFBSSxFQUFFLElBQUksQ0FBQyxlQUFlLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxTQUFTO1lBQzdFLElBQUksRUFBRSxDQUFDLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFLFFBQVEsRUFBRSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxTQUFTO1lBQ3pFLGdCQUFnQixFQUFFLEVBQUU7U0FDckIsQ0FBQyxDQUFDO1FBRUgscUJBQXFCO1FBQ3JCLFFBQVEsQ0FBQyxPQUFPLENBQUM7WUFDZixDQUFDLFdBQVcsQ0FBQyxFQUFFO2dCQUNiLE1BQU0sRUFBRSxlQUFlO2dCQUN2QixTQUFTLEVBQUUsSUFBSSxDQUFDLGNBQWMsQ0FBQyxDQUFDLENBQUMsRUFBRSxLQUFLLEVBQUUsSUFBSSxDQUFDLGNBQWMsRUFBRSxDQUFDLENBQUMsQ0FBQyxTQUFTO2dCQUMzRSxHQUFHLEVBQUU7b0JBQ0gsRUFBRSxFQUFFLE1BQU07b0JBQ1YsT0FBTyxFQUFFLE1BQU07aUJBQ2hCO2dCQUNELFdBQVcsRUFBRTtvQkFDWCxRQUFRLEVBQUUsa0JBQVMsQ0FBQyxhQUFhLENBQUMsS0FBSztpQkFDeEM7Z0JBQ0QsS0FBSyxFQUFFLEtBQUs7YUFDYjtTQUNGLENBQUMsQ0FBQztRQUVILE9BQU8sUUFBUSxDQUFDO0lBQ2xCLENBQUM7O0FBL09ILDBCQWdQQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IENvbXBvbmVudCB9IGZyb20gJy4uL2NvbXBvbmVudCc7XG5pbXBvcnQgeyB3b3JrZmxvd3MsIEdpdGh1YldvcmtmbG93IH0gZnJvbSAnLi4vZ2l0aHViJztcbmltcG9ydCB7IFByb2plY3QgfSBmcm9tICcuLi9wcm9qZWN0JztcbmltcG9ydCB7IFB1Ymxpc2hlciB9IGZyb20gJy4uL3B1Ymxpc2hlcic7XG5pbXBvcnQgeyBUYXNrIH0gZnJvbSAnLi4vdGFza3MnO1xuaW1wb3J0IHsgVmVyc2lvbiB9IGZyb20gJy4uL3ZlcnNpb24nO1xuXG5jb25zdCBCVUlMRF9KT0JJRCA9ICdyZWxlYXNlJztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIFJlbGVhc2VQcm9qZWN0T3B0aW9ucyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHJlbGVhc2VFdmVyeUNvbW1pdD86IGJvb2xlYW47XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSByZWxlYXNlU2NoZWR1bGU/OiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGFydGlmYWN0c0RpcmVjdG9yeT86IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgcmVsZWFzZVdvcmtmbG93U2V0dXBTdGVwcz86IHdvcmtmbG93cy5Kb2JTdGVwW107XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB3b3JrZmxvd0NvbnRhaW5lckltYWdlPzogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGpzaWlSZWxlYXNlVmVyc2lvbj86IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgcG9zdEJ1aWxkU3RlcHM/OiB3b3JrZmxvd3MuSm9iU3RlcFtdO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBhbnRpdGFtcGVyPzogYm9vbGVhbjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgbWFqb3JWZXJzaW9uPzogbnVtYmVyO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHByZXJlbGVhc2U/OiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSByZWxlYXNlV29ya2Zsb3dOYW1lPzogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSByZWxlYXNlQnJhbmNoZXM/OiB7IFtuYW1lOiBzdHJpbmddOiBCcmFuY2hPcHRpb25zIH07XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBSZWxlYXNlT3B0aW9ucyBleHRlbmRzIFJlbGVhc2VQcm9qZWN0T3B0aW9ucyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB0YXNrOiBUYXNrO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgdmVyc2lvbkZpbGU6IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGJyYW5jaDogc3RyaW5nO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGNsYXNzIFJlbGVhc2UgZXh0ZW5kcyBDb21wb25lbnQge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IHB1Ymxpc2hlcjogUHVibGlzaGVyO1xuXG4gIHByaXZhdGUgcmVhZG9ubHkgdGFzazogVGFzaztcbiAgcHJpdmF0ZSByZWFkb25seSB2ZXJzaW9uOiBWZXJzaW9uO1xuICBwcml2YXRlIHJlYWRvbmx5IHBvc3RCdWlsZFN0ZXBzOiB3b3JrZmxvd3MuSm9iU3RlcFtdO1xuICBwcml2YXRlIHJlYWRvbmx5IGFudGl0YW1wZXI6IGJvb2xlYW47XG4gIHByaXZhdGUgcmVhZG9ubHkgYXJ0aWZhY3REaXJlY3Rvcnk6IHN0cmluZztcbiAgcHJpdmF0ZSByZWFkb25seSB2ZXJzaW9uRmlsZTogc3RyaW5nO1xuICBwcml2YXRlIHJlYWRvbmx5IHJlbGVhc2VTY2hlZHVsZT86IHN0cmluZztcbiAgcHJpdmF0ZSByZWFkb25seSByZWxlYXNlRXZlcnlDb21taXQ6IGJvb2xlYW47XG4gIHByaXZhdGUgcmVhZG9ubHkgcHJlQnVpbGRTdGVwczogd29ya2Zsb3dzLkpvYlN0ZXBbXTtcbiAgcHJpdmF0ZSByZWFkb25seSBjb250YWluZXJJbWFnZT86IHN0cmluZztcbiAgcHJpdmF0ZSByZWFkb25seSBicmFuY2hlcyA9IG5ldyBBcnJheTxSZWxlYXNlQnJhbmNoPigpO1xuICBwcml2YXRlIHJlYWRvbmx5IGpvYnM6IFJlY29yZDxzdHJpbmcsIHdvcmtmbG93cy5Kb2I+ID0ge307XG4gIHByaXZhdGUgcmVhZG9ubHkgZGVmYXVsdEJyYW5jaDogUmVsZWFzZUJyYW5jaDtcblxuICBjb25zdHJ1Y3Rvcihwcm9qZWN0OiBQcm9qZWN0LCBvcHRpb25zOiBSZWxlYXNlT3B0aW9ucykge1xuICAgIHN1cGVyKHByb2plY3QpO1xuXG4gICAgaWYgKEFycmF5LmlzQXJyYXkob3B0aW9ucy5yZWxlYXNlQnJhbmNoZXMpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ1wicmVsZWFzZUJyYW5jaGVzXCIgaXMgbm8gbG9uZ2VyIGFuIGFycmF5LiBTZWUgdHlwZSBhbm5vdGF0aW9ucycpO1xuICAgIH1cblxuICAgIHRoaXMudGFzayA9IG9wdGlvbnMudGFzaztcbiAgICB0aGlzLnByZUJ1aWxkU3RlcHMgPSBvcHRpb25zLnJlbGVhc2VXb3JrZmxvd1NldHVwU3RlcHMgPz8gW107XG4gICAgdGhpcy5wb3N0QnVpbGRTdGVwcyA9IG9wdGlvbnMucG9zdEJ1aWxkU3RlcHMgPz8gW107XG4gICAgdGhpcy5hbnRpdGFtcGVyID0gb3B0aW9ucy5hbnRpdGFtcGVyID8/IHRydWU7XG4gICAgdGhpcy5hcnRpZmFjdERpcmVjdG9yeSA9IG9wdGlvbnMuYXJ0aWZhY3RzRGlyZWN0b3J5ID8/ICdkaXN0JztcbiAgICB0aGlzLnZlcnNpb25GaWxlID0gb3B0aW9ucy52ZXJzaW9uRmlsZTtcbiAgICB0aGlzLnJlbGVhc2VTY2hlZHVsZSA9IG9wdGlvbnMucmVsZWFzZVNjaGVkdWxlO1xuICAgIHRoaXMucmVsZWFzZUV2ZXJ5Q29tbWl0ID0gb3B0aW9ucy5yZWxlYXNlRXZlcnlDb21taXQgPz8gdHJ1ZTtcbiAgICB0aGlzLmNvbnRhaW5lckltYWdlID0gb3B0aW9ucy53b3JrZmxvd0NvbnRhaW5lckltYWdlO1xuXG4gICAgdGhpcy52ZXJzaW9uID0gbmV3IFZlcnNpb24ocHJvamVjdCwge1xuICAgICAgdmVyc2lvbkZpbGU6IG9wdGlvbnMudmVyc2lvbkZpbGUsXG4gICAgfSk7XG5cbiAgICB0aGlzLnB1Ymxpc2hlciA9IG5ldyBQdWJsaXNoZXIocHJvamVjdCwge1xuICAgICAgYXJ0aWZhY3ROYW1lOiB0aGlzLmFydGlmYWN0RGlyZWN0b3J5LFxuICAgICAgYnVpbGRKb2JJZDogQlVJTERfSk9CSUQsXG4gICAgICBqc2lpUmVsZWFzZVZlcnNpb246IG9wdGlvbnMuanNpaVJlbGVhc2VWZXJzaW9uLFxuICAgIH0pO1xuXG4gICAgLy8gYWRkIHRoZSBkZWZhdWx0IGJyYW5jaFxuICAgIHRoaXMuZGVmYXVsdEJyYW5jaCA9IHtcbiAgICAgIG5hbWU6IG9wdGlvbnMuYnJhbmNoLFxuICAgICAgcHJlcmVsZWFzZTogb3B0aW9ucy5wcmVyZWxlYXNlLFxuICAgICAgbWFqb3JWZXJzaW9uOiBvcHRpb25zLm1ham9yVmVyc2lvbixcbiAgICAgIHdvcmtmbG93TmFtZTogb3B0aW9ucy5yZWxlYXNlV29ya2Zsb3dOYW1lID8/ICdSZWxlYXNlJyxcbiAgICB9O1xuXG4gICAgdGhpcy5icmFuY2hlcy5wdXNoKHRoaXMuZGVmYXVsdEJyYW5jaCk7XG5cbiAgICBmb3IgKGNvbnN0IFtuYW1lLCBvcHRzXSBvZiBPYmplY3QuZW50cmllcyhvcHRpb25zLnJlbGVhc2VCcmFuY2hlcyA/PyB7fSkpIHtcbiAgICAgIHRoaXMuYWRkQnJhbmNoKG5hbWUsIG9wdHMpO1xuICAgIH1cbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIGFkZEJyYW5jaChicmFuY2g6IHN0cmluZywgb3B0aW9uczogQnJhbmNoT3B0aW9ucykge1xuICAgIGlmICh0aGlzLmJyYW5jaGVzLmZpbmQoYiA9PiBiLm5hbWUgPT09IGJyYW5jaCkpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgVGhlIHJlbGVhc2UgYnJhbmNoICR7YnJhbmNofSBpcyBhbHJlYWR5IGRlZmluZWRgKTtcbiAgICB9XG5cbiAgICAvLyBpZiB3ZSBhZGQgYSBicmFuY2gsIHdlIHJlcXVpcmUgdGhhdCB0aGUgZGVmYXVsdCBicmFuY2ggd2lsbCBhbHNvIGRlZmluZSBhXG4gICAgLy8gbWFqb3IgdmVyc2lvbi5cbiAgICBpZiAodGhpcy5kZWZhdWx0QnJhbmNoLm1ham9yVmVyc2lvbiA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ3lvdSBtdXN0IHNwZWNpZnkgXCJtYWpvclZlcnNpb25cIiBmb3IgdGhlIGRlZmF1bHQgYnJhbmNoIHdoZW4gYWRkaW5nIG11bHRpcGxlIHJlbGVhc2UgYnJhbmNoZXMnKTtcbiAgICB9XG5cbiAgICB0aGlzLmJyYW5jaGVzLnB1c2goe1xuICAgICAgbmFtZTogYnJhbmNoLFxuICAgICAgLi4ub3B0aW9ucyxcbiAgICB9KTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIGFkZEpvYnMoam9iczogUmVjb3JkPHN0cmluZywgd29ya2Zsb3dzLkpvYj4pIHtcbiAgICBmb3IgKGNvbnN0IFtuYW1lLCBqb2JdIG9mIE9iamVjdC5lbnRyaWVzKGpvYnMpKSB7XG4gICAgICB0aGlzLmpvYnNbbmFtZV0gPSBqb2I7XG4gICAgfVxuICB9XG5cbiAgLy8gcmVuZGVyIGEgd29ya2Zsb3cgcGVyIGJyYW5jaCBhbmQgYWxsIHRoZSBqb2JzIHRvIGl0XG4gIHB1YmxpYyBwcmVTeW50aGVzaXplKCkge1xuICAgIGZvciAoY29uc3QgYnJhbmNoIG9mIHRoaXMuYnJhbmNoZXMpIHtcbiAgICAgIGNvbnN0IHdvcmtmbG93ID0gdGhpcy5jcmVhdGVXb3JrZmxvdyhicmFuY2gpO1xuICAgICAgd29ya2Zsb3cuYWRkSm9icyh0aGlzLnB1Ymxpc2hlci5yZW5kZXIoKSk7XG4gICAgICB3b3JrZmxvdy5hZGRKb2JzKHRoaXMuam9icyk7XG4gICAgfVxuICB9XG5cbiAgcHJpdmF0ZSBjcmVhdGVXb3JrZmxvdyhicmFuY2g6IFJlbGVhc2VCcmFuY2gpOiBHaXRodWJXb3JrZmxvdyB7XG4gICAgY29uc3QgZ2l0aHViID0gdGhpcy5wcm9qZWN0LmdpdGh1YjtcbiAgICBpZiAoIWdpdGh1YikgeyB0aHJvdyBuZXcgRXJyb3IoJ25vIGdpdGh1YiBzdXBwb3J0Jyk7IH1cblxuICAgIGNvbnN0IHdvcmtmbG93TmFtZSA9IGJyYW5jaC53b3JrZmxvd05hbWUgPz8gYHJlbGVhc2UtJHticmFuY2gubmFtZX1gO1xuXG4gICAgLy8gdG8gYXZvaWQgcmFjZSBjb25kaXRpb25zIGJldHdlZW4gdHdvIGNvbW1pdHMgdHJ5aW5nIHRvIHJlbGVhc2UgdGhlIHNhbWVcbiAgICAvLyB2ZXJzaW9uLCB3ZSBjaGVjayBpZiB0aGUgaGVhZCBzaGEgaXMgaWRlbnRpY2FsIHRvIHRoZSByZW1vdGUgc2hhLiBpZlxuICAgIC8vIG5vdCwgd2Ugd2lsbCBza2lwIHRoZSByZWxlYXNlIGFuZCBqdXN0IGZpbmlzaCB0aGUgYnVpbGQuXG4gICAgY29uc3QgZ2l0UmVtb3RlU3RlcCA9ICdnaXRfcmVtb3RlJztcbiAgICBjb25zdCBsYXRlc3RDb21taXRPdXRwdXQgPSAnbGF0ZXN0X2NvbW1pdCc7XG4gICAgY29uc3Qgbm9OZXdDb21taXRzID0gYFxcJHt7IHN0ZXBzLiR7Z2l0UmVtb3RlU3RlcH0ub3V0cHV0cy4ke2xhdGVzdENvbW1pdE91dHB1dH0gPT0gZ2l0aHViLnNoYSB9fWA7XG5cbiAgICBjb25zdCBzdGVwcyA9IG5ldyBBcnJheTx3b3JrZmxvd3MuSm9iU3RlcD4oKTtcblxuICAgIC8vIGNoZWNrIG91dCBzb3VyY2VzLlxuICAgIHN0ZXBzLnB1c2goe1xuICAgICAgbmFtZTogJ0NoZWNrb3V0JyxcbiAgICAgIHVzZXM6ICdhY3Rpb25zL2NoZWNrb3V0QHYyJyxcbiAgICAgIHdpdGg6IHtcbiAgICAgICAgLy8gd2UgbXVzdCB1c2UgJ2ZldGNoLWRlcHRoPTAnIGluIG9yZGVyIHRvIGZldGNoIGFsbCB0YWdzXG4gICAgICAgIC8vIG90aGVyd2lzZSB0YWdzIGFyZSBub3QgY2hlY2tlZCBvdXRcbiAgICAgICAgJ2ZldGNoLWRlcHRoJzogMCxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICAvLyBzZXRzIGdpdCBpZGVudGl0eSBzbyB3ZSBjYW4gcHVzaCBsYXRlclxuICAgIHN0ZXBzLnB1c2goe1xuICAgICAgbmFtZTogJ1NldCBnaXQgaWRlbnRpdHknLFxuICAgICAgcnVuOiBbXG4gICAgICAgICdnaXQgY29uZmlnIHVzZXIubmFtZSBcIkF1dG9tYXRpb25cIicsXG4gICAgICAgICdnaXQgY29uZmlnIHVzZXIuZW1haWwgXCJnaXRodWItYWN0aW9uc0BnaXRodWIuY29tXCInLFxuICAgICAgXS5qb2luKCdcXG4nKSxcbiAgICB9KTtcblxuICAgIHN0ZXBzLnB1c2goLi4udGhpcy5wcmVCdWlsZFN0ZXBzID8/IFtdKTtcblxuICAgIGNvbnN0IGVudjogUmVjb3JkPHN0cmluZywgc3RyaW5nPiA9IHt9O1xuXG4gICAgaWYgKGJyYW5jaC5tYWpvclZlcnNpb24gIT09IHVuZGVmaW5lZCkge1xuICAgICAgZW52Lk1BSk9SID0gYnJhbmNoLm1ham9yVmVyc2lvbi50b1N0cmluZygpO1xuICAgIH1cblxuICAgIGlmIChicmFuY2gucHJlcmVsZWFzZSkge1xuICAgICAgZW52LlBSRVJFTEVBU0UgPSBicmFuY2gucHJlcmVsZWFzZTtcbiAgICB9XG5cbiAgICBzdGVwcy5wdXNoKHtcbiAgICAgIG5hbWU6ICdCdW1wIHRvIG5leHQgdmVyc2lvbicsXG4gICAgICBydW46IHRoaXMucHJvamVjdC5ydW5UYXNrQ29tbWFuZCh0aGlzLnZlcnNpb24uYnVtcFRhc2spLFxuICAgICAgZW52OiBPYmplY3Qua2V5cyhlbnYpLmxlbmd0aCA/IGVudiA6IHVuZGVmaW5lZCxcbiAgICB9KTtcblxuICAgIC8vIHJ1biB0aGUgbWFpbiBidWlsZCB0YXNrXG4gICAgc3RlcHMucHVzaCh7XG4gICAgICBuYW1lOiB0aGlzLnRhc2submFtZSxcbiAgICAgIHJ1bjogdGhpcy5wcm9qZWN0LnJ1blRhc2tDb21tYW5kKHRoaXMudGFzayksXG4gICAgfSk7XG5cbiAgICAvLyBydW4gcG9zdC1idWlsZCBzdGVwc1xuICAgIHN0ZXBzLnB1c2goLi4udGhpcy5wb3N0QnVpbGRTdGVwcyk7XG5cbiAgICAvLyBjcmVhdGUgYSBiYWNrdXAgb2YgdGhlIHZlcnNpb24gSlNPTiBmaWxlIChlLmcuIHBhY2thZ2UuanNvbikgYmVjYXVzZSB3ZVxuICAgIC8vIGFyZSBnb2luZyB0byByZXZlcnQgdGhlIGJ1bXAgYW5kIHdlIG5lZWQgdGhlIHZlcnNpb24gbnVtYmVyIGluIG9yZGVyIHRvXG4gICAgLy8gY3JlYXRlIHRoZSBnaXRodWIgcmVsZWFzZS5cbiAgICBjb25zdCB2ZXJzaW9uSnNvbkJhY2t1cCA9IGAke3RoaXMudmVyc2lvbkZpbGV9LmJhay5qc29uYDtcbiAgICBzdGVwcy5wdXNoKHtcbiAgICAgIG5hbWU6ICdCYWNrdXAgdmVyc2lvbiBmaWxlJyxcbiAgICAgIHJ1bjogYGNwIC1mICR7dGhpcy52ZXJzaW9uRmlsZX0gJHt2ZXJzaW9uSnNvbkJhY2t1cH1gLFxuICAgIH0pO1xuXG4gICAgLy8gcmV2ZXJ0IHRoZSBidW1wIHNvIGFudGktdGFtcGVyIHdpbGwgbm90IGZhaWxcbiAgICBzdGVwcy5wdXNoKHtcbiAgICAgIG5hbWU6ICdVbmJ1bXAnLFxuICAgICAgcnVuOiB0aGlzLnByb2plY3QucnVuVGFza0NvbW1hbmQodGhpcy52ZXJzaW9uLnVuYnVtcFRhc2spLFxuICAgIH0pO1xuXG4gICAgLy8gYW50aS10YW1wZXIgY2hlY2sgKGZhaWxzIGlmIHRoZXJlIHdlcmUgY2hhbmdlcyB0byBjb21taXR0ZWQgZmlsZXMpXG4gICAgLy8gdGhpcyB3aWxsIGlkZW50aWZ5IGFueSBub24tY29tbWl0dGVkIGZpbGVzIGdlbmVyYXRlZCBkdXJpbmcgYnVpbGQgKGUuZy4gdGVzdCBzbmFwc2hvdHMpXG4gICAgaWYgKHRoaXMuYW50aXRhbXBlcikge1xuICAgICAgc3RlcHMucHVzaCh7XG4gICAgICAgIG5hbWU6ICdBbnRpLXRhbXBlciBjaGVjaycsXG4gICAgICAgIHJ1bjogJ2dpdCBkaWZmIC0taWdub3JlLXNwYWNlLWF0LWVvbCAtLWV4aXQtY29kZScsXG4gICAgICB9KTtcbiAgICB9XG5cbiAgICAvLyBjaGVjayBpZiBuZXcgY29tbWl0cyB3ZXJlIHB1c2hlZCB0byB0aGUgcmVwbyB3aGlsZSB3ZSB3ZXJlIGJ1aWxkaW5nLlxuICAgIC8vIGlmIG5ldyBjb21taXRzIGhhdmUgYmVlbiBwdXNoZWQsIHdlIHdpbGwgY2FuY2VsIHRoaXMgcmVsZWFzZVxuICAgIHN0ZXBzLnB1c2goe1xuICAgICAgbmFtZTogJ0NoZWNrIGZvciBuZXcgY29tbWl0cycsXG4gICAgICBpZDogZ2l0UmVtb3RlU3RlcCxcbiAgICAgIHJ1bjogYGVjaG8gOjpzZXQtb3V0cHV0IG5hbWU9JHtsYXRlc3RDb21taXRPdXRwdXR9OjpcIiQoZ2l0IGxzLXJlbW90ZSBvcmlnaW4gLWggXFwke3sgZ2l0aHViLnJlZiB9fSB8IGN1dCAtZjEpXCJgLFxuICAgIH0pO1xuXG4gICAgLy8gY3JlYXRlIGEgZ2l0aHViIHJlbGVhc2VcbiAgICBjb25zdCBnZXRWZXJzaW9uID0gYHYkKG5vZGUgLXAgXFxcInJlcXVpcmUoXFwnLi8ke3ZlcnNpb25Kc29uQmFja3VwfVxcJykudmVyc2lvblxcXCIpYDtcbiAgICBzdGVwcy5wdXNoKHtcbiAgICAgIG5hbWU6ICdDcmVhdGUgcmVsZWFzZScsXG4gICAgICBpZjogbm9OZXdDb21taXRzLFxuICAgICAgcnVuOiBbXG4gICAgICAgIGBnaCByZWxlYXNlIGNyZWF0ZSAke2dldFZlcnNpb259YCxcbiAgICAgICAgYC1GICR7dGhpcy52ZXJzaW9uLmNoYW5nZWxvZ0ZpbGV9YCxcbiAgICAgICAgYC10ICR7Z2V0VmVyc2lvbn1gLFxuICAgICAgXS5qb2luKCcgJyksXG4gICAgICBlbnY6IHtcbiAgICAgICAgR0lUSFVCX1RPS0VOOiAnJHt7IHNlY3JldHMuR0lUSFVCX1RPS0VOIH19JyxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICBzdGVwcy5wdXNoKHtcbiAgICAgIG5hbWU6ICdVcGxvYWQgYXJ0aWZhY3QnLFxuICAgICAgaWY6IG5vTmV3Q29tbWl0cyxcbiAgICAgIHVzZXM6ICdhY3Rpb25zL3VwbG9hZC1hcnRpZmFjdEB2Mi4xLjEnLFxuICAgICAgd2l0aDoge1xuICAgICAgICBuYW1lOiB0aGlzLmFydGlmYWN0RGlyZWN0b3J5LFxuICAgICAgICBwYXRoOiB0aGlzLmFydGlmYWN0RGlyZWN0b3J5LFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIGNvbnN0IHdvcmtmbG93ID0gZ2l0aHViLmFkZFdvcmtmbG93KHdvcmtmbG93TmFtZSk7XG5cbiAgICAvLyBkZXRlcm1pbmUgcmVsZWFzZSB0cmlnZ2Vyc1xuICAgIHdvcmtmbG93Lm9uKHtcbiAgICAgIHNjaGVkdWxlOiB0aGlzLnJlbGVhc2VTY2hlZHVsZSA/IFt7IGNyb246IHRoaXMucmVsZWFzZVNjaGVkdWxlIH1dIDogdW5kZWZpbmVkLFxuICAgICAgcHVzaDogKHRoaXMucmVsZWFzZUV2ZXJ5Q29tbWl0KSA/IHsgYnJhbmNoZXM6IFticmFuY2gubmFtZV0gfSA6IHVuZGVmaW5lZCxcbiAgICAgIHdvcmtmbG93RGlzcGF0Y2g6IHt9LCAvLyBhbGxvdyBtYW51YWwgdHJpZ2dlcmluZ1xuICAgIH0pO1xuXG4gICAgLy8gYWRkIG1haW4gYnVpbGQgam9iXG4gICAgd29ya2Zsb3cuYWRkSm9icyh7XG4gICAgICBbQlVJTERfSk9CSURdOiB7XG4gICAgICAgIHJ1bnNPbjogJ3VidW50dS1sYXRlc3QnLFxuICAgICAgICBjb250YWluZXI6IHRoaXMuY29udGFpbmVySW1hZ2UgPyB7IGltYWdlOiB0aGlzLmNvbnRhaW5lckltYWdlIH0gOiB1bmRlZmluZWQsXG4gICAgICAgIGVudjoge1xuICAgICAgICAgIENJOiAndHJ1ZScsXG4gICAgICAgICAgUkVMRUFTRTogJ3RydWUnLFxuICAgICAgICB9LFxuICAgICAgICBwZXJtaXNzaW9uczoge1xuICAgICAgICAgIGNvbnRlbnRzOiB3b3JrZmxvd3MuSm9iUGVybWlzc2lvbi5XUklURSxcbiAgICAgICAgfSxcbiAgICAgICAgc3RlcHM6IHN0ZXBzLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIHJldHVybiB3b3JrZmxvdztcbiAgfVxufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIEJyYW5jaE9wdGlvbnMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHdvcmtmbG93TmFtZT86IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgbWFqb3JWZXJzaW9uOiBudW1iZXI7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHByZXJlbGVhc2U/OiBzdHJpbmc7XG59XG5cbmludGVyZmFjZSBSZWxlYXNlQnJhbmNoIGV4dGVuZHMgUGFydGlhbDxCcmFuY2hPcHRpb25zPiB7XG4gIHJlYWRvbmx5IG5hbWU6IHN0cmluZztcbn1cbiJdfQ==