"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.bump = void 0;
const path_1 = require("path");
const fs_extra_1 = require("fs-extra");
const logging = require("../logging");
const util_1 = require("../util");
/**
 * Resolves the latest version from git tags and uses `standard-version` to bump
 * to the next version based on commits.
 *
 * This expects `standard-version` to be installed in the path.
 *
 * @param cwd working directory (git repository)
 * @param options options
 */
async function bump(cwd, options) {
    const versionFile = path_1.join(cwd, options.versionFile);
    const prerelease = options.prerelease;
    const major = options.majorVersion;
    const changjelog = options.changelog;
    // filter only tags for this major version if specified (start  with "vNN.").
    const prefix = major ? `v${major}.*` : 'v*';
    const listGitTags = [
        'git',
        '-c "versionsort.suffix=-"',
        'tag',
        '--sort="-version:refname"',
        '--list',
        `"${prefix}"`,
    ].join(' ');
    const stdout = util_1.execCapture(listGitTags, { cwd }).toString('utf8');
    let tags = stdout === null || stdout === void 0 ? void 0 : stdout.split('\n');
    // if "pre" is set, filter versions that end with "-PRE.ddd".
    if (prerelease) {
        tags = tags.filter(x => new RegExp(`-${prerelease}\.[0-9]+$`).test(x));
    }
    tags = tags.filter(x => x);
    // if a pre-release tag is used, then add it to the initial version
    let firstRelease = false;
    let latest;
    if (tags.length > 0) {
        latest = tags[0];
    }
    else {
        const initial = `v${major !== null && major !== void 0 ? major : 0}.0.0`;
        latest = prerelease ? `${initial}-${prerelease}.0` : initial;
        firstRelease = true;
    }
    // remove "v" prefix (if exists)
    if (latest.startsWith('v')) {
        latest = latest.substr(1);
    }
    const content = await tryReadVersionFile(versionFile);
    // update version
    content.version = latest;
    logging.info(`Update ${versionFile} to latest resolved version: ${latest}`);
    await fs_extra_1.writeFile(versionFile, JSON.stringify(content, undefined, 2));
    // create a standard-version configuration file
    const rcfile = path_1.join(cwd, '.versionrc.json');
    await fs_extra_1.writeFile(rcfile, JSON.stringify({
        packageFiles: [{
                filename: versionFile,
                type: 'json',
            }],
        bumpFiles: [{
                filename: versionFile,
                type: 'json',
            }],
        commitAll: false,
        infile: changjelog,
        prerelease: prerelease,
        header: '',
        skip: {
            commit: true,
            tag: true,
        },
    }, undefined, 2));
    const cmd = ['npx', 'standard-version@^9'];
    if (firstRelease) {
        cmd.push('--first-release');
    }
    util_1.execCapture(cmd.join(' '), { stdio: 'inherit', cwd });
    await fs_extra_1.remove(rcfile);
    const newVersion = (await tryReadVersionFile(versionFile)).version;
    if (!newVersion) {
        throw new Error(`bump failed: ${versionFile} does not have a new version`);
    }
    // if MAJOR is defined, ensure that the new version is within the same major version
    if (major) {
        if (!newVersion.startsWith(`${major}.`)) {
            throw new Error(`bump failed: this branch is configured to only publish v${major} releases - bump resulted in ${newVersion}`);
        }
    }
}
exports.bump = bump;
async function tryReadVersionFile(versionFile) {
    if (!(await fs_extra_1.pathExists(versionFile))) {
        return {};
    }
    return JSON.parse(await fs_extra_1.readFile(versionFile, 'utf8'));
}
//# sourceMappingURL=data:application/json;base64,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