"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProjectType = exports.Project = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const cleanup_1 = require("./cleanup");
const clobber_1 = require("./clobber");
const common_1 = require("./common");
const deps_1 = require("./deps");
const file_1 = require("./file");
const gitattributes_1 = require("./git/gitattributes");
const github_1 = require("./github");
const gitpod_1 = require("./gitpod");
const ignore_file_1 = require("./ignore-file");
const render_options_1 = require("./javascript/render-options");
const json_1 = require("./json");
const logger_1 = require("./logger");
const object_file_1 = require("./object-file");
const readme_1 = require("./readme");
const tasks_1 = require("./tasks/tasks");
const util_1 = require("./util");
const vscode_1 = require("./vscode");
/**
 * (experimental) Base project.
 *
 * @experimental
 */
class Project {
    /**
     * @experimental
     */
    constructor(options) {
        var _b, _c, _d, _e;
        this._components = new Array();
        this.subprojects = new Array();
        this.tips = new Array();
        this.newProject = render_options_1.resolveNewProject(options);
        this.name = options.name;
        this.parent = options.parent;
        this.excludeFromCleanup = [];
        this.projectType = (_b = options.projectType) !== null && _b !== void 0 ? _b : ProjectType.UNKNOWN;
        if (this.parent && options.outdir && path.isAbsolute(options.outdir)) {
            throw new Error('"outdir" must be a relative path');
        }
        let outdir;
        if (options.parent) {
            if (!options.outdir) {
                throw new Error('"outdir" must be specified for subprojects');
            }
            outdir = path.join(options.parent.outdir, options.outdir);
        }
        else {
            outdir = (_c = options.outdir) !== null && _c !== void 0 ? _c : '.';
        }
        this.outdir = path.resolve(outdir);
        this.root = this.parent ? this.parent.root : this;
        // must happen after this.outdir, this.parent and this.root are initialized
        (_d = this.parent) === null || _d === void 0 ? void 0 : _d._addSubProject(this);
        // ------------------------------------------------------------------------
        this.gitattributes = new gitattributes_1.GitAttributesFile(this);
        this.annotateGenerated('/.projen/**'); // contents  of the .projen/ directory are generated by projen
        this.annotateGenerated(`/${this.gitattributes.path}`); // the .gitattributes file itself is generated
        this.gitignore = new ignore_file_1.IgnoreFile(this, '.gitignore');
        this.gitignore.exclude('node_modules/'); // created by running `npx projen`
        this.gitignore.include(`/${this.gitattributes.path}`);
        // oh no: tasks depends on gitignore so it has to be initialized after
        // smells like dep injectionn but god forbid.
        this.tasks = new tasks_1.Tasks(this);
        this.deps = new deps_1.Dependencies(this);
        this.logger = new logger_1.Logger(this, options.logging);
        // we only allow these global services to be used in root projects
        this.github = !this.parent ? new github_1.GitHub(this, options) : undefined;
        this.vscode = !this.parent ? new vscode_1.VsCode(this) : undefined;
        this.gitpod = options.gitpod ? new gitpod_1.Gitpod(this) : undefined;
        this.devContainer = options.devContainer ? new vscode_1.DevContainer(this) : undefined;
        if ((_e = options.clobber) !== null && _e !== void 0 ? _e : true) {
            new clobber_1.Clobber(this);
        }
        new readme_1.SampleReadme(this, options.readme);
        if (options.autoApproveOptions) {
            this.autoApprove = new github_1.AutoApprove(this, options.autoApproveOptions);
        }
    }
    /**
     * (experimental) Returns all the components within this project.
     *
     * @experimental
     */
    get components() {
        return [...this._components];
    }
    /**
     * (experimental) All files in this project.
     *
     * @experimental
     */
    get files() {
        const isFile = (c) => c instanceof file_1.FileBase;
        return this._components.filter(isFile).sort((f1, f2) => f1.path.localeCompare(f2.path));
    }
    /**
     * (experimental) Adds a new task to this project.
     *
     * This will fail if the project already has
     * a task with this name.
     *
     * @param name The task name to add.
     * @param props Task properties.
     * @experimental
     */
    addTask(name, props = {}) {
        return this.tasks.addTask(name, props);
    }
    /**
     * (experimental) Removes a task from a project.
     *
     * @param name The name of the task to remove.
     * @returns The `Task` that was removed, otherwise `undefined`.
     * @experimental
     */
    removeTask(name) {
        return this.tasks.removeTask(name);
    }
    /**
     * (experimental) Finds a file at the specified relative path within this project and all its subprojects.
     *
     * @param filePath The file path.
     * @returns a `FileBase` or undefined if there is no file in that path
     * @experimental
     */
    tryFindFile(filePath) {
        const absolute = path.isAbsolute(filePath) ? filePath : path.resolve(this.outdir, filePath);
        for (const file of this.files) {
            if (absolute === file.absolutePath) {
                return file;
            }
        }
        for (const child of this.subprojects) {
            const file = child.tryFindFile(absolute);
            if (file) {
                return file;
            }
        }
        return undefined;
    }
    /**
     * (deprecated) Finds a json file by name.
     *
     * @param filePath The file path.
     * @deprecated use `tryFindObjectFile`
     */
    tryFindJsonFile(filePath) {
        const file = this.tryFindObjectFile(filePath);
        if (!file) {
            return undefined;
        }
        if (!(file instanceof json_1.JsonFile)) {
            throw new Error(`found file ${filePath} but it is not a JsonFile. got: ${file.constructor.name}`);
        }
        return file;
    }
    /**
     * (experimental) Finds an object file (like JsonFile, YamlFile, etc.) by name.
     *
     * @param filePath The file path.
     * @experimental
     */
    tryFindObjectFile(filePath) {
        const file = this.tryFindFile(filePath);
        if (!file) {
            return undefined;
        }
        if (!(file instanceof object_file_1.ObjectFile)) {
            throw new Error(`found file ${filePath} but it is not a ObjectFile. got: ${file.constructor.name}`);
        }
        return file;
    }
    /**
     * (deprecated) Prints a "tip" message during synthesis.
     *
     * @param message The message.
     * @deprecated - use `project.logger.info(message)` to show messages during synthesis
     */
    addTip(message) {
        this.tips.push(message);
    }
    /**
     * (experimental) Exclude the matching files from pre-synth cleanup.
     *
     * Can be used when, for example, some
     * source files include the projen marker and we don't want them to be erased during synth.
     *
     * @param globs The glob patterns to match.
     * @experimental
     */
    addExcludeFromCleanup(...globs) {
        this.excludeFromCleanup.push(...globs);
    }
    /**
     * (experimental) Returns the shell command to execute in order to run a task.
     *
     * By default, this is `npx projen@<version> <task>`
     *
     * @param task The task for which the command is required.
     * @experimental
     */
    runTaskCommand(task) {
        return `npx projen@${common_1.PROJEN_VERSION} ${task.name}`;
    }
    /**
     * (experimental) Exclude these files from the bundled package.
     *
     * Implemented by project types based on the
     * packaging mechanism. For example, `NodeProject` delegates this to `.npmignore`.
     *
     * @param _pattern The glob pattern to exclude.
     * @experimental
     */
    addPackageIgnore(_pattern) {
        // nothing to do at the abstract level
    }
    /**
     * (experimental) Adds a .gitignore pattern.
     *
     * @param pattern The glob pattern to ignore.
     * @experimental
     */
    addGitIgnore(pattern) {
        this.gitignore.addPatterns(pattern);
    }
    /**
     * (experimental) Marks the provided file(s) as being generated.
     *
     * This is achieved using the
     * github-linguist attributes. Generated files do not count against the
     * repository statistics and language breakdown.
     *
     * @param glob the glob pattern to match (could be a file path).
     * @see https://github.com/github/linguist/blob/master/docs/overrides.md
     * @experimental
     */
    annotateGenerated(glob) {
        this.gitattributes.addAttributes(glob, 'linguist-generated');
    }
    /**
     * (experimental) Synthesize all project files into `outdir`.
     *
     * 1. Call "this.preSynthesize()"
     * 2. Delete all generated files
     * 3. Synthesize all sub-projects
     * 4. Synthesize all components of this project
     * 5. Call "postSynthesize()" for all components of this project
     * 6. Call "this.postSynthesize()"
     *
     * @experimental
     */
    synth() {
        const outdir = this.outdir;
        this.logger.info('Synthesizing project...');
        this.preSynthesize();
        for (const comp of this._components) {
            comp.preSynthesize();
        }
        // we exclude all subproject directories to ensure that when subproject.synth()
        // gets called below after cleanup(), subproject generated files are left intact
        for (const subproject of this.subprojects) {
            this.addExcludeFromCleanup(subproject.outdir + '/**');
        }
        // delete all generated files before we start synthesizing new ones
        cleanup_1.cleanup(outdir, this.excludeFromCleanup);
        for (const subproject of this.subprojects) {
            subproject.synth();
        }
        for (const comp of this._components) {
            comp.synthesize();
        }
        if (!util_1.isTruthy(process.env.PROJEN_DISABLE_POST)) {
            for (const comp of this._components) {
                comp.postSynthesize();
            }
            // project-level hook
            this.postSynthesize();
        }
        this.logger.info('Synthesis complete');
    }
    /**
     * (experimental) Called before all components are synthesized.
     *
     * @experimental
     */
    preSynthesize() { }
    /**
     * (experimental) Called after all components are synthesized.
     *
     * Order is *not* guaranteed.
     *
     * @experimental
     */
    postSynthesize() { }
    /**
     * Adds a component to the project.
     * @internal
     */
    _addComponent(component) {
        this._components.push(component);
    }
    /**
     * Adds a sub-project to this project.
     *
     * This is automatically called when a new project is created with `parent`
     * pointing to this project, so there is no real need to call this manually.
     *
     * @param sub-project The child project to add.
     * @internal
     */
    _addSubProject(subproject) {
        if (subproject.parent !== this) {
            throw new Error('"parent" of child project must be this project');
        }
        // check that `outdir` is exclusive
        for (const p of this.subprojects) {
            if (path.resolve(p.outdir) === path.resolve(subproject.outdir)) {
                throw new Error(`there is already a sub-project with "outdir": ${subproject.outdir}`);
            }
        }
        this.subprojects.push(subproject);
    }
}
exports.Project = Project;
_a = JSII_RTTI_SYMBOL_1;
Project[_a] = { fqn: "projen.Project", version: "0.23.2" };
/**
 * (experimental) The name of the default task (the task executed when `projen` is run without arguments).
 *
 * Normally
 * this task should synthesize the project files.
 *
 * @experimental
 */
Project.DEFAULT_TASK = 'default';
/**
 * (experimental) Which type of project this is.
 *
 * @experimental
 */
var ProjectType;
(function (ProjectType) {
    ProjectType["UNKNOWN"] = "unknown";
    ProjectType["LIB"] = "lib";
    ProjectType["APP"] = "app";
})(ProjectType = exports.ProjectType || (exports.ProjectType = {}));
//# sourceMappingURL=data:application/json;base64,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