"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NpmAccess = exports.NodePackageManager = exports.NodePackage = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path_1 = require("path");
const url_1 = require("url");
const fs_extra_1 = require("fs-extra");
const semver = require("semver");
const _resolve_1 = require("./_resolve");
const component_1 = require("./component");
const deps_1 = require("./deps");
const json_1 = require("./json");
const util_1 = require("./util");
const UNLICENSED = 'UNLICENSED';
const DEFAULT_NPM_REGISTRY_URL = 'https://registry.npmjs.org/';
const DEFAULT_NPM_TAG = 'latest';
const DEFAULT_NPM_TOKEN_SECRET = 'NPM_TOKEN';
/**
 * (experimental) Represents the npm `package.json` file.
 *
 * @experimental
 */
class NodePackage extends component_1.Component {
    /**
     * @experimental
     */
    constructor(project, options = {}) {
        var _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p;
        super(project);
        this.keywords = new Set();
        this.bin = {};
        this.engines = {};
        this.packageName = (_b = options.packageName) !== null && _b !== void 0 ? _b : project.name;
        this.projenCommand = (_c = options.projenCommand) !== null && _c !== void 0 ? _c : 'npx projen';
        this.peerDependencyOptions = (_d = options.peerDependencyOptions) !== null && _d !== void 0 ? _d : {};
        this.allowLibraryDependencies = (_e = options.allowLibraryDependencies) !== null && _e !== void 0 ? _e : true;
        this.packageManager = (_f = options.packageManager) !== null && _f !== void 0 ? _f : NodePackageManager.YARN;
        this.entrypoint = (_g = options.entrypoint) !== null && _g !== void 0 ? _g : 'lib/index.js';
        if (this.packageManager === NodePackageManager.YARN) {
            project.root.annotateGenerated('/yarn.lock');
        }
        const { npmDistTag, npmAccess, npmRegistry, npmRegistryUrl, npmTokenSecret } = this.parseNpmOptions(options);
        this.npmDistTag = npmDistTag;
        this.npmAccess = npmAccess;
        this.npmRegistry = npmRegistry;
        this.npmRegistryUrl = npmRegistryUrl;
        this.npmTokenSecret = npmTokenSecret;
        this.processDeps(options);
        const prev = (_h = this.readPackageJson()) !== null && _h !== void 0 ? _h : {};
        // empty objects are here to preserve order for backwards compatibility
        this.manifest = {
            name: this.packageName,
            description: options.description,
            repository: !options.repository ? undefined : {
                type: 'git',
                url: options.repository,
                directory: options.repositoryDirectory,
            },
            bin: () => this.renderBin(),
            scripts: () => this.renderScripts(),
            author: this.renderAuthor(options),
            devDependencies: {},
            peerDependencies: {},
            dependencies: {},
            bundledDependencies: [],
            keywords: () => this.renderKeywords(),
            engines: () => this.renderEngines(),
            main: this.entrypoint !== '' ? this.entrypoint : undefined,
            license: () => { var _b; return (_b = this.license) !== null && _b !== void 0 ? _b : UNLICENSED; },
            homepage: options.homepage,
            publishConfig: () => this.renderPublishConfig(),
            // in release CI builds we bump the version before we run "build" so we want
            // to preserve the version number. otherwise, we always set it to 0.0.0
            version: this.determineVersion(prev === null || prev === void 0 ? void 0 : prev.version),
        };
        // override any scripts from options (if specified)
        for (const [cmdname, shell] of Object.entries((_j = options.scripts) !== null && _j !== void 0 ? _j : {})) {
            project.addTask(cmdname, { exec: shell });
        }
        this.file = new json_1.JsonFile(this.project, 'package.json', {
            obj: this.manifest,
            readonly: false,
        });
        this.addKeywords(...(_k = options.keywords) !== null && _k !== void 0 ? _k : []);
        this.addBin((_l = options.bin) !== null && _l !== void 0 ? _l : {});
        // automatically add all executable files under "bin"
        if ((_m = options.autoDetectBin) !== null && _m !== void 0 ? _m : true) {
            this.autoDiscoverBinaries();
        }
        // node version
        this.minNodeVersion = options.minNodeVersion;
        this.maxNodeVersion = options.maxNodeVersion;
        this.addNodeEngine();
        // license
        if ((_o = options.licensed) !== null && _o !== void 0 ? _o : true) {
            this.license = (_p = options.license) !== null && _p !== void 0 ? _p : 'Apache-2.0';
        }
    }
    /**
     * (experimental) Defines normal dependencies.
     *
     * @param deps Names modules to install.
     * @experimental
     */
    addDeps(...deps) {
        for (const dep of deps) {
            this.project.deps.addDependency(dep, deps_1.DependencyType.RUNTIME);
        }
    }
    /**
     * (experimental) Defines development/test dependencies.
     *
     * @param deps Names modules to install.
     * @experimental
     */
    addDevDeps(...deps) {
        for (const dep of deps) {
            this.project.deps.addDependency(dep, deps_1.DependencyType.BUILD);
        }
    }
    /**
     * (experimental) Defines peer dependencies.
     *
     * When adding peer dependencies, a devDependency will also be added on the
     * pinned version of the declared peer. This will ensure that you are testing
     * your code against the minimum version required from your consumers.
     *
     * @param deps Names modules to install.
     * @experimental
     */
    addPeerDeps(...deps) {
        if (Object.keys(deps).length && !this.allowLibraryDependencies) {
            throw new Error(`cannot add peer dependencies to an APP project: ${Object.keys(deps).join(',')}`);
        }
        for (const dep of deps) {
            this.project.deps.addDependency(dep, deps_1.DependencyType.PEER);
        }
    }
    /**
     * (experimental) Defines bundled dependencies.
     *
     * Bundled dependencies will be added as normal dependencies as well as to the
     * `bundledDependencies` section of your `package.json`.
     *
     * @param deps Names modules to install.
     * @experimental
     */
    addBundledDeps(...deps) {
        if (deps.length && !this.allowLibraryDependencies) {
            throw new Error(`cannot add bundled dependencies to an APP project: ${deps.join(',')}`);
        }
        for (const dep of deps) {
            this.project.deps.addDependency(dep, deps_1.DependencyType.BUNDLED);
        }
    }
    /**
     * (experimental) Adds an `engines` requirement to your package.
     *
     * @param engine The engine (e.g. `node`).
     * @param version The semantic version requirement (e.g. `^10`).
     * @experimental
     */
    addEngine(engine, version) {
        this.engines[engine] = version;
    }
    /**
     * (experimental) Adds keywords to package.json (deduplicated).
     *
     * @param keywords The keywords to add.
     * @experimental
     */
    addKeywords(...keywords) {
        for (const k of keywords) {
            this.keywords.add(k);
        }
    }
    /**
     * @experimental
     */
    addBin(bins) {
        for (const [k, v] of Object.entries(bins)) {
            this.bin[k] = v;
        }
    }
    /**
     * (experimental) Override the contents of an npm package.json script.
     *
     * @param name The script name.
     * @param command The command to execute.
     * @experimental
     */
    setScript(name, command) {
        this.file.addOverride(`scripts.${name}`, command);
    }
    /**
     * (experimental) Removes the npm script (always successful).
     *
     * @param name The name of the script.
     * @experimental
     */
    removeScript(name) {
        this.file.addDeletionOverride(`scripts.${name}`);
    }
    /**
     * (deprecated) Indicates if a script by the given name is defined.
     *
     * @param name The name of the script.
     * @deprecated Use `project.tasks.tryFind(name)`
     */
    hasScript(name) {
        return this.project.tasks.tryFind(name) !== undefined;
    }
    /**
     * (experimental) Directly set fields in `package.json`.
     *
     * @param name field name.
     * @param value field value.
     * @experimental
     * @escape true
     */
    addField(name, value) {
        this.manifest[name] = value;
    }
    /**
     * (experimental) Sets the package version.
     *
     * @param version Package version.
     * @experimental
     */
    addVersion(version) {
        this.manifest.version = version;
    }
    /**
     * (experimental) Returns the command to execute in order to install all dependencies (always frozen).
     *
     * @experimental
     */
    get installCommand() {
        return this.renderInstallCommand(true);
    }
    /**
     * (experimental) Renders `yarn install` or `npm install` with lockfile update (not frozen).
     *
     * @experimental
     */
    get installAndUpdateLockfileCommand() {
        return this.renderInstallCommand(false);
    }
    /**
     * (experimental) Render a package manager specific command to upgrade all requested dependencies.
     *
     * @experimental
     */
    renderUpgradePackagesCommand(exclude, include) {
        const project = this.project;
        function upgradePackages(command) {
            return () => {
                if (exclude.length === 0 && !include) {
                    // request to upgrade all packages
                    // separated for asthetic reasons.
                    return command;
                }
                // filter by exclude and include.
                return `${command} ${project.deps.all
                    .map(d => d.name)
                    .filter(d => include ? include.includes(d) : true)
                    .filter(d => !exclude.includes(d))
                    .join(' ')}`;
            };
        }
        let lazy = undefined;
        switch (this.packageManager) {
            case NodePackageManager.YARN:
                lazy = upgradePackages('yarn upgrade');
                break;
            case NodePackageManager.NPM:
                lazy = upgradePackages('npm update');
                break;
            case NodePackageManager.PNPM:
                lazy = upgradePackages('pnpm update');
                break;
            default:
                throw new Error(`unexpected package manager ${this.packageManager}`);
        }
        // return a lazy function so that dependencies include ones that were
        // added post project instantiation (i.e using project.addDeps)
        return lazy;
    }
    // ---------------------------------------------------------------------------------------
    /**
     * (experimental) Called before synthesis.
     *
     * @experimental
     */
    preSynthesize() {
        super.preSynthesize();
        this._renderedDeps = this.renderDependencies();
    }
    /**
     * (experimental) Called after synthesis.
     *
     * Order is *not* guaranteed.
     *
     * @experimental
     */
    postSynthesize() {
        super.postSynthesize();
        const outdir = this.project.outdir;
        // now we run `yarn install`, but before we do that, remove the
        // `node_modules/projen` symlink so that yarn won't hate us.
        const projenModule = path_1.resolve('node_modules', 'projen');
        try {
            if (fs_extra_1.lstatSync(projenModule).isSymbolicLink()) {
                fs_extra_1.unlinkSync(projenModule);
            }
        }
        catch (e) { }
        util_1.exec(this.renderInstallCommand(this.isAutomatedBuild), { cwd: outdir });
        this.resolveDepsAndWritePackageJson(outdir);
    }
    /**
     * Returns `true` if we are running within a CI build.
     */
    get isAutomatedBuild() {
        return util_1.isTruthy(process.env.CI);
    }
    determineVersion(currVersion) {
        if (!this.isRelaseBuild) {
            return '0.0.0';
        }
        return currVersion !== null && currVersion !== void 0 ? currVersion : '0.0.0';
    }
    /**
     * Returns `true` if this is a CI release build.
     */
    get isRelaseBuild() {
        return util_1.isTruthy(process.env.RELEASE);
    }
    // -------------------------------------------------------------------------------------------
    parseNpmOptions(options) {
        var _b, _c, _d;
        let npmRegistryUrl = options.npmRegistryUrl;
        if (options.npmRegistry) {
            if (npmRegistryUrl) {
                throw new Error('cannot use the deprecated "npmRegistry" together with "npmRegistryUrl". please use "npmRegistryUrl" instead.');
            }
            npmRegistryUrl = `https://${options.npmRegistry}`;
        }
        const npmr = url_1.parse(npmRegistryUrl !== null && npmRegistryUrl !== void 0 ? npmRegistryUrl : DEFAULT_NPM_REGISTRY_URL);
        if (!npmr || !npmr.hostname || !npmr.href) {
            throw new Error(`unable to determine npm registry host from url ${npmRegistryUrl}. Is this really a URL?`);
        }
        const npmAccess = (_b = options.npmAccess) !== null && _b !== void 0 ? _b : defaultNpmAccess(this.packageName);
        if (!isScoped(this.packageName) && npmAccess === NpmAccess.RESTRICTED) {
            throw new Error(`"npmAccess" cannot be RESTRICTED for non-scoped npm package "${this.packageName}"`);
        }
        return {
            npmDistTag: (_c = options.npmDistTag) !== null && _c !== void 0 ? _c : DEFAULT_NPM_TAG,
            npmAccess,
            npmRegistry: npmr.hostname,
            npmRegistryUrl: npmr.href,
            npmTokenSecret: (_d = options.npmTokenSecret) !== null && _d !== void 0 ? _d : DEFAULT_NPM_TOKEN_SECRET,
        };
    }
    addNodeEngine() {
        if (!this.minNodeVersion && !this.maxNodeVersion) {
            return;
        }
        let nodeVersion = '';
        if (this.minNodeVersion) {
            nodeVersion += `>= ${this.minNodeVersion}`;
        }
        if (this.maxNodeVersion) {
            nodeVersion += ` <= ${this.maxNodeVersion}`;
        }
        this.addEngine('node', nodeVersion);
    }
    renderInstallCommand(frozen) {
        switch (this.packageManager) {
            case NodePackageManager.YARN:
                return [
                    'yarn install',
                    '--check-files',
                    ...frozen ? ['--frozen-lockfile'] : [],
                ].join(' ');
            case NodePackageManager.NPM:
                return frozen
                    ? 'npm ci'
                    : 'npm install';
            case NodePackageManager.PNPM:
                return [
                    'pnpm i',
                    ...frozen ? ['--frozen-lockfile'] : [],
                ].join(' ');
            default:
                throw new Error(`unexpected package manager ${this.packageManager}`);
        }
    }
    processDeps(options) {
        var _b, _c, _d, _e;
        this.addDeps(...(_b = options.deps) !== null && _b !== void 0 ? _b : []);
        this.addDevDeps(...(_c = options.devDeps) !== null && _c !== void 0 ? _c : []);
        this.addPeerDeps(...(_d = options.peerDeps) !== null && _d !== void 0 ? _d : []);
        this.addBundledDeps(...(_e = options.bundledDeps) !== null && _e !== void 0 ? _e : []);
    }
    renderDependencies() {
        var _b, _c, _d;
        const devDependencies = {};
        const peerDependencies = {};
        const dependencies = {};
        const bundledDependencies = new Array();
        // synthetic dependencies: add a pinned build dependency to ensure we are
        // testing against the minimum requirement of the peer.
        const pinned = (_b = this.peerDependencyOptions.pinnedDevDependency) !== null && _b !== void 0 ? _b : true;
        if (pinned) {
            for (const dep of this.project.deps.all.filter(d => d.type === deps_1.DependencyType.PEER)) {
                let req = dep.name;
                if (dep.version) {
                    const ver = (_c = semver.minVersion(dep.version)) === null || _c === void 0 ? void 0 : _c.version;
                    if (!ver) {
                        throw new Error(`unable to determine minimum semver for peer dependency ${dep.name}@${dep.version}`);
                    }
                    req += '@' + ver;
                }
                this.addDevDeps(req);
            }
        }
        for (const dep of this.project.deps.all) {
            const version = (_d = dep.version) !== null && _d !== void 0 ? _d : '*';
            switch (dep.type) {
                case deps_1.DependencyType.BUNDLED:
                    bundledDependencies.push(dep.name);
                    if (this.project.deps.all.find(d => d.name === dep.name && d.type === deps_1.DependencyType.PEER)) {
                        throw new Error(`unable to bundle "${dep.name}". it cannot appear as a peer dependency`);
                    }
                    // also add as a runtime dependency
                    dependencies[dep.name] = version;
                    break;
                case deps_1.DependencyType.PEER:
                    peerDependencies[dep.name] = version;
                    break;
                case deps_1.DependencyType.RUNTIME:
                    dependencies[dep.name] = version;
                    break;
                case deps_1.DependencyType.TEST:
                case deps_1.DependencyType.DEVENV:
                case deps_1.DependencyType.BUILD:
                    devDependencies[dep.name] = version;
                    break;
            }
        }
        // update the manifest we are about to save into `package.json`
        this.manifest.devDependencies = devDependencies;
        this.manifest.peerDependencies = peerDependencies;
        this.manifest.dependencies = dependencies;
        this.manifest.bundledDependencies = bundledDependencies;
        // nothing further to do if package.json file does not exist
        const pkg = this.readPackageJson();
        if (!pkg) {
            return { devDependencies, peerDependencies, dependencies };
        }
        const readDeps = (user, current = {}) => {
            for (const [name, userVersion] of Object.entries(user)) {
                const currentVersion = current[name];
                // respect user version if it's not '*' or if current version is undefined
                if (userVersion !== '*' || !currentVersion || currentVersion === '*') {
                    continue;
                }
                // memoize current version in memory so it is preserved when saving
                user[name] = currentVersion;
            }
            // report removals
            for (const name of Object.keys(current !== null && current !== void 0 ? current : {})) {
                if (!user[name]) {
                    this.project.logger.verbose(`${name}: removed`);
                }
            }
        };
        readDeps(devDependencies, pkg.devDependencies);
        readDeps(dependencies, pkg.dependencies);
        readDeps(peerDependencies, pkg.peerDependencies);
        return { devDependencies, dependencies, peerDependencies };
    }
    resolveDepsAndWritePackageJson(outdir) {
        const root = path_1.join(outdir, 'package.json');
        const pkg = fs_extra_1.readJsonSync(root);
        const resolveDeps = (current, user) => {
            const result = {};
            current = current !== null && current !== void 0 ? current : {};
            user = user !== null && user !== void 0 ? user : {};
            for (const [name, currentDefinition] of Object.entries(user)) {
                // find actual version from node_modules
                let desiredVersion = currentDefinition;
                if (currentDefinition === '*') {
                    try {
                        const modulePath = require.resolve(`${name}/package.json`, { paths: [outdir] });
                        const module = fs_extra_1.readJsonSync(modulePath);
                        desiredVersion = `^${module.version}`;
                    }
                    catch (e) { }
                    if (!desiredVersion) {
                        this.project.logger.warn(`unable to resolve version for ${name} from installed modules`);
                        continue;
                    }
                }
                if (currentDefinition !== desiredVersion) {
                    this.project.logger.verbose(`${name}: ${currentDefinition} => ${desiredVersion}`);
                }
                result[name] = desiredVersion;
            }
            // print removed packages
            for (const name of Object.keys(current)) {
                if (!result[name]) {
                    this.project.logger.verbose(`${name} removed`);
                }
            }
            return util_1.sorted(result);
        };
        const rendered = this._renderedDeps;
        if (!rendered) {
            throw new Error('assertion failed');
        }
        pkg.dependencies = resolveDeps(pkg.dependencies, rendered.dependencies);
        pkg.devDependencies = resolveDeps(pkg.devDependencies, rendered.devDependencies);
        pkg.peerDependencies = resolveDeps(pkg.peerDependencies, rendered.peerDependencies);
        util_1.writeFile(root, JSON.stringify(pkg, undefined, 2));
    }
    renderPublishConfig() {
        // omit values if they are the same as the npm defaults
        return _resolve_1.resolve({
            registry: this.npmRegistryUrl !== DEFAULT_NPM_REGISTRY_URL ? this.npmRegistryUrl : undefined,
            tag: this.npmDistTag !== DEFAULT_NPM_TAG ? this.npmDistTag : undefined,
            access: this.npmAccess !== defaultNpmAccess(this.packageName) ? this.npmAccess : undefined,
        }, { omitEmpty: true });
    }
    renderKeywords() {
        const kwds = Array.from(this.keywords);
        return util_1.sorted(kwds.sort());
    }
    renderEngines() {
        return util_1.sorted(this.engines);
    }
    autoDiscoverBinaries() {
        const binrel = 'bin';
        const bindir = path_1.join(this.project.outdir, binrel);
        if (fs_extra_1.existsSync(bindir)) {
            for (const file of fs_extra_1.readdirSync(bindir)) {
                try {
                    fs_extra_1.accessSync(path_1.join(bindir, file), fs_extra_1.constants.X_OK);
                    this.bin[file] = path_1.join(binrel, file).replace(/\\/g, '/');
                }
                catch (e) {
                    // not executable, skip
                }
            }
        }
    }
    renderAuthor(options) {
        var _b;
        let author;
        if (options.authorName) {
            author = {
                name: options.authorName,
                email: options.authorEmail,
                url: options.authorUrl,
                organization: (_b = options.authorOrganization) !== null && _b !== void 0 ? _b : false,
            };
        }
        else {
            if (options.authorEmail || options.authorUrl || options.authorOrganization !== undefined) {
                throw new Error('"authorName" is required if specifying "authorEmail" or "authorUrl"');
            }
        }
        return author;
    }
    renderBin() {
        return util_1.sorted(this.bin);
    }
    renderScripts() {
        const result = {};
        for (const task of this.project.tasks.all) {
            result[task.name] = this.npmScriptForTask(task);
        }
        return result;
    }
    npmScriptForTask(task) {
        return `${this.projenCommand} ${task.name}`;
    }
    readPackageJson() {
        const file = path_1.join(this.project.outdir, 'package.json');
        if (!fs_extra_1.existsSync(file)) {
            return undefined;
        }
        return fs_extra_1.readJsonSync(file);
    }
}
exports.NodePackage = NodePackage;
_a = JSII_RTTI_SYMBOL_1;
NodePackage[_a] = { fqn: "projen.NodePackage", version: "0.23.2" };
/**
 * (experimental) The node package manager to use.
 *
 * @experimental
 */
var NodePackageManager;
(function (NodePackageManager) {
    NodePackageManager["YARN"] = "yarn";
    NodePackageManager["NPM"] = "npm";
    NodePackageManager["PNPM"] = "pnpm";
})(NodePackageManager = exports.NodePackageManager || (exports.NodePackageManager = {}));
/**
 * (experimental) Npm package access level.
 *
 * @experimental
 */
var NpmAccess;
(function (NpmAccess) {
    NpmAccess["PUBLIC"] = "public";
    NpmAccess["RESTRICTED"] = "restricted";
})(NpmAccess = exports.NpmAccess || (exports.NpmAccess = {}));
/**
 * Determines if an npm package is "scoped" (i.e. it starts with "xxx@").
 */
function isScoped(packageName) {
    return packageName.includes('@');
}
function defaultNpmAccess(packageName) {
    return isScoped(packageName) ? NpmAccess.RESTRICTED : NpmAccess.PUBLIC;
}
//# sourceMappingURL=data:application/json;base64,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