"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.readJsiiManifest = exports.resolveProjectType = exports.discover = void 0;
const path = require("path");
const fs = require("fs-extra");
// eslint-disable-next-line @typescript-eslint/no-require-imports
const decamelize = require('decamelize');
const PROJEN_MODULE_ROOT = path.join(__dirname, '..');
const PROJECT_BASE_FQN = 'projen.Project';
/**
 * Returns a list of project types exported the modules defined in `moduleDirs`.
 * This list will always also include the built-in projen project types.
 * Modules without a .jsii manifest are skipped.
 *
 * @param moduleDirs A list of npm module directories
 */
function discover(...moduleDirs) {
    const jsii = {};
    const discoverJsii = (dir) => {
        const jsiiFile = path.join(dir, '.jsii');
        if (!fs.existsSync(jsiiFile)) {
            return;
        } // no jsii manifest
        const manifest = fs.readJsonSync(jsiiFile);
        for (const [fqn, type] of Object.entries(manifest.types)) {
            jsii[fqn] = type;
        }
    };
    // read all .jsii manifests from all modules (incl. projen itself) and merge
    // them all into a single map of fqn->type.
    for (const dir of [...moduleDirs, PROJEN_MODULE_ROOT]) {
        discoverJsii(dir);
        if (dir.includes('@') && fs.lstatSync(dir).isDirectory()) {
            const childDirs = fs.readdirSync(dir).map(file => path.join(dir, file));
            for (const child of childDirs) {
                discoverJsii(child);
            }
        }
    }
    const result = new Array();
    for (const fqn of Object.keys(jsii)) {
        const p = toProjectType(jsii, fqn);
        if (!p) {
            continue;
        }
        result.push(p);
    }
    return result.sort((r1, r2) => r1.pjid.localeCompare(r2.pjid));
}
exports.discover = discover;
function resolveProjectType(projectFqn) {
    const manifest = readJsiiManifest(projectFqn);
    const jsii = {};
    for (const [fqn, type] of Object.entries(manifest.types)) {
        jsii[fqn] = type;
    }
    // Read Projen JSII types
    const projenManifest = readJsiiManifest('projen');
    for (const [fqn, type] of Object.entries(projenManifest.types)) {
        jsii[fqn] = type;
    }
    return toProjectType(jsii, projectFqn);
}
exports.resolveProjectType = resolveProjectType;
function toProjectType(jsii, fqn) {
    var _a, _b, _c, _d;
    if (!isProjectType(jsii, fqn)) {
        return undefined;
    }
    const typeinfo = jsii[fqn];
    // projen.web.ReactProject -> web.ReactProject
    const typename = fqn.substring(fqn.indexOf('.') + 1);
    const docsurl = `https://github.com/projen/projen/blob/master/API.md#projen-${typename.toLocaleLowerCase()}`;
    let pjid = (_c = (_b = (_a = typeinfo.docs) === null || _a === void 0 ? void 0 : _a.custom) === null || _b === void 0 ? void 0 : _b.pjid) !== null && _c !== void 0 ? _c : decamelize(typename).replace(/_project$/, '');
    return {
        moduleName: typeinfo.assembly,
        typename,
        pjid,
        fqn,
        options: discoverOptions(jsii, fqn).sort((o1, o2) => o1.name.localeCompare(o2.name)),
        docs: (_d = typeinfo.docs) === null || _d === void 0 ? void 0 : _d.summary,
        docsurl,
    };
}
function readJsiiManifest(jsiiFqn) {
    let [moduleName] = jsiiFqn.split('.');
    if (moduleName === 'projen') {
        moduleName = PROJEN_MODULE_ROOT;
    }
    const jsiiManifestFile = require.resolve(`${moduleName}/.jsii`);
    return fs.readJsonSync(jsiiManifestFile);
}
exports.readJsiiManifest = readJsiiManifest;
function discoverOptions(jsii, fqn) {
    var _a, _b, _c, _d;
    const options = {};
    const params = (_c = (_b = (_a = jsii[fqn]) === null || _a === void 0 ? void 0 : _a.initializer) === null || _b === void 0 ? void 0 : _b.parameters) !== null && _c !== void 0 ? _c : [];
    const optionsParam = params[0];
    const optionsTypeFqn = (_d = optionsParam === null || optionsParam === void 0 ? void 0 : optionsParam.type) === null || _d === void 0 ? void 0 : _d.fqn;
    if (params.length > 1 || (params.length === 1 && (optionsParam === null || optionsParam === void 0 ? void 0 : optionsParam.name) !== 'options')) {
        throw new Error(`constructor for project ${fqn} must have a single "options" argument of a struct type. got ${JSON.stringify(params)}`);
    }
    addOptions(optionsTypeFqn);
    const opts = Object.values(options);
    return opts.sort((a, b) => a.switch.localeCompare(b.switch));
    function addOptions(ofqn, basePath = [], optional = false) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o;
        if (!ofqn) {
            return;
        }
        const struct = jsii[ofqn];
        if (!struct) {
            throw new Error(`unable to find options type ${ofqn} for project ${fqn}`);
        }
        for (const prop of (_a = struct.properties) !== null && _a !== void 0 ? _a : []) {
            const propPath = [...basePath, prop.name];
            // protect against double-booking
            if (prop.name in options) {
                throw new Error(`duplicate option "${prop.name}" in ${fqn} (already declared in ${options[prop.name].parent})`);
            }
            let typeName;
            let jsiiKind;
            if ((_b = prop.type) === null || _b === void 0 ? void 0 : _b.primitive) {
                typeName = (_c = prop.type) === null || _c === void 0 ? void 0 : _c.primitive; // e.g. 'string', 'boolean', 'number'
            }
            else if ((_d = prop.type) === null || _d === void 0 ? void 0 : _d.fqn) {
                typeName = (_e = prop.type) === null || _e === void 0 ? void 0 : _e.fqn.split('.').pop(); // projen.NodeProjectOptions -> NodeProjectOptions
                jsiiKind = jsii[(_f = prop.type) === null || _f === void 0 ? void 0 : _f.fqn].kind; // e.g. 'class', 'interface', 'enum'
            }
            else { // any other types such as collection types
                typeName = 'unknown';
            }
            const isOptional = optional || prop.optional;
            let defaultValue = (_g = prop.docs) === null || _g === void 0 ? void 0 : _g.default;
            if (defaultValue === 'undefined') {
                defaultValue = undefined;
            }
            // if this is a mandatory option and we have a default value, it has to be JSON-parsable to the correct type
            if (!isOptional && defaultValue) {
                if (!((_h = prop.type) === null || _h === void 0 ? void 0 : _h.primitive)) {
                    throw new Error(`required option "${prop.name}" with a @default must use primitive types (string, number or boolean). type found is: ${typeName}`);
                }
                checkDefaultIsParsable(prop.name, defaultValue, (_j = prop.type) === null || _j === void 0 ? void 0 : _j.primitive);
            }
            options[prop.name] = filterUndefined({
                path: propPath,
                parent: struct.name,
                name: prop.name,
                fqn: (_k = prop.type) === null || _k === void 0 ? void 0 : _k.fqn,
                docs: prop.docs.summary,
                type: typeName,
                kind: jsiiKind,
                switch: propPath.map(p => decamelize(p).replace(/_/g, '-')).join('-'),
                default: defaultValue,
                optional: isOptional,
                featured: ((_m = (_l = prop.docs) === null || _l === void 0 ? void 0 : _l.custom) === null || _m === void 0 ? void 0 : _m.featured) === 'true',
                deprecated: prop.docs.stability === 'deprecated' ? true : undefined,
            });
        }
        for (const ifc of (_o = struct.interfaces) !== null && _o !== void 0 ? _o : []) {
            addOptions(ifc);
        }
    }
}
function filterUndefined(obj) {
    const ret = {};
    for (const [k, v] of Object.entries(obj)) {
        if (v !== undefined) {
            ret[k] = v;
        }
    }
    return ret;
}
function isProjectType(jsii, fqn) {
    var _a;
    const type = jsii[fqn];
    if (type.kind !== 'class') {
        return false;
    }
    if (type.abstract) {
        return false;
    }
    if ((_a = type.docs) === null || _a === void 0 ? void 0 : _a.deprecated) {
        return false;
    }
    let curr = type;
    while (true) {
        if (curr.fqn === PROJECT_BASE_FQN) {
            return true;
        }
        if (!curr.base) {
            return false;
        }
        curr = jsii[curr.base];
        if (!curr) {
            return false;
        }
    }
}
function checkDefaultIsParsable(prop, value, type) {
    // macros are pass-through
    if (value.startsWith('$')) {
        return;
    }
    try {
        const parsed = JSON.parse(value);
        if (typeof (parsed) !== type) {
            throw new Error(`cannot parse @default value for mandatory option ${prop} as a ${type}: ${parsed}`);
        }
    }
    catch (e) {
        throw new Error(`unable to JSON.parse() value "${value}" specified as @default for mandatory option "${prop}": ${e.message}`);
    }
}
//# sourceMappingURL=data:application/json;base64,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