"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Eslint = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const common_1 = require("./common");
const component_1 = require("./component");
const json_1 = require("./json");
const tasks_1 = require("./tasks");
/**
 * @experimental
 */
class Eslint extends component_1.Component {
    /**
     * @experimental
     */
    constructor(project, options) {
        var _b, _c, _d, _e, _f, _g;
        super(project);
        project.addDevDeps('eslint', '@typescript-eslint/eslint-plugin', '@typescript-eslint/parser', 'eslint-import-resolver-node', 'eslint-import-resolver-typescript', 'eslint-plugin-import', 'json-schema');
        if (options.prettier) {
            project.addDevDeps('prettier', 'eslint-plugin-prettier', 'eslint-config-prettier');
        }
        const devdirs = (_b = options.devdirs) !== null && _b !== void 0 ? _b : [];
        const dirs = [...options.dirs, ...devdirs];
        const fileExtensions = (_c = options.fileExtensions) !== null && _c !== void 0 ? _c : ['.ts'];
        this._allowDevDeps = (devdirs !== null && devdirs !== void 0 ? devdirs : []).map(dir => `**/${dir}/**`);
        const lintProjenRc = (_d = options.lintProjenRc) !== null && _d !== void 0 ? _d : true;
        const eslint = project.addTask('eslint', {
            description: 'Runs eslint against the codebase',
            category: tasks_1.TaskCategory.TEST,
            exec: [
                'eslint',
                `--ext ${fileExtensions.join(',')}`,
                '--fix',
                '--no-error-on-unmatched-pattern',
                ...dirs,
                ...lintProjenRc ? [common_1.PROJEN_RC] : [],
            ].join(' '),
        });
        project.testTask.spawn(eslint);
        // exclude some files
        (_e = project.npmignore) === null || _e === void 0 ? void 0 : _e.exclude('/.eslintrc.json');
        const formattingRules = options.prettier ? {
            'prettier/prettier': ['error'],
        } : {
            // see https://github.com/typescript-eslint/typescript-eslint/blob/master/packages/eslint-plugin/docs/rules/indent.md
            'indent': ['off'],
            '@typescript-eslint/indent': ['error', 2],
            // Style
            'quotes': ['error', 'single', { avoidEscape: true }],
            'comma-dangle': ['error', 'always-multiline'],
            'comma-spacing': ['error', { before: false, after: true }],
            'no-multi-spaces': ['error', { ignoreEOLComments: false }],
            'array-bracket-spacing': ['error', 'never'],
            'array-bracket-newline': ['error', 'consistent'],
            'object-curly-spacing': ['error', 'always'],
            'object-curly-newline': ['error', { multiline: true, consistent: true }],
            'object-property-newline': ['error', { allowAllPropertiesOnSameLine: true }],
            'keyword-spacing': ['error'],
            'brace-style': ['error', '1tbs', { allowSingleLine: true }],
            'space-before-blocks': ['error'],
            'curly': ['error', 'multi-line', 'consistent'],
            '@typescript-eslint/member-delimiter-style': ['error'],
            // Require semicolons
            'semi': ['error', 'always'],
            // Max line lengths
            'max-len': ['error', {
                    code: 150,
                    ignoreUrls: true,
                    ignoreStrings: true,
                    ignoreTemplateLiterals: true,
                    ignoreComments: true,
                    ignoreRegExpLiterals: true,
                }],
            // Don't unnecessarily quote properties
            'quote-props': ['error', 'consistent-as-needed'],
        };
        this.rules = {
            ...formattingRules,
            // Require use of the `import { foo } from 'bar';` form instead of `import foo = require('bar');`
            '@typescript-eslint/no-require-imports': ['error'],
            // Require all imported dependencies are actually declared in package.json
            'import/no-extraneous-dependencies': [
                'error',
                {
                    // Only allow importing devDependencies from "devdirs".
                    devDependencies: () => this.renderDevDepsAllowList(),
                    optionalDependencies: false,
                    peerDependencies: true,
                },
            ],
            // Require all imported libraries actually resolve (!!required for import/no-extraneous-dependencies to work!!)
            'import/no-unresolved': ['error'],
            // Require an ordering on all imports
            'import/order': ['warn', {
                    groups: ['builtin', 'external'],
                    alphabetize: { order: 'asc', caseInsensitive: true },
                }],
            // Cannot import from the same module twice
            'no-duplicate-imports': ['error'],
            // Cannot shadow names
            'no-shadow': ['off'],
            '@typescript-eslint/no-shadow': ['error'],
            // Required spacing in property declarations (copied from TSLint, defaults are good)
            'key-spacing': ['error'],
            // No multiple empty lines
            'no-multiple-empty-lines': ['error'],
            // One of the easiest mistakes to make
            '@typescript-eslint/no-floating-promises': ['error'],
            // Make sure that inside try/catch blocks, promises are 'return await'ed
            // (must disable the base rule as it can report incorrect errors)
            'no-return-await': ['off'],
            '@typescript-eslint/return-await': ['error'],
            // Useless diff results
            'no-trailing-spaces': ['error'],
            // Must use foo.bar instead of foo['bar'] if possible
            'dot-notation': ['error'],
            // Are you sure | is not a typo for || ?
            'no-bitwise': ['error'],
            // Member ordering
            '@typescript-eslint/member-ordering': ['error', {
                    default: [
                        'public-static-field',
                        'public-static-method',
                        'protected-static-field',
                        'protected-static-method',
                        'private-static-field',
                        'private-static-method',
                        'field',
                        // Constructors
                        'constructor',
                        // Methods
                        'method',
                    ],
                }],
        };
        // Overrides for .projenrc.js
        this.overrides = [
            {
                files: [common_1.PROJEN_RC],
                rules: {
                    '@typescript-eslint/no-require-imports': 'off',
                    'import/no-extraneous-dependencies': 'off',
                },
            },
        ];
        this.ignorePatterns = (_f = options.ignorePatterns) !== null && _f !== void 0 ? _f : [
            '*.js',
            `!${common_1.PROJEN_RC}`,
            '*.d.ts',
            'node_modules/',
            '*.generated.ts',
            'coverage',
        ];
        const tsconfig = (_g = options.tsconfigPath) !== null && _g !== void 0 ? _g : './tsconfig.json';
        const plugins = [
            '@typescript-eslint',
            'import',
            ...(options.prettier ? ['prettier'] : []),
        ];
        const extendsConf = [
            'plugin:import/typescript',
            ...(options.prettier ? [
                'prettier',
                'plugin:prettier/recommended',
            ] : []),
        ];
        this.config = {
            env: {
                jest: true,
                node: true,
            },
            root: true,
            plugins,
            parser: '@typescript-eslint/parser',
            parserOptions: {
                ecmaVersion: 2018,
                sourceType: 'module',
                project: tsconfig,
            },
            extends: extendsConf,
            settings: {
                'import/parsers': {
                    '@typescript-eslint/parser': ['.ts', '.tsx'],
                },
                'import/resolver': {
                    node: {},
                    typescript: {
                        project: tsconfig,
                    },
                },
            },
            ignorePatterns: this.ignorePatterns,
            rules: this.rules,
            overrides: this.overrides,
        };
        new json_1.JsonFile(project, '.eslintrc.json', { obj: this.config, marker: false });
    }
    /**
     * (experimental) Add an eslint rule.
     *
     * @experimental
     */
    addRules(rules) {
        for (const [k, v] of Object.entries(rules)) {
            this.rules[k] = v;
        }
    }
    /**
     * (experimental) Add an eslint override.
     *
     * @experimental
     */
    addOverride(override) {
        this.overrides.push(override);
    }
    /**
     * (experimental) Do not lint these files.
     *
     * @experimental
     */
    addIgnorePattern(pattern) {
        this.ignorePatterns.push(pattern);
    }
    /**
     * (experimental) Add a glob file pattern which allows importing dev dependencies.
     *
     * @param pattern glob pattern.
     * @experimental
     */
    allowDevDeps(pattern) {
        this._allowDevDeps.push(pattern);
    }
    renderDevDepsAllowList() {
        return this._allowDevDeps;
    }
}
exports.Eslint = Eslint;
_a = JSII_RTTI_SYMBOL_1;
Eslint[_a] = { fqn: "projen.Eslint", version: "0.23.2" };
//# sourceMappingURL=data:application/json;base64,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