"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Dependencies = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const common_1 = require("../common");
const component_1 = require("../component");
const json_1 = require("../json");
/**
 * (experimental) The `Dependencies` component is responsible to track the list of dependencies a project has, and then used by project types as the model for rendering project-specific dependency manifests such as the dependencies section `package.json` files.
 *
 * To add a dependency you can use a project-type specific API such as
 * `nodeProject.addDeps()` or use the generic API of `project.deps`:
 *
 * @experimental
 */
class Dependencies extends component_1.Component {
    /**
     * (experimental) Adds a dependencies component to the project.
     *
     * @param project The parent project.
     * @experimental
     */
    constructor(project) {
        super(project);
        this._deps = new Array();
        // this is not really required at the moment, but actually quite useful as a
        // checked-in source of truth for dependencies and will potentially be
        // valuable in the future for CLI tools.
        new json_1.JsonFile(project, Dependencies.MANIFEST_FILE, {
            omitEmpty: true,
            obj: () => this.toJson(),
        });
    }
    /**
     * (experimental) Returns the coordinates of a dependency spec.
     *
     * Given `foo@^3.4.0` returns `{ name: "foo", version: "^3.4.0" }`.
     * Given `bar@npm:@bar/legacy` returns `{ name: "bar", version: "npm:@bar/legacy" }`.
     *
     * @experimental
     */
    static parseDependency(spec) {
        const scope = spec.startsWith('@');
        if (scope) {
            spec = spec.substr(1);
        }
        const [module, ...version] = spec.split('@');
        const name = scope ? `@${module}` : module;
        if (version.length == 0) {
            return { name };
        }
        else {
            return { name, version: version === null || version === void 0 ? void 0 : version.join('@') };
        }
    }
    /**
     * (experimental) A copy of all dependencies recorded for this project.
     *
     * The list is sorted by type->name->version
     *
     * @experimental
     */
    get all() {
        return [...this._deps].sort(compareDeps).map(normalizeDep);
    }
    /**
     * (experimental) Returns a dependency by name.
     *
     * Fails if there is no dependency defined by that name or if `type` is not
     * provided and there is more then one dependency type for this dependency.
     *
     * @param name The name of the dependency.
     * @param type The dependency type.
     * @returns a copy (cannot be modified)
     * @experimental
     */
    getDependency(name, type) {
        const idx = this.tryGetDependencyIndex(name, type);
        if (idx === -1) {
            const msg = type
                ? `there is no ${type} dependency defined on "${name}"`
                : `there is no dependency defined on "${name}"`;
            throw new Error(msg);
        }
        return {
            ...normalizeDep(this._deps[idx]),
        };
    }
    /**
     * (experimental) Adds a dependency to this project.
     *
     * @param spec The dependency spec in the format `MODULE[@VERSION]` where `MODULE` is the package-manager-specific module name and `VERSION` is an optional semantic version requirement (e.g. `^3.4.0`).
     * @param type The type of the dependency.
     * @experimental
     */
    addDependency(spec, type, metadata = {}) {
        this.project.logger.debug(`${type}-dep ${spec}`);
        const dep = {
            ...Dependencies.parseDependency(spec),
            type,
            metadata,
        };
        const existingDepIndex = this.tryGetDependencyIndex(dep.name, type);
        if (existingDepIndex !== -1) {
            const existingDep = this._deps[existingDepIndex];
            if (dep.version === existingDep.version || existingDep.version === undefined) {
                this.project.logger.debug(`updating existing ${dep.type}-dep ${dep.name} with more specific version/metadata`);
                this._deps[existingDepIndex] = dep;
            }
            else {
                throw new Error(`"${dep.name}" is already specified with different version: ${existingDep.version}`);
            }
        }
        else {
            this._deps.push(dep);
        }
        return dep;
    }
    /**
     * (experimental) Removes a dependency.
     *
     * @param name The name of the module to remove (without the version).
     * @param type The dependency type.
     * @experimental
     */
    removeDependency(name, type) {
        const removeIndex = this.tryGetDependencyIndex(name, type);
        if (removeIndex === -1) {
            return;
        }
        this._deps.splice(removeIndex, 1);
    }
    tryGetDependencyIndex(name, type) {
        const deps = this._deps.filter(d => d.name === name);
        if (deps.length === 0) {
            return -1; // not found
        }
        if (!type) {
            if (deps.length > 1) {
                throw new Error(`"${name}" is defined for multiple dependency types: ${deps.map(d => d.type).join(',')}. Please specify dependency type`);
            }
            type = deps[0].type;
        }
        return this._deps.findIndex(dep => dep.name === name && dep.type === type);
    }
    toJson() {
        if (this._deps.length === 0) {
            return undefined;
        }
        return {
            dependencies: this._deps.sort(compareDeps).map(normalizeDep),
        };
    }
}
exports.Dependencies = Dependencies;
_a = JSII_RTTI_SYMBOL_1;
Dependencies[_a] = { fqn: "projen.deps.Dependencies", version: "0.23.2" };
/**
 * (experimental) The project-relative path of the deps manifest file.
 *
 * @experimental
 */
Dependencies.MANIFEST_FILE = path.posix.join(common_1.PROJEN_DIR, 'deps.json');
function normalizeDep(d) {
    const obj = {};
    for (const [k, v] of Object.entries(d)) {
        if (v == undefined) {
            continue;
        }
        if (typeof (v) === 'object' && Object.keys(v).length === 0) {
            continue;
        }
        if (Array.isArray(v) && v.length === 0) {
            continue;
        }
        obj[k] = v;
    }
    return obj;
}
function compareDeps(d1, d2) {
    return specOf(d1).localeCompare(specOf(d2));
    function specOf(dep) {
        let spec = dep.type + ':' + dep.name;
        if (dep.version) {
            spec += '@' + dep.version;
        }
        return spec;
    }
}
//# sourceMappingURL=data:application/json;base64,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