"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const os = require("os");
const path = require("path");
const vm = require("vm");
const fs = require("fs-extra");
const inquirer = require("inquirer");
const yargs = require("yargs");
const inventory = require("../../inventory");
const render_options_1 = require("../../javascript/render-options");
const logging = require("../../logging");
const option_hints_1 = require("../../option-hints");
const util_1 = require("../../util");
const macros_1 = require("../macros");
class Command {
    constructor() {
        this.command = 'new [PROJECT-TYPE] [OPTIONS]';
        this.describe = 'Creates a new projen project';
    }
    builder(args) {
        var _a;
        args.positional('PROJECT-TYPE', { describe: 'optional only when --from is used and there is a single project type in the external module', type: 'string' });
        args.option('synth', { type: 'boolean', default: true, desc: 'Synthesize after creating .projenrc.js' });
        args.option('comments', { type: 'boolean', default: true, desc: 'Include commented out options in .projenrc.js (use --no-comments to disable)' });
        args.option('from', { type: 'string', alias: 'f', desc: 'External jsii npm module to create project from. Supports any package spec supported by yarn (such as "my-pack@^2.0")' });
        args.example('projen new awscdk-app-ts', 'Creates a new project of built-in type "awscdk-app-ts"');
        args.example('projen new --from projen-vue@^2', 'Creates a new project from an external module "projen-vue" with the specified version');
        for (const type of inventory.discover()) {
            args.command(type.pjid, (_a = type.docs) !== null && _a !== void 0 ? _a : '', {
                builder: cargs => {
                    var _a, _b, _c;
                    cargs.showHelpOnFail(true);
                    for (const option of (_a = type.options) !== null && _a !== void 0 ? _a : []) {
                        if (option.type !== 'string' && option.type !== 'number' && option.type !== 'boolean' && option.kind !== 'enum') {
                            continue; // we only support primitive and enum fields as command line options
                        }
                        let desc = [(_c = (_b = option.docs) === null || _b === void 0 ? void 0 : _b.replace(/\ *\.$/, '')) !== null && _c !== void 0 ? _c : ''];
                        const required = !option.optional;
                        let defaultValue;
                        if (option.default && option.default !== 'undefined') {
                            if (!required) {
                                // if the field is not required, just describe the default but don't actually assign a value
                                desc.push(`[default: ${option.default.replace(/^\ *-/, '').replace(/\.$/, '').trim()}]`);
                            }
                            else {
                                // if the field is required and we have a @default, then assign
                                // the value here so it appears in `--help`
                                defaultValue = renderDefault(option.default);
                            }
                        }
                        const argType = option.kind === 'enum' ? 'string' : option.type;
                        cargs.option(option.switch, {
                            group: required ? 'Required:' : 'Optional:',
                            type: argType,
                            description: desc.join(' '),
                            default: defaultValue,
                            required,
                        });
                    }
                    return cargs;
                },
                handler: argv => newProject(process.cwd(), type, argv),
            });
        }
        return args;
    }
    async handler(args) {
        // handle --from which means we want to first install a jsii module and then
        // create a project defined within this module.
        if (args.from) {
            return newProjectFromModule(process.cwd(), args.from, args);
        }
        // project type is defined but was not matched by yargs, so print the list of supported types
        if (args.projectType) {
            console.log(`Invalid project type ${args.projectType}. Supported types:`);
            for (const pjid of inventory.discover().map(x => x.pjid)) {
                console.log(`  ${pjid}`);
            }
            return;
        }
        // Handles the use case that nothing was specified since PROJECT-TYPE is now an optional positional parameter
        yargs.showHelp();
    }
}
/**
 * Creates a new project with defaults.
 *
 * This function creates the project type in-process (with in VM) and calls
 * `.synth()` on it (if `options.synth` is not `false`).
 *
 * At the moment, it also generates a `.projenrc.js` file with the same code
 * that was just executed. In the future, this will also be done by the project
 * type, so we can easily support multiple languages of projenrc.
 */
function createProject(opts) {
    // Default project resolution location
    let mod = '../../index';
    // External projects need to load the module from the modules directory
    if (opts.type.moduleName !== 'projen') {
        try {
            mod = path.dirname(require.resolve(path.join(opts.type.moduleName, 'package.json'), { paths: [process.cwd()] }));
        }
        catch (err) {
            throw new Error(`External project module '${opts.type.moduleName}' could not be resolved.`);
        }
    }
    // pass the FQN of the project type to the project initializer so it can
    // generate the projenrc file.
    const { renderedOptions } = render_options_1.renderJavaScriptOptions({
        bootstrap: true,
        comments: opts.comments,
        type: opts.type,
        args: opts.params,
    });
    const newProjectCode = `const project = new ${opts.type.typename}(${renderedOptions});`;
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const module = require(mod);
    const ctx = vm.createContext(module);
    process.env.PROJEN_DISABLE_POST = (!opts.post).toString();
    vm.runInContext([
        newProjectCode,
        opts.synth ? 'project.synth();' : '',
    ].join('\n'), ctx);
}
/**
 * Given a value from "@default", processes macros and returns a stringified
 * (quoted) result.
 *
 * @returns a javascript primitive (could be a string, number or boolean)
 */
function renderDefault(value) {
    var _a;
    return (_a = macros_1.tryProcessMacro(value)) !== null && _a !== void 0 ? _a : JSON.parse(value);
}
/**
 * Converts yargs command line switches to project type props.
 * @param type Project type
 * @param argv Command line switches
 */
function commandLineToProps(type, argv) {
    var _a;
    const props = {};
    // initialize props with default values
    for (const prop of type.options) {
        if (prop.default && prop.default !== 'undefined' && !prop.optional) {
            props[prop.name] = renderDefault(prop.default);
        }
    }
    for (const [arg, value] of Object.entries(argv)) {
        for (const prop of type.options) {
            if (prop.switch === arg) {
                let curr = props;
                const queue = [...prop.path];
                while (true) {
                    const p = queue.shift();
                    if (!p) {
                        break;
                    }
                    if (queue.length === 0) {
                        curr[p] = value;
                    }
                    else {
                        curr[p] = (_a = curr[p]) !== null && _a !== void 0 ? _a : {};
                        curr = curr[p];
                    }
                }
            }
        }
    }
    return props;
}
/**
 * Generates a new project from an external module.
 *
 * @param spec The name of the external module to load
 * @param args Command line arguments (incl. project type)
 */
async function newProjectFromModule(baseDir, spec, args) {
    var _a;
    const specDependencyInfo = yarnAdd(baseDir, spec);
    // collect projects by looking up all .jsii modules in `node_modules`.
    const modulesDir = path.join(baseDir, 'node_modules');
    const modules = fs.readdirSync(modulesDir).map(file => path.join(modulesDir, file));
    const projects = inventory
        .discover(...modules)
        .filter(x => x.moduleName !== 'projen'); // filter built-in project types
    if (projects.length < 1) {
        throw new Error(`No projects found after installing ${spec}. The module must export at least one class which extends projen.Project`);
    }
    const requested = args.projectType;
    const types = projects.map(p => p.pjid);
    // if user did not specify a project type but the module has more than one, we need them to tell us which one...
    if (!requested && projects.length > 1) {
        throw new Error(`Multiple projects found after installing ${spec}: ${types.join(',')}. Please specify a project name.\nExample: npx projen new --from ${spec} ${types[0]}`);
    }
    // if user did not specify a type (and we know we have only one), the select it. otherwise, search by pjid.
    const type = !requested ? projects[0] : projects.find(p => p.pjid === requested);
    if (!type) {
        throw new Error(`Project type ${requested} not found. Found ${types.join(',')}`);
    }
    for (const option of (_a = type.options) !== null && _a !== void 0 ? _a : []) {
        if (option.type !== 'string' && option.type !== 'number' && option.type !== 'boolean') {
            continue; // we don't support non-primitive fields as command line options
        }
        if (args[option.name] !== undefined) {
            continue; // do not overwrite passed arguments
        }
        if (option.default && option.default !== 'undefined') {
            if (!option.optional) {
                const defaultValue = renderDefault(option.default);
                args[option.name] = defaultValue;
                args[option.switch] = defaultValue;
            }
        }
    }
    // include a dev dependency for the external module
    await newProject(baseDir, type, args, {
        devDeps: [specDependencyInfo],
    });
}
/**
 * Generates a new project.
 * @param type Project type
 * @param args Command line arguments
 * @param additionalProps Additional parameters to include in .projenrc.js
 */
async function newProject(baseDir, type, args, additionalProps) {
    // convert command line arguments to project props using type information
    const props = commandLineToProps(type, args);
    // merge in additional props if specified
    for (const [k, v] of Object.entries(additionalProps !== null && additionalProps !== void 0 ? additionalProps : {})) {
        props[k] = v;
    }
    createProject({
        dir: baseDir,
        type,
        params: props,
        comments: args.comments ? option_hints_1.NewProjectOptionHints.FEATURED : option_hints_1.NewProjectOptionHints.NONE,
        synth: args.synth,
        post: args.post,
    });
    // interactive git and github setup
    const gitFolder = path.resolve(baseDir, '.git');
    let pushInitialToGithub = false;
    if (!fs.existsSync(gitFolder)) {
        pushInitialToGithub = await askAboutGit(baseDir);
    }
    if (pushInitialToGithub) {
        util_1.exec('git add .', { cwd: baseDir });
        util_1.exec('git commit -m \'Initial commit generated by projen\'', { cwd: baseDir });
        util_1.exec('git branch -M main', { cwd: baseDir });
        util_1.exec('git push --set-upstream origin main', { cwd: baseDir });
    }
}
/**
 * Installs the npm module (through `yarn add`) to node_modules under `projectDir`.
 * @param spec The npm package spec (e.g. foo@^1.2)
 * @returns String info for the project devDeps (e.g. foo@^1.2 or foo@/var/folders/8k/qcw0ls5pv_ph0000gn/T/projen-RYurCw/pkg.tgz)
 */
function yarnAdd(baseDir, spec) {
    const packageJsonPath = path.join(baseDir, 'package.json');
    const packageJsonExisted = fs.existsSync(packageJsonPath);
    let dependencyInfo = spec;
    // workaround: yarn fails to extract tgz if it contains '@' in the name, so we
    // create a temp copy called pkg.tgz and install from there.
    // see: https://github.com/yarnpkg/yarn/issues/6339
    if (spec.endsWith('.tgz') && spec.includes('@')) {
        // if user passes in a file spec then we have to specify the project name and the package location
        // (e.g foo@/var/folders/8k/qcw0ls5pv_ph0000gn/T/projen-RYurCw/pkg.tgz)
        const moduleName = spec.split('/').slice(-1)[0].trim().split('@')[0].trim(); // Example: ./cdk-project/dist/js/cdk-project@1.0.0.jsii.tgz
        const packageDir = fs.mkdtempSync(path.join(os.tmpdir(), 'projen-'));
        const copy = path.join(packageDir, 'pkg.tgz');
        fs.copyFileSync(spec, copy);
        spec = copy;
        dependencyInfo = `${moduleName}@${spec}`;
    }
    logging.info(`installing external module ${spec}...`);
    util_1.exec(`yarn add --modules-folder=${baseDir}/node_modules --silent --no-lockfile --dev ${spec}`, { cwd: baseDir });
    // if package.json did not exist before calling yarn add, we should remove it
    // so we can start off clean.
    if (!packageJsonExisted) {
        fs.removeSync(packageJsonPath);
    }
    return dependencyInfo;
}
/**
 * Returns the last path element for use as a repository name default.
 */
function repoName() {
    return path.basename(path.basename(process.cwd()));
}
async function askAboutGit(cwd) {
    logging.info('We notice that you do not have a local git repository.');
    const { setUpGit } = await inquirer.prompt([
        {
            type: 'confirm',
            name: 'setUpGit',
            message: 'Do you want to set that up now?',
        },
    ]);
    if (setUpGit) {
        const { plan } = await inquirer.prompt(githubPlanOptions);
        const { gh, git } = plan;
        if (!git && !gh) {
            util_1.exec('git init', { cwd });
            logging.info('Ok, we have run \'git init\' for you! Have a great day.');
        }
        if (git) {
            const { gitRepoURL } = await inquirer.prompt([
                {
                    type: 'input',
                    name: 'gitRepoURL',
                    message: 'What is the repo? (example: https://github.com/projen/projen)',
                },
            ]);
            util_1.exec('git init', { cwd });
            let formattedGitRepoURL = gitRepoURL;
            if (!gitRepoURL.includes('https')) {
                formattedGitRepoURL = `https://github.com/${gitRepoURL}`;
            }
            util_1.exec(`git remote add origin ${formattedGitRepoURL}`, { cwd });
            logging.info(`Great! We have run 'git init' for you and set the remote to ${formattedGitRepoURL}`);
        }
        if (!git && gh) {
            logging.info('Ok! We will make you a repository on GitHub.');
            const ghCLIPath = util_1.execOrUndefined(`${os.platform() === 'win32' ? 'where' : 'which'} gh`, { cwd });
            if (!ghCLIPath) {
                logging.warn('Looks like you do not have the GitHub CLI installed. Please go to https://cli.github.com/ to install and try again.');
            }
            else {
                const { gitProjectName } = await inquirer.prompt([
                    {
                        type: 'input',
                        name: 'gitProjectName',
                        message: 'What would you like to name it?',
                        default: repoName(),
                    },
                ]);
                logging.info(`Wow! ${gitProjectName} is such a great name!`);
                util_1.exec('git init', { cwd });
                util_1.exec(`gh repo create ${gitProjectName}`, { cwd });
                return true;
            }
        }
    }
    return false;
}
const githubPlanOptions = [
    {
        type: 'list',
        name: 'plan',
        message: 'We\'ll need some more info. Please choose one:',
        choices: [
            {
                value: {
                    git: true,
                },
                name: 'I already have a git repository',
            },
            {
                value: {
                    gh: true,
                    git: false,
                },
                name: 'I don\'t have a git repository and want to make one on GitHub',
            },
            {
                value: {
                    gh: false,
                    git: false,
                },
                name: 'I don\'t have a git repository and I don\'t want to use GitHub',
            },
        ],
    },
];
module.exports = new Command();
//# sourceMappingURL=data:application/json;base64,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