"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.sanitizeOutput = exports.directorySnapshot = exports.synthSnapshotWithPost = exports.synthSnapshot = exports.mkdtemp = exports.execProjenCLI = exports.TestProject = void 0;
const os = require("os");
const path = require("path");
const fs = require("fs-extra");
const glob_1 = require("glob");
const __1 = require("..");
const logging = require("../logging");
const util_1 = require("../util");
const PROJEN_CLI = require.resolve('../../bin/projen');
logging.disable(); // no logging during tests
class TestProject extends __1.Project {
    constructor(options = {}) {
        const tmpdir = mkdtemp();
        super({
            name: 'my-project',
            outdir: tmpdir,
            clobber: false,
            logging: {
                level: __1.LogLevel.OFF,
            },
            ...options,
        });
    }
    // override runTaskCommand in tests since the default includes the version
    // number and that will break regresion tests.
    runTaskCommand(task) {
        return `projen ${task.name}`;
    }
    postSynthesize() {
        fs.writeFileSync(path.join(this.outdir, '.postsynth'), '# postsynth');
    }
}
exports.TestProject = TestProject;
function execProjenCLI(workdir, args = []) {
    const command = [
        process.execPath,
        PROJEN_CLI,
        ...args,
    ];
    return util_1.exec(command.map(x => `"${x}"`).join(' '), { cwd: workdir, stdio: 'inherit' });
}
exports.execProjenCLI = execProjenCLI;
const autoRemove = new Set();
// Hook to automatically remove temporary directories without needing each
// place to actually handle this specifically.
afterAll((done) => {
    // Array.from used to get a copy, so we can safely remove from the set
    for (const dir of Array.from(autoRemove)) {
        try {
            // Note - fs-extra.removeSync is idempotent, so we're safe if the
            // directory has already been cleaned up before we get there!
            fs.removeSync(dir);
        }
        catch (e) {
            done.fail(e);
        }
        autoRemove.delete(dir);
    }
    done();
});
function mkdtemp() {
    const tmpdir = fs.mkdtempSync(path.join(os.tmpdir(), 'projen-test-'));
    autoRemove.add(tmpdir);
    return tmpdir;
}
exports.mkdtemp = mkdtemp;
function synthSnapshot(project) {
    const synthed = Symbol.for('synthed');
    if (synthed in project) {
        throw new Error('duplicate synth()');
    }
    project[synthed] = true;
    const ENV_PROJEN_DISABLE_POST = process.env.PROJEN_DISABLE_POST;
    try {
        process.env.PROJEN_DISABLE_POST = 'true';
        project.synth();
        return directorySnapshot(project.outdir);
    }
    finally {
        fs.removeSync(project.outdir);
        // values assigned to process.env.XYZ are automatically converted to strings
        if (ENV_PROJEN_DISABLE_POST === undefined) {
            delete process.env.PROJEN_DISABLE_POST;
        }
        else {
            process.env.PROJEN_DISABLE_POST = ENV_PROJEN_DISABLE_POST;
        }
    }
}
exports.synthSnapshot = synthSnapshot;
function synthSnapshotWithPost(project) {
    try {
        project.synth();
        return directorySnapshot(project.outdir);
    }
    finally {
        fs.removeSync(project.outdir);
    }
}
exports.synthSnapshotWithPost = synthSnapshotWithPost;
function directorySnapshot(root, options = {}) {
    var _a;
    const output = {};
    const files = glob_1.glob.sync('**', {
        ignore: (_a = options.excludeGlobs) !== null && _a !== void 0 ? _a : [],
        cwd: root,
        nodir: true,
        dot: true,
    }); // returns relative file paths with POSIX separators
    for (const file of files) {
        const filePath = path.join(root, file);
        let content;
        if (path.extname(filePath) === '.json') {
            content = fs.readJsonSync(filePath);
        }
        else {
            content = fs.readFileSync(filePath, 'utf-8');
        }
        output[file] = content;
    }
    return output;
}
exports.directorySnapshot = directorySnapshot;
/**
 * Removes any non-deterministic aspects from the synthesized output.
 * @param dir The output directory.
 */
function sanitizeOutput(dir) {
    const filepath = path.join(dir, 'package.json');
    const pkg = fs.readJsonSync(filepath);
    const prev = pkg.devDependencies.projen;
    if (!prev) {
        throw new Error(`expecting "${filepath}" to include a devDependency on "projen"`);
    }
    // replace the current projen version with 999.999.999 for deterministic output.
    // this will preserve any semantic version requirements (e.g. "^", "~", etc).
    pkg.devDependencies.projen = prev.replace(/\d+\.\d+\.\d+/, '999.999.999');
    fs.writeJsonSync(filepath, pkg);
    // we will also patch deps.json so that all projen deps will be set to 999.999.999
    const depsPath = path.join(dir, '.projen', 'deps.json');
    const deps = fs.readJsonSync(depsPath);
    for (const dep of deps.dependencies) {
        if (dep.name === 'projen') {
            dep.version = dep.version.replace(/\d+\.\d+\.\d+/, '999.999.999');
        }
    }
    fs.chmodSync(depsPath, '777');
    fs.writeJsonSync(depsPath, deps);
}
exports.sanitizeOutput = sanitizeOutput;
//# sourceMappingURL=data:application/json;base64,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