"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tasks_1 = require("../../tasks");
const util_1 = require("../util");
test('no tasks, no tasks file', () => {
    const p = new util_1.TestProject();
    expect(synthTasksManifest(p)).toBeUndefined();
});
test('empty task', () => {
    const p = new util_1.TestProject();
    // WHEN
    p.addTask('empty');
    // THEN
    expectManifest(p, {
        tasks: {
            empty: {
                name: 'empty',
            },
        },
    });
});
test('remove task', () => {
    const p = new util_1.TestProject();
    // WHEN
    const task = p.addTask('task1');
    p.addTask('task2');
    const removeTask = p.removeTask('task1');
    // THEN
    expect(removeTask).toEqual(task);
    expectManifest(p, {
        tasks: {
            task2: {
                name: 'task2',
            },
        },
    });
});
test('re-add removed task', () => {
    const p = new util_1.TestProject();
    // WHEN
    p.addTask('task1');
    p.addTask('task2');
    const removeTask = p.removeTask('task2');
    p.addTask('task2');
    // THEN
    expect(removeTask).toBeTruthy();
    expectManifest(p, {
        tasks: {
            task1: {
                name: 'task1',
            },
            task2: {
                name: 'task2',
            },
        },
    });
});
test('throw when removing a dependent task', () => {
    const p = new util_1.TestProject();
    // WHEN
    const primary = p.addTask('primary');
    const dependent = p.addTask('dependent');
    primary.spawn(dependent);
    // THEN
    expect(() => p.removeTask('dependent'))
        .toThrowError('Unable to remove task "dependent" because the following tasks depend on it: primary');
});
test('remove already removed task', () => {
    const p = new util_1.TestProject();
    expect(p.removeTask('task1')).toBe(undefined);
});
test('multiple "exec" commands', () => {
    const p = new util_1.TestProject();
    // WHEN
    const task = p.addTask('hello', {
        description: 'hello, world',
        exec: 'echo hello',
        env: {
            FOO: 'bar',
        },
    });
    task.exec('echo world');
    task.exec('echo "with quotes"');
    task.env('BAR', 'baz');
    // THEN
    expectManifest(p, {
        tasks: {
            hello: {
                name: 'hello',
                description: 'hello, world',
                env: {
                    FOO: 'bar',
                    BAR: 'baz',
                },
                steps: [
                    { exec: 'echo hello' },
                    { exec: 'echo world' },
                    { exec: 'echo "with quotes"' },
                ],
            },
        },
    });
});
test('subtasks', () => {
    // GIVEN
    const p = new util_1.TestProject();
    const hello = p.addTask('hello', { exec: 'echo hello' });
    const world = p.addTask('world');
    // WHEN
    world.exec('echo "running hello"');
    world.spawn(hello);
    // THEN
    expectManifest(p, {
        tasks: {
            hello: {
                name: 'hello',
                steps: [{ exec: 'echo hello' }],
            },
            world: {
                name: 'world',
                steps: [
                    { exec: 'echo "running hello"' },
                    { spawn: 'hello' },
                ],
            },
        },
    });
});
test('reset() can be used to reset task steps', () => {
    // GIVEN
    const p = new util_1.TestProject();
    const t0 = p.addTask('your-task');
    const t = p.addTask('my-task');
    t.exec('line1');
    t.spawn(t0);
    t.exec('line2');
    // WHEN
    t.reset('line3');
    t.exec('line4', { cwd: 'foo' });
    // THEN
    expectManifest(p, {
        tasks: {
            'your-task': {
                name: 'your-task',
            },
            'my-task': {
                name: 'my-task',
                steps: [
                    { exec: 'line3' },
                    { cwd: 'foo', exec: 'line4' },
                ],
            },
        },
    });
});
test('prependXXX() can be used to add steps from the top', () => {
    // GIVEN
    const p = new util_1.TestProject();
    const sub = p.addTask('my-sub-task', { exec: 'subexec' });
    const t = p.addTask('my-task');
    t.exec('line1');
    // WHEN
    t.prependExec('line2');
    t.prependSpawn(sub);
    t.prependSay('message');
    // THEN
    expectManifest(p, {
        tasks: {
            'my-sub-task': {
                name: 'my-sub-task',
                steps: [
                    { exec: 'subexec' },
                ],
            },
            'my-task': {
                name: 'my-task',
                steps: [
                    { say: 'message' },
                    { spawn: 'my-sub-task' },
                    { exec: 'line2' },
                    { exec: 'line1' },
                ],
            },
        },
    });
});
test('env() can be used to add environment variables', () => {
    // GIVEN
    const p = new util_1.TestProject();
    const t = p.addTask('my-task', {
        env: {
            INITIAL: '123',
            ENV: '456',
        },
    });
    // WHEN
    t.env('FOO', 'BAR');
    t.env('HELLO', 'world');
    // THEN
    expectManifest(p, {
        tasks: {
            'my-task': {
                name: 'my-task',
                env: {
                    INITIAL: '123',
                    ENV: '456',
                    FOO: 'BAR',
                    HELLO: 'world',
                },
            },
        },
    });
});
test('.steps can be used to list all steps in the current task', () => {
    // GIVEN
    const p = new util_1.TestProject();
    const t0 = p.addTask('your');
    const t = p.addTask('my');
    t.exec('step1');
    t.exec('step2');
    t.exec('step3');
    t.spawn(t0);
    t.exec('step4');
    // WHEN
    const steps = t.steps;
    // THEN
    expect(steps).toStrictEqual([
        { exec: 'step1' },
        { exec: 'step2' },
        { exec: 'step3' },
        { spawn: 'your' },
        { exec: 'step4' },
    ]);
});
test('"condition" can be used to define a command that will determine if a task should be skipped', () => {
    // GIVEN
    const p = new util_1.TestProject();
    p.addTask('foo', {
        condition: 'false',
        exec: 'foo bar',
    });
    // THEN
    expectManifest(p, {
        tasks: {
            foo: {
                name: 'foo',
                condition: 'false',
                steps: [
                    { exec: 'foo bar' },
                ],
            },
        },
    });
});
test('"builtin" can be used to execute builtin commands', () => {
    const p = new util_1.TestProject();
    const task = p.addTask('foo', {
        condition: 'false',
    });
    task.builtin('my/builtin');
    task.builtin('your/builtin');
    // THEN
    expectManifest(p, {
        tasks: {
            foo: {
                name: 'foo',
                condition: 'false',
                steps: [
                    { builtin: 'my/builtin' },
                    { builtin: 'your/builtin' },
                ],
            },
        },
    });
});
function expectManifest(p, toStrictEqual) {
    const manifest = synthTasksManifest(p);
    delete manifest['//'];
    expect(manifest).toStrictEqual(toStrictEqual);
}
function synthTasksManifest(p) {
    return util_1.synthSnapshot(p)[tasks_1.Tasks.MANIFEST_FILE];
    ;
}
//# sourceMappingURL=data:application/json;base64,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