"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const YAML = require("yaml");
const workflows_model_1 = require("../../github/workflows-model");
const release_1 = require("../../release");
const util_1 = require("../util");
test('minimal', () => {
    // GIVEN
    const project = new util_1.TestProject();
    const task = project.addTask('release');
    // WHEN
    new release_1.Release(project, {
        task: task,
        versionFile: 'version.json',
        branch: 'main',
    });
    const outdir = util_1.synthSnapshot(project);
    expect(outdir).toMatchSnapshot();
});
test('with major version filter', () => {
    // GIVEN
    const project = new util_1.TestProject();
    const task = project.addTask('release');
    // WHEN
    new release_1.Release(project, {
        task: task,
        versionFile: 'version.json',
        branch: '10.x',
        majorVersion: 10,
        releaseWorkflowName: 'release',
    });
    // THEN
    const outdir = util_1.synthSnapshot(project);
    expect(outdir['.github/workflows/release.yml']).toBeDefined();
    expect(outdir).toMatchSnapshot();
});
test('addBranch() can be used for additional release branches', () => {
    // GIVEN
    const project = new util_1.TestProject();
    const task = project.addTask('release');
    const release = new release_1.Release(project, {
        task: task,
        versionFile: 'version.json',
        branch: 'main',
        majorVersion: 1,
    });
    // WHEN
    release.addBranch('2.x', { majorVersion: 2 });
    release.addBranch('10.x', { majorVersion: 10 });
    // THEN
    const outdir = util_1.synthSnapshot(project);
    expect(outdir['.github/workflows/release.yml']).toBeDefined();
    expect(outdir['.github/workflows/release-2.x.yml']).toBeDefined();
    expect(outdir['.github/workflows/release-10.x.yml']).toBeDefined();
    expect(outdir).toMatchSnapshot();
});
test('if multiple branches are defined, the default branch requires a "majorVersion"', () => {
    // GIVEN
    const project = new util_1.TestProject();
    const task = project.addTask('release');
    const release = new release_1.Release(project, {
        task: task,
        versionFile: 'version.json',
        branch: 'main',
    });
    // WHEN
    const addBranch = () => release.addBranch('2.x', { majorVersion: 2 });
    // THEN
    expect(addBranch).toThrow(/you must specify \"majorVersion\" for the default branch when adding multiple release branches/);
});
test('publisher (defaults)', () => {
    // GIVEN
    const project = new util_1.TestProject();
    const task = project.addTask('release');
    const release = new release_1.Release(project, {
        task: task,
        versionFile: 'version.json',
        branch: 'main',
    });
    // WHEN
    release.publisher.publishToGo();
    release.publisher.publishToMaven();
    release.publisher.publishToNpm();
    release.publisher.publishToNuget();
    release.publisher.publishToPyPi();
    // THEN
    const outdir = util_1.synthSnapshot(project);
    expect(outdir['.github/workflows/release.yml']).toMatchSnapshot();
});
test('publishers are added as jobs to all release workflows', () => {
    // GIVEN
    const project = new util_1.TestProject();
    const task = project.addTask('release');
    const release = new release_1.Release(project, {
        task: task,
        versionFile: 'version.json',
        branch: 'main',
        majorVersion: 1,
    });
    // WHEN
    release.addBranch('2.x', { majorVersion: 2 });
    release.publisher.publishToNpm();
    // THEN
    const outdir = util_1.synthSnapshot(project);
    const wf1 = YAML.parse(outdir['.github/workflows/release.yml']);
    expect(wf1).toMatchObject({
        on: { push: { branches: ['main'] } },
        jobs: {
            release: {},
            release_npm: {},
        },
    });
    const wf2 = YAML.parse(outdir['.github/workflows/release-2.x.yml']);
    expect(wf2).toMatchObject({
        on: { push: { branches: ['2.x'] } },
        jobs: {
            release: {},
            release_npm: {},
        },
    });
});
test('addJobs() can be used to add arbitrary jobs to the release workflows', () => {
    // GIVEN
    const project = new util_1.TestProject();
    const task = project.addTask('release');
    const release = new release_1.Release(project, {
        task: task,
        versionFile: 'version.json',
        branch: 'main',
        majorVersion: 0,
    });
    release.addBranch('foo', { majorVersion: 4, workflowName: 'foo-workflow' });
    release.publisher.publishToPyPi();
    // WHEN
    release.addJobs({
        random_job: {
            runsOn: 'foo',
            permissions: {
                actions: workflows_model_1.JobPermission.NONE,
            },
            steps: [],
        },
    });
    // THEN
    const outdir = util_1.synthSnapshot(project);
    expect(outdir).toMatchSnapshot();
});
test('majorVersion can be 0', () => {
    // GIVEN
    const project = new util_1.TestProject();
    const task = project.addTask('release');
    // WHEN
    new release_1.Release(project, {
        task: task,
        versionFile: 'goo.json',
        branch: 'main',
        majorVersion: 0,
    });
    // THEN
    const outdir = util_1.synthSnapshot(project);
    expect(outdir['.github/workflows/release.yml']).toMatchSnapshot();
});
test('prerelease can be specified per branch', () => {
    // GIVEN
    const project = new util_1.TestProject();
    const task = project.addTask('release');
    // WHEN
    const release = new release_1.Release(project, {
        task: task,
        versionFile: 'goo.json',
        branch: 'main',
        majorVersion: 0,
    });
    release.addBranch('10.x', { majorVersion: 10, prerelease: 'pre' });
    // THEN
    const outdir = util_1.synthSnapshot(project);
    expect(outdir['.github/workflows/release.yml']).toMatchSnapshot();
    expect(outdir['.github/workflows/release.10.x.yml']).toMatchSnapshot();
});
test('releaseBranches can be use to define additional branches', () => {
    // GIVEN
    const project = new util_1.TestProject();
    const task = project.addTask('release');
    // WHEN
    new release_1.Release(project, {
        task: task,
        versionFile: 'goo.json',
        branch: 'main',
        majorVersion: 1,
        releaseBranches: {
            '3.x': { majorVersion: 3 },
            'next': { majorVersion: 4, prerelease: 'pre' },
        },
    });
    const outdir = util_1.synthSnapshot(project);
    expect(outdir).toMatchSnapshot();
});
test('releaseBranches as an array throws an error since type was changed', () => {
    // GIVEN
    const project = new util_1.TestProject();
    const task = project.addTask('release');
    // WHEN
    expect(() => new release_1.Release(project, {
        task: task,
        versionFile: 'goo.json',
        branch: 'main',
        majorVersion: 1,
        releaseBranches: ['10.x', '2.x'],
    })).toThrow(/\"releaseBranches\" is no longer an array. See type annotations/);
});
//# sourceMappingURL=data:application/json;base64,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