"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const child_process_1 = require("child_process");
const fs_1 = require("fs");
const os_1 = require("os");
const path_1 = require("path");
const fs_extra_1 = require("fs-extra");
const logging = require("../../logging");
const bump_version_1 = require("../../release/bump-version");
logging.disable();
jest.setTimeout(1000 * 60 * 5); // 5min
test('first release', async () => {
    const result = await testBump();
    expect(result.version).toStrictEqual('0.0.0');
    expect(result.changelog).toMatch(/.*## 0\.0\.0 \(\d{4}-\d{2}-\d{2}\).*/); // ## 0.0.0 (2021-01-01)
});
test('first release, with major', async () => {
    const result = await testBump({
        options: { majorVersion: 2 },
    });
    expect(result.version).toStrictEqual('2.0.0');
    expect(result.changelog).toMatch(/.*## 2\.0\.0 \(\d{4}-\d{2}-\d{2}\).*/); // ## 2.0.0 (2021-01-01)
});
test('first release, with new major', async () => {
    const result = await testBump({
        options: { majorVersion: 4 },
        commits: [
            { message: 'v1', tag: 'v1.2.3' },
            { message: 'commit2' },
        ],
    });
    expect(result.version).toStrictEqual('4.0.0');
    expect(result.changelog.includes('## [4.0.0]')).toBeTruthy();
});
test('first release, with prerelease', async () => {
    const result = await testBump({
        options: { prerelease: 'beta' },
    });
    expect(result.version).toStrictEqual('0.0.0-beta.0');
    expect(result.changelog.includes('## 0.0.0-beta.0')).toBeTruthy();
});
test('select latest', async () => {
    const result = await testBump({
        commits: [
            { message: 'first version', tag: 'v1.1.0' },
            { message: 'second version', tag: 'v1.2.0' },
            { message: 'fix: bug' },
            { message: 'fix: another bug' },
        ],
    });
    expect(result.version).toEqual('1.2.1');
    expect(result.changelog.includes('Bug Fixes')).toBeTruthy();
    expect(result.changelog.includes('another bug')).toBeTruthy();
    expect(result.changelog.includes('bug')).toBeTruthy();
});
test('select latest with major', async () => {
    const commits = [
        { message: 'first version', tag: 'v1.1.0' },
        { message: 'boom', tag: 'v10.21.0' },
        { message: 'second version', tag: 'v1.2.0' },
        { message: 'fix: bug' },
        { message: 'fix: another bug' },
    ];
    const result1 = await testBump({
        options: { majorVersion: 1 },
        commits: commits,
    });
    expect(result1.version).toEqual('1.2.1');
    const result10 = await testBump({
        options: { majorVersion: 10 },
        commits: commits,
    });
    expect(result10.version).toEqual('10.21.1');
});
test('bump fails due to crossing major version', async () => {
    const commits = [
        { message: 'first version', tag: 'v1.1.0' },
        { message: 'boom', tag: 'v10.21.0' },
        { message: 'second version', tag: 'v1.2.0' },
        { message: 'fix: another bug\n\nBREAKING CHANGE: cause major version bump' },
        { message: 'fix: bug' },
    ];
    const promise = testBump({
        options: { majorVersion: 1 },
        commits: commits,
    });
    await expect(promise).rejects.toThrow(/bump failed: this branch is configured to only publish v1 releases - bump resulted in 2.0.0/);
});
//----------------------------------------------------------------------------------------------------------------------------------
async function testBump(opts = {}) {
    var _a;
    const workdir = fs_1.mkdtempSync(path_1.join(os_1.tmpdir(), 'bump-test-'));
    const git = (cmd) => child_process_1.execSync(`git ${cmd}`, { cwd: workdir, stdio: 'inherit' });
    // init a git repository
    git('init');
    git('config user.email "you@example.com"');
    git('config user.name "Your Name"');
    git('config commit.gpgsign false');
    const commit = async (message) => {
        await fs_extra_1.writeFile(path_1.join(workdir, 'dummy.txt'), message);
        git('add .');
        git(`commit -m "${message}"`);
    };
    await commit('initial commit');
    for (const c of (_a = opts.commits) !== null && _a !== void 0 ? _a : []) {
        await commit(c.message);
        if (c.tag) {
            git(`tag ${c.tag}`);
        }
    }
    await bump_version_1.bump(workdir, {
        changelog: 'changelog.md',
        versionFile: 'version.json',
        ...opts.options,
    });
    return {
        version: (await fs_extra_1.readJson(path_1.join(workdir, 'version.json'))).version,
        changelog: (await fs_extra_1.readFile(path_1.join(workdir, 'changelog.md'), 'utf8')),
    };
}
//# sourceMappingURL=data:application/json;base64,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