"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
// tests for `projen new`: we run `projen new` for each supported project type
// and compare against a golden snapshot.
const child_process_1 = require("child_process");
const path_1 = require("path");
const fs_extra_1 = require("fs-extra");
const inventory = require("../inventory");
const util_1 = require("./util");
for (const type of inventory.discover()) {
    test(`projen new ${type.pjid}`, () => {
        const outdir = util_1.mkdtemp();
        try {
            const projectdir = createProjectDir(outdir);
            // execute `projen new PJID --no-synth` in the project directory
            util_1.execProjenCLI(projectdir, ['new', '--no-synth', type.pjid]);
            // compare generated .projenrc.js to the snapshot
            const actual = util_1.directorySnapshot(projectdir, {
                excludeGlobs: [
                    '.git/**',
                ],
            });
            expect(actual).toMatchSnapshot();
        }
        finally {
            fs_extra_1.removeSync(outdir);
        }
    });
}
test('post-synthesis option enabled', () => {
    const project = new util_1.TestProject();
    expect(util_1.synthSnapshotWithPost(project)['.postsynth']).toContain('postsynth');
});
test('post-synthesis option disabled', () => {
    const project = new util_1.TestProject();
    expect(util_1.synthSnapshot(project)['.postsynth']).toBeUndefined();
});
test('projen new --from external', () => {
    const outdir = util_1.mkdtemp();
    try {
        const projectdir = createProjectDir(outdir);
        // execute `projen new --from cdk-appsync-project` in the project directory
        util_1.execProjenCLI(projectdir, ['new', '--from', 'cdk-appsync-project@1.1.2', '--no-post']);
        // patch the projen version in package.json to match the current version
        // otherwise, every bump would need to update these snapshots.
        util_1.sanitizeOutput(projectdir);
        // compare generated .projenrc.js to the snapshot
        const actual = util_1.directorySnapshot(projectdir, {
            excludeGlobs: [
                '.git/**',
                '.github/**',
                'node_modules/**',
                'yarn.lock',
            ],
        });
        expect(actual).toMatchSnapshot();
        expect(actual['schema.graphql']).toBeDefined();
    }
    finally {
        fs_extra_1.removeSync(outdir);
    }
});
test('options are not overwritten when creating external projects', () => {
    const outdir = util_1.mkdtemp();
    try {
        const projectdir = createProjectDir(outdir);
        // execute `projen new --from cdk-appsync-project` in the project directory
        util_1.execProjenCLI(projectdir, ['new', '--from', 'cdk-appsync-project@1.1.2', '--no-synth', '--cdk-version', '1.63.0']);
        // compare generated .projenrc.js to the snapshot
        const actual = util_1.directorySnapshot(projectdir, {
            excludeGlobs: [
                '.git/**',
                '.github/**',
                'node_modules/**',
                'yarn.lock',
            ],
        });
        expect(actual['.projenrc.js']).toContain('cdkVersion: \'1.63.0\'');
    }
    finally {
        fs_extra_1.removeSync(outdir);
    }
});
test('projen new --no-comments', () => {
    const outdir = util_1.mkdtemp();
    try {
        const projectdir = createProjectDir(outdir);
        util_1.execProjenCLI(projectdir, ['new', 'node', '--no-comments', '--no-synth']);
        const projenrc = util_1.directorySnapshot(projectdir)['.projenrc.js'];
        expect(projenrc).toBeDefined();
        expect(projenrc).not.toMatch('//');
    }
    finally {
        fs_extra_1.removeSync(outdir);
    }
});
test('creating node project with enum-typed CLI arg', () => {
    const outdir = util_1.mkdtemp();
    try {
        const projectdir = createProjectDir(outdir);
        util_1.execProjenCLI(projectdir, ['new', 'node', '--package-manager', 'npm', '--no-synth']);
        const projenrc = util_1.directorySnapshot(projectdir)['.projenrc.js'];
        expect(projenrc).toMatchSnapshot();
    }
    finally {
        fs_extra_1.removeSync(outdir);
    }
});
test('creating python project with enum-typed CLI arg', () => {
    const outdir = util_1.mkdtemp();
    try {
        const projectdir = createProjectDir(outdir);
        util_1.execProjenCLI(projectdir, ['new', 'python', '--project-type', 'lib', '--projenrc-python', '--no-synth']);
        const projenrc = util_1.directorySnapshot(projectdir)['.projenrc.py'];
        expect(projenrc).toMatchSnapshot();
    }
    finally {
        fs_extra_1.removeSync(outdir);
    }
});
test('creating java project with enum-typed CLI arg', () => {
    const outdir = util_1.mkdtemp();
    try {
        const projectdir = createProjectDir(outdir);
        util_1.execProjenCLI(projectdir, ['new', 'java', '--project-type', 'lib', '--projenrc-java', '--no-synth']);
        const projenrc = util_1.directorySnapshot(projectdir)['src/test/java/projenrc.java'];
        expect(projenrc).toMatchSnapshot();
    }
    finally {
        fs_extra_1.removeSync(outdir);
    }
});
function createProjectDir(workdir) {
    // create project under "my-project" so that basedir is deterministic
    const projectdir = path_1.join(workdir, 'my-project');
    fs_extra_1.mkdirSync(projectdir);
    const git = (command) => child_process_1.execSync(`git ${command}`, { cwd: projectdir });
    git('init');
    git('remote add origin git@boom.com:foo/bar.git');
    git('config user.name "My User Name"');
    git('config user.email "my@user.email.com"');
    return projectdir;
}
//# sourceMappingURL=data:application/json;base64,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