"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const __1 = require("..");
const util_1 = require("./util");
describe('author', () => {
    test('authorEmail and authorAddress can be the same value', () => {
        const project = new TestJsiiProject({
            authorAddress: 'hello@hello.com',
            authorEmail: 'hello@hello.com',
            repositoryUrl: 'https://github.com/foo/bar.git',
            author: 'My Name',
            name: 'project',
            defaultReleaseBranch: 'master',
        });
        const pkgjson = util_1.synthSnapshot(project)['package.json'];
        expect(pkgjson.author).toStrictEqual({
            email: 'hello@hello.com',
            name: 'My Name',
            organization: false,
        });
    });
    test('authorUrl and authorAddress can be the same value', () => {
        const project = new TestJsiiProject({
            authorAddress: 'https://foo.bar',
            authorUrl: 'https://foo.bar',
            repositoryUrl: 'https://github.com/foo/bar.git',
            author: 'My Name',
            name: 'project',
            defaultReleaseBranch: 'master',
        });
        const pkgjson = util_1.synthSnapshot(project)['package.json'];
        expect(pkgjson.author).toStrictEqual({
            name: 'My Name',
            organization: false,
            url: 'https://foo.bar',
        });
    });
});
describe('maven repository options', () => {
    test('use maven central as repository', () => {
        const project = new TestJsiiProject({
            authorAddress: 'https://foo.bar',
            authorUrl: 'https://foo.bar',
            repositoryUrl: 'https://github.com/foo/bar.git',
            author: 'My Name',
            name: 'testproject',
            publishToMaven: {
                javaPackage: 'com.github.eladb.watchful',
                mavenGroupId: 'com.github.eladb',
                mavenArtifactId: 'cdk-watchful',
            },
            defaultReleaseBranch: 'master',
        });
        const workflow = util_1.synthSnapshot(project)['.github/workflows/release.yml'];
        expect(workflow).toContain('run: npx -p jsii-release@latest jsii-release-maven');
        expect(workflow).not.toContainEqual('MAVEN_SERVER_ID');
        expect(workflow).not.toContainEqual('MAVEN_REPOSITORY_URL');
    });
    test('use nexus repo new endpoint', () => {
        const project = new TestJsiiProject({
            authorAddress: 'https://foo.bar',
            authorUrl: 'https://foo.bar',
            repositoryUrl: 'https://github.com/foo/bar.git',
            author: 'My Name',
            name: 'testproject',
            publishToMaven: {
                javaPackage: 'com.github.eladb.watchful',
                mavenGroupId: 'com.github.eladb',
                mavenArtifactId: 'cdk-watchful',
                mavenEndpoint: 'https://s01.oss.sonatype.org',
            },
            defaultReleaseBranch: 'master',
        });
        const workflow = util_1.synthSnapshot(project)['.github/workflows/release.yml'];
        expect(workflow).toContain('run: npx -p jsii-release@latest jsii-release-maven');
        expect(workflow).toContain('MAVEN_ENDPOINT: https://s01.oss.sonatype.org');
        expect(workflow).not.toContainEqual('MAVEN_SERVER_ID');
        expect(workflow).not.toContainEqual('MAVEN_REPOSITORY_URL');
    });
    test('use github as repository', () => {
        const project = new TestJsiiProject({
            authorAddress: 'https://foo.bar',
            authorUrl: 'https://foo.bar',
            repositoryUrl: 'https://github.com/foo/bar.git',
            author: 'My Name',
            name: 'testproject',
            publishToMaven: {
                javaPackage: 'com.github.eladb.watchful',
                mavenGroupId: 'com.github.eladb',
                mavenArtifactId: 'cdk-watchful',
                mavenServerId: 'github',
                mavenRepositoryUrl: 'https://maven.pkg.github.com/eladb',
            },
            defaultReleaseBranch: 'master',
        });
        const workflow = util_1.synthSnapshot(project)['.github/workflows/release.yml'];
        expect(workflow).toContain('MAVEN_SERVER_ID: github');
        expect(workflow).toContain('MAVEN_REPOSITORY_URL: https://maven.pkg.github.com/eladb');
    });
});
describe('publish to go', () => {
    test('defaults', () => {
        const project = new TestJsiiProject({
            authorAddress: 'https://foo.bar',
            authorUrl: 'https://foo.bar',
            repositoryUrl: 'https://github.com/foo/bar.git',
            author: 'My Name',
            name: 'testproject',
            publishToGo: {
                moduleName: 'github.com/foo/bar',
            },
            defaultReleaseBranch: 'master',
        });
        const output = util_1.synthSnapshot(project);
        const targets = output['package.json'].jsii.targets;
        expect(targets).toStrictEqual({
            go: {
                moduleName: 'github.com/foo/bar',
            },
        });
        expect(output['.github/workflows/release.yml']).toMatchSnapshot();
    });
    test('release to npm undefined', () => {
        const project = new TestJsiiProject({
            authorAddress: 'https://foo.bar',
            authorUrl: 'https://foo.bar',
            repositoryUrl: 'https://github.com/foo/bar.git',
            author: 'My Name',
            name: 'testproject',
            defaultReleaseBranch: 'main',
        });
        const output = util_1.synthSnapshot(project);
        expect(output['.github/workflows/release.yml']).toContain('release_npm');
    });
    test('release to npm true', () => {
        const project = new TestJsiiProject({
            authorAddress: 'https://foo.bar',
            authorUrl: 'https://foo.bar',
            repositoryUrl: 'https://github.com/foo/bar.git',
            author: 'My Name',
            name: 'testproject',
            defaultReleaseBranch: 'main',
            releaseToNpm: true,
        });
        const output = util_1.synthSnapshot(project);
        expect(output['.github/workflows/release.yml']).toContain('release_npm');
    });
    test('release to npm false', () => {
        const project = new TestJsiiProject({
            authorAddress: 'https://foo.bar',
            authorUrl: 'https://foo.bar',
            repositoryUrl: 'https://github.com/foo/bar.git',
            author: 'My Name',
            name: 'testproject',
            defaultReleaseBranch: 'main',
            releaseToNpm: false,
        });
        const output = util_1.synthSnapshot(project);
        expect(output['.github/workflows/release.yml']).not.toContain('release_npm');
    });
    test('customizations', () => {
        const project = new TestJsiiProject({
            authorAddress: 'https://foo.bar',
            authorUrl: 'https://foo.bar',
            repositoryUrl: 'https://github.com/foo/bar.git',
            author: 'My Name',
            name: 'testproject',
            publishToGo: {
                moduleName: 'github.com/foo/bar',
                gitBranch: 'custom-branch',
                gitCommitMessage: 'custom commit message',
                gitUserEmail: 'custom@email.com',
                gitUserName: 'custom user',
                githubRepo: 'github.com/foo/bar',
                githubTokenSecret: 'CUSTOM_SECRET',
            },
            defaultReleaseBranch: 'master',
            excludeTypescript: ['src/**/test/*.ts', 'src/**/__tests__/*.ts'],
        });
        const output = util_1.synthSnapshot(project);
        expect(output['package.json'].jsii.targets.go).toStrictEqual({ moduleName: 'github.com/foo/bar' });
        expect(output['.github/workflows/release.yml']).toMatchSnapshot();
        expect(output['package.json'].jsii.excludeTypescript).toStrictEqual(['src/**/test/*.ts', 'src/**/__tests__/*.ts']);
    });
});
class TestJsiiProject extends __1.JsiiProject {
    constructor(options) {
        super({
            outdir: util_1.mkdtemp(),
            logging: {
                level: __1.LogLevel.OFF,
            },
            ...options,
        });
    }
}
//# sourceMappingURL=data:application/json;base64,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