"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path_1 = require("path");
const fs_extra_1 = require("fs-extra");
const deps_1 = require("../deps");
const util_1 = require("./util");
test('no dependencies, no manifest', () => {
    // GIVEN
    const p = new util_1.TestProject();
    // THEN
    expect(depsManifest(p)).toBeUndefined();
});
test('minimal case', () => {
    // GIVEN
    const p = new util_1.TestProject();
    // WHEN
    const dep = p.deps.addDependency('my-first-dep', deps_1.DependencyType.RUNTIME);
    // THEN
    expect(dep.name).toBe('my-first-dep');
    expect(dep.version).toBeUndefined();
    expect(dep.type).toBe(deps_1.DependencyType.RUNTIME);
    expect(depsManifest(p)).toMatchSnapshot();
});
test('with version requirement', () => {
    // GIVEN
    const p = new util_1.TestProject();
    // WHEN
    const dep = p.deps.addDependency('depy@^7', deps_1.DependencyType.PEER);
    // THEN
    expect(dep.name).toBe('depy');
    expect(dep.version).toBe('^7');
    expect(dep.type).toBe(deps_1.DependencyType.PEER);
    expect(depsManifest(p)).toMatchSnapshot();
});
test('with package alias', () => {
    // GIVEN
    const p = new util_1.TestProject();
    // WHEN
    const dep = p.deps.addDependency('bar@npm:@bar/legacy', deps_1.DependencyType.PEER);
    // THEN
    expect(dep.name).toBe('bar');
    expect(dep.version).toBe('npm:@bar/legacy');
    expect(dep.type).toBe(deps_1.DependencyType.PEER);
    expect(depsManifest(p)).toMatchSnapshot();
});
test('deps.all returns all the dependencies', () => {
    // GIVEN
    const p = new util_1.TestProject();
    // WHEN
    p.deps.addDependency('dep1', deps_1.DependencyType.DEVENV);
    p.deps.addDependency('dep2', deps_1.DependencyType.RUNTIME);
    p.deps.addDependency('dep3', deps_1.DependencyType.PEER);
    // THEN
    expect(p.deps.all).toMatchSnapshot();
});
test('duplicates are ignored', () => {
    // GIVEN
    const p = new util_1.TestProject();
    // WHEN
    p.deps.addDependency('depy', deps_1.DependencyType.PEER);
    p.deps.addDependency('depy', deps_1.DependencyType.PEER);
    // THEN
    expect(p.deps.getDependency('depy')).toBeDefined();
    expect(p.deps.all.length).toEqual(1);
    expect(depsManifest(p)).toMatchSnapshot();
});
test('can be overridden with more specific version', () => {
    // GIVEN
    const p = new util_1.TestProject();
    // WHEN
    p.deps.addDependency('depy', deps_1.DependencyType.PEER);
    p.deps.addDependency('depy@^9', deps_1.DependencyType.PEER);
    // THEN
    expect(p.deps.getDependency('depy').version).toEqual('^9');
    expect(depsManifest(p)).toMatchSnapshot();
});
test('cannot be overridden with different version', () => {
    // GIVEN
    const p = new util_1.TestProject();
    // WHEN
    p.deps.addDependency('depy@^7', deps_1.DependencyType.PEER);
    // THEN
    expect(() => p.deps.addDependency('depy@^9', deps_1.DependencyType.PEER)).toThrow(/\"depy\" is already specified with different version: \^7/);
});
describe('removeDependency()', () => {
    test('can be used to remove a dependency', () => {
        // GIVEN
        const p = new util_1.TestProject();
        p.deps.addDependency('mydep', deps_1.DependencyType.RUNTIME);
        // WHEN
        p.deps.removeDependency('mydep');
        // THEN
        expect(p.deps.all).toStrictEqual([]);
    });
    test('dep is defined for multiple types (e.g. dev + runtime)', () => {
        // GIVEN
        const p = new util_1.TestProject();
        p.deps.addDependency('mydep', deps_1.DependencyType.RUNTIME);
        p.deps.addDependency('mydep', deps_1.DependencyType.BUILD);
        // WHEN
        p.deps.removeDependency('mydep', deps_1.DependencyType.BUILD);
        // THEN
        expect(p.deps.all).toStrictEqual([
            {
                name: 'mydep',
                type: 'runtime',
            },
        ]);
    });
    test('fails if type is not provided and there are more then one', () => {
        // GIVEN
        const p = new util_1.TestProject();
        // WHEN
        p.deps.addDependency('foo', deps_1.DependencyType.BUILD);
        p.deps.addDependency('foo', deps_1.DependencyType.RUNTIME);
        // THEN
        expect(() => p.deps.removeDependency('foo')).toThrow(/"foo" is defined for multiple dependency types\: build\,runtime/);
    });
    test('no-op if the dependency is not defined', () => {
        // GIVEN
        const p = new util_1.TestProject();
        p.deps.addDependency('hey', deps_1.DependencyType.RUNTIME);
        // WHEN
        p.deps.removeDependency('bam');
        // THEN
        expect(p.deps.all).toStrictEqual([
            {
                name: 'hey',
                type: 'runtime',
            },
        ]);
    });
});
describe('getDependency()', () => {
    test('returns a single dependency', () => {
        // GIVEN
        const p = new util_1.TestProject();
        // WHEN
        p.deps.addDependency('bar@^1.1.1', deps_1.DependencyType.RUNTIME);
        // THEN
        expect(p.deps.getDependency('bar')).toStrictEqual({
            name: 'bar',
            version: '^1.1.1',
            type: deps_1.DependencyType.RUNTIME,
        });
    });
    test('matches type', () => {
        // GIVEN
        const p = new util_1.TestProject();
        p.deps.addDependency('boo', deps_1.DependencyType.BUILD);
        p.deps.addDependency('zar', deps_1.DependencyType.RUNTIME);
        p.deps.addDependency('hey@^1.0.0', deps_1.DependencyType.RUNTIME);
        // WHEN
        p.deps.addDependency('bar@^1.1.1', deps_1.DependencyType.RUNTIME);
        // THEN
        expect(p.deps.getDependency('bar', deps_1.DependencyType.RUNTIME)).toStrictEqual({
            name: 'bar',
            version: '^1.1.1',
            type: deps_1.DependencyType.RUNTIME,
        });
    });
    test('matches type (multiple)', () => {
        // GIVEN
        const p = new util_1.TestProject();
        // WHEN
        p.deps.addDependency('bar@^1.2.1', deps_1.DependencyType.RUNTIME);
        p.deps.addDependency('bar@^1.1.1', deps_1.DependencyType.BUILD);
        // THEN
        expect(p.deps.getDependency('bar', deps_1.DependencyType.RUNTIME)).toStrictEqual({
            name: 'bar',
            version: '^1.2.1',
            type: deps_1.DependencyType.RUNTIME,
        });
    });
    test('fails if there is no dependency by that name', () => {
        // GIVEN
        const p = new util_1.TestProject();
        p.deps.addDependency('bar@1.1.1', deps_1.DependencyType.RUNTIME);
        // THEN
        expect(() => p.deps.getDependency('test')).toThrow(/there is no dependency defined on \"test\"/);
    });
    test('fails if there is more then one type and type is not provided', () => {
        // GIVEN
        const p = new util_1.TestProject();
        // WHEN
        p.deps.addDependency('zoo', deps_1.DependencyType.RUNTIME);
        p.deps.addDependency('zoo', deps_1.DependencyType.DEVENV);
        // THEN
        expect(() => p.deps.getDependency('zoo')).toThrow(/\"zoo\" is defined for multiple dependency types: runtime,devenv. Please specify dependency type/);
    });
    test('fails if type does not match', () => {
        // GIVEN
        const p = new util_1.TestProject();
        // WHEN
        p.deps.addDependency('zoo', deps_1.DependencyType.RUNTIME);
        // THEN
        expect(() => p.deps.getDependency('zoo', deps_1.DependencyType.BUILD)).toThrow(/there is no build dependency defined on \"zoo\"/);
    });
});
function depsManifest(p) {
    p.synth();
    const filepath = path_1.join(p.outdir, deps_1.Dependencies.MANIFEST_FILE);
    if (!fs_extra_1.existsSync(filepath)) {
        return undefined;
    }
    return JSON.parse(fs_extra_1.readFileSync(filepath, 'utf-8'));
}
//# sourceMappingURL=data:application/json;base64,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