from .base import BaseMapz

from typing import (
    Any,
    Dict,
    Hashable,
    Mapping,
    MutableMapping,
    List,
    Sequence,
    Union,
    Callable,
    Tuple,
)


# [not-sequence-types]
# These python types are not considered to be Sequences in ilexconf,
# even though, technically, they are Sequences in Python.
STR_TYPES = (str, bytes, bytearray)
# [not-sequence-types]


def issequence(arg: Any) -> bool:
    return isinstance(arg, Sequence) and not isinstance(arg, STR_TYPES)


def iskwresult(result: Any) -> bool:
    return (
        result is not None and isinstance(result, tuple) and len(result) == 2
    )


def isinsresult(result: Any) -> bool:
    return result is not None


def traverse(
    arg: Any,
    func=lambda *args, **kwargs: args,
    key_order=lambda k: k,
    list_order=lambda l: l,
    **kwargs,
) -> Any:
    if "_depth" not in kwargs:
        kwargs["_depth"] = 0
    kwargs["_depth"] += 1

    result = None

    if isinstance(arg, Mapping):
        # This branch always returns ``dic``
        d = dict()

        keys = key_order(arg.keys())
        for k in keys:
            v = arg[k]

            # At this point, ``func``` can trnsform both ``k`` and ``v``
            # to anything, even to None. Or turn ``v`` into a plain value.
            result = func(k, v, **kwargs)

            if iskwresult(result):
                k, v = result

            v = traverse(
                v,
                func,
                key_order=key_order,
                list_order=list_order,
                **kwargs,
            )

            dict.__setitem__(d, k, v)

        return d

    elif issequence(arg):
        # This branch always returns sequence of the same type as ``arg``.
        l = list()

        items = list_order(arg)
        for i in items:

            result = func(i, **kwargs)

            if isinsresult(result):
                i = result

            v = traverse(
                i, func, key_order=key_order, list_order=list_order, **kwargs
            )

            l.append(v)

        t = type(arg)
        return t(l)

    else:
        # This branch returns whatever results we get from ``func`` or
        # ``arg`` itself if there were no results.
        result = func(arg, **kwargs)

        if isinsresult(result):
            return result
        else:
            return arg


def update(
    dest: MutableMapping, data: Mapping, method: str = "recursive"
) -> BaseMapz:
    for key in data:
        if (
            method == "recursive"
            and key in dest
            and isinstance(dict.__getitem__(dest, key), MutableMapping)
            and isinstance(dict.__getitem__(data, key), Mapping)
        ):
            update(
                dict.__getitem__(dest, key),
                dict.__getitem__(data, key),
                method=method,
            )

        else:
            dict.__setitem__(dest, key, dict.__getitem__(data, key))
    return dest


def getsert(data: MutableMapping, key: Hashable, default: Any = None) -> Any:
    """Implements defaultdict feature."""

    if key not in data:
        # Do not insert default value into data if it's None.
        if default is None:
            return default
        dict.__setitem__(data, key, default)
    return dict.__getitem__(data, key)


def get(
    data: Union[MutableMapping, List],
    address: Hashable,
    sep: str = ".",
    default: Any = None,
) -> Any:

    # If key is a string, then try to split it if it contains the separator
    head = address
    tail = None
    if isinstance(address, str) and sep and sep in address:
        # "my.0.key"
        head, tail = address.split(sep, maxsplit=1)

    if isinstance(data, MutableMapping):
        if address in data:
            return dict.__getitem__(data, address)

        v = getsert(data, head, default=default)

        # if tail and isinstance(v, MutableMapping):
        if tail and isinstance(v, (MutableMapping, List)):
            return get(v, tail, sep=sep, default=default)

        # Return whatever got generated by "getsert" otherwise
        return v

    if isinstance(data, List):
        idx = None

        # If key is string
        if isinstance(head, str):
            try:
                idx = int(head)
            except ValueError:
                return None
        elif isinstance(head, int):
            idx = head
        else:
            return None

        if idx >= len(data) or idx < 0 and abs(idx) > len(data):
            return None

        if tail and isinstance(data[idx], (MutableMapping, List)):
            return get(data[idx], tail, sep=sep, default=default)

        return data[idx]

    # ?
    return default


def set(
    self,
    key: Hashable,
    value: Any,
    method: str = "recursive",
    sep: str = ".",
    inverse: bool = False,
) -> Any:
    parsed = from_keyvalue(key, value, sep=sep)

    if inverse:
        parsed.update(self, method=method)
        self.clear()

    self.update(parsed, method=method)

    return self


def map(
    self,
    func: Callable[
        [Hashable, Any], Tuple[Hashable, Any]
    ] = lambda k, v, **kwargs: (k, v),
    inplace: bool = False,
    **kwargs,
):
    config = traverse(self, func, **kwargs)

    if inplace:
        self.clear()
        self.update(config)
        config = self

    return config


def copy(self):
    """Return deep copy of the Config object."""
    return self.map(inplace=False)


def merge(
    self,
    *mappings: Mapping[Hashable, Any],
    _method: str = "recursive",
    _sep: str = "__",
    _inverse: bool = False,
    **kwargs,
) -> BaseMapz:
    # Collect all key-value pairs from mappings and keyword arguments
    # into a single ordered list with last element having the highest
    # priority.
    items = []
    # From mappings
    for mapping in mappings:
        items += mapping.items()
    # From keyword arguments
    items += kwargs.items()

    for k, v in items:
        self.set(k, v, method=_method, sep=_sep, inverse=_inverse)

    return self


def submerge(
    self,
    *mappings: Mapping[Hashable, Any],
    _method: str = "recursive",
    _sep: str = "__",
    **kwargs,
) -> BaseMapz:
    return self.merge(
        *mappings,
        _method=_method,
        _sep=_sep,
        _inverse=True,
        **kwargs,
    )


def from_keyvalue(
    key: Hashable,
    value: Any,
    prefix: str = "",
    sep: str = "__",
    lowercase: bool = False,
    uppercase: bool = False,
) -> BaseMapz:
    parts = parse_key(
        key,
        prefix=prefix,
        sep=sep,
        lowercase=lowercase,
        uppercase=uppercase,
    )
    if not parts:
        return BaseMapz()

    # value = Config.parse_value(value)
    value = traverse(value)

    # Fill in a hierarchical structure by
    # continuously building up the config in reverse order.
    result = value
    while parts:

        # Take the last part of the key no processed yet
        k = parts.pop()

        # Create an empty config and assign current saved ``result``
        # to ``k`` in it.
        config = BaseMapz()
        dict.__setitem__(config, k, result)
        # config = Config().set(k, result)
        # config[k] = result

        # Rebind result to point to the newly created config
        result = config

    return result


def parse_key(
    key: Hashable,
    prefix: str = "",
    sep: str = "__",
    lowercase: bool = False,
    uppercase: bool = False,
) -> List[Hashable]:
    if not isinstance(key, str):
        # When key is not a string, then it cannot be split.
        # Thus, return the key as is
        return [key]

    if not isinstance(prefix, str):
        prefix = str(prefix)

    if prefix and not key.startswith(prefix):
        # If prefix is specified, then return nothing
        return []

    # Strip key off of prefix
    key = key[len(prefix) :]

    # Convert to lowercase/uppercase if needed
    key, prefix, sep = [
        v.lower() if lowercase else v.upper() if uppercase else v
        for v in [key, prefix, sep]
    ]

    # Strip any dangling separator leftovers around the key
    if sep and key.startswith(sep):
        key = key[len(sep) :]
    if sep and key.endswith(sep):
        key = key[: -len(sep)]

    # Split the key into 2 parts using the separator.
    # If the key does not contain a separator string in it, then just return a parts
    # list consisting of the key itself.
    parts = key.split(sep, maxsplit=1) if sep else [key]

    if len(parts) > 1:
        # When key has been split successfully, then the second part of the split
        # is eligible for the same processing routine and a recursive call is made.
        key, subkey = parts  # unpack split parts for readability
        return [key] + parse_key(
            subkey,
            prefix="",
            sep=sep,
            lowercase=lowercase,
            uppercase=uppercase,
        )

    else:
        # If key was not split, then there is nothing to split anymore and we just
        # return the key
        return [parts[0]]
