"""Define common functions shared by CLI and REPL."""
import atexit
import json
import logging
import os
import sys
from argparse import Namespace, _StoreAction
from pathlib import Path
from types import ModuleType
from typing import Callable, Literal

from .. import APPDIRS, CONFIG_FILE, CONFIG_PATH, HISTORY_FILE, HISTORY_PATH
from ..__main__ import get_parser
from ..config import Configuration
from ..external import readline
from ..external.rich import traceback
from ..external.rich.logging import RichHandler
from ..translate import translate
from ..translators import TRANSLATORS, get_dummy
from ..utils.speakers import get_speaker
from ..utils.youdaozhiyun import get_youdaozhiyun_app_info

traceback.install()
logging.basicConfig(
    format="%(message)s",
    handlers=[RichHandler(rich_tracebacks=True, markup=True)],
)

logger = logging.getLogger(__name__)


def main(args: Namespace) -> None:
    """Call CLI/REPL (containing GUI).

    :param args:
    :type args: Namespace
    :rtype: None
    """
    if args.print_setting != "":
        from translate_shell.utils.setting import print_setting

        exit(print_setting(args.print_setting))

    try:
        import vim  # type: ignore
    except ImportError:
        if not sys.stdin.isatty():
            args.text = [sys.stdin.read()] + args.text
    if args.text:
        from translate_shell.ui.cli import run
    else:
        from translate_shell.ui.repl import run
    run(args)


def init_readline() -> ModuleType:
    """Init readline.

    :rtype: ModuleType
    """
    APPDIRS.user_data_path.mkdir(parents=True, exist_ok=True)
    HISTORY_PATH.touch(exist_ok=True)
    atexit.register(readline.write_history_file, HISTORY_FILE)  # type: ignore
    readline.read_history_file(HISTORY_FILE)
    return readline


def init_config(
    path: Path, mode: Literal["cli", "gui", "repl"]
) -> Configuration:
    """Read config file.

    :param path:
    :type path: Path
    :param mode:
    :type mode: Literal["cli", "gui", "repl"]
    :rtype: Configuration
    """
    config = Configuration(mode)
    try:
        configure_code = path.read_text()
    except FileNotFoundError:
        logger.info(CONFIG_FILE + "is not found!")
        return config
    namespace = {}
    try:
        exec(configure_code, namespace, namespace)  # nosec: B102
    except Exception as e:
        logger.error(e)
        logger.warning("Ignore " + CONFIG_FILE)
        return config
    configure = namespace.get("configure")
    if not isinstance(configure, Callable):
        return config
    try:
        new_config = configure(mode)
    except Exception as e:
        logger.error(e)
        logger.warning("Ignore configuration of " + CONFIG_FILE)
        return config
    if not isinstance(new_config, Configuration):
        logger.error("configuration of " + CONFIG_FILE + "is not legal!")
        return config
    return new_config


def init(args: Namespace) -> Namespace:
    """Init args.

    Because ``langdetect`` and many online translators use ``zh-cn`` not
    ``zh_CN``, we need preprocess. We must ignore ``--help``, ``--version``,
    and other arguments which cannot be customized.

    :param args:
    :type args: Namespace
    """
    if args.config:
        config_path = Path(args.config)
    else:
        config_path = CONFIG_PATH
    if args.text:
        mode = "cli"
    elif args.gui:
        mode = "gui"
    else:
        mode = "repl"
    config = init_config(config_path, mode)
    for action in get_parser()._get_optional_actions():
        if (
            not isinstance(action, _StoreAction)
            or getattr(args, action.dest) != action.default
        ):
            continue
        value = getattr(config, action.dest, None)
        if isinstance(value, str):
            setattr(args, action.dest, value)
    for attr in config.__all__:
        value = getattr(config, attr, None)
        if value is not None:
            setattr(args, attr, value)
    args.text = " ".join(args.text)
    readline = init_readline()
    if args.text:
        readline.add_history(args.text)
    args.last_text = ""
    logging.root.level += 10 * (args.quiet - args.verbose)

    global get_speaker, get_youdaozhiyun_app_info
    get_speaker = args.get_speaker
    get_youdaozhiyun_app_info = args.get_youdaozhiyun_app_info
    return args


def process(args: Namespace, is_repl: bool = False) -> tuple[str, str]:
    """process.

    :param args:
    :type args: Namespace
    :param is_repl: If the input is REPL's stdin, it is ``True``.
    :type is_repl: bool
    :rtype: tuple[str, str]
    """
    (
        text,
        args.target_lang,
        args.source_lang,
        args.translators,
    ) = args.process_input(
        args.text,
        args.target_lang,
        args.source_lang,
        args.translators,
        is_repl,
    )
    if text == "" or (not is_repl and text == args.last_text):
        return text, ""
    target_lang = args.target_lang
    if target_lang == "auto":
        target_lang = os.getenv("LANG", "zh_CN.UTF-8").split(".")[0]
        if target_lang not in ["zh_CN", "zh_TW"]:
            target_lang = target_lang.split("_")[0]
    target_lang = target_lang.lower().replace("_", "-")
    source_lang = args.source_lang.lower().replace("_", "-")
    translator_names = filter(
        len, map(lambda x: x.strip(), args.translators.split(","))
    )
    translators = [
        TRANSLATORS.get(translator, get_dummy(translator))
        for translator in translator_names
    ]
    translation = translate(
        text,
        target_lang,
        source_lang,
        translators,
    )
    if args.format == "json":
        rst = json.dumps(vars(translation))
    elif args.format == "yaml":
        from ..external import yaml

        rst = yaml.dump(vars(translation))
    else:
        rst = args.process_output(translation)
    return text, rst
