#!/usr/bin/env python
# -*- coding: utf-8 -*

import numpy as np
from sclmd.units import *


class siesta:
    """
    construct necessory information to do a siesta force run
    assuming all the fdf files are available

    22Jan2022:
    add the possibility of using socket AND pipe to 
    communicate between client and server, controlled by:
    comm_type: pipes, socket
    NOTE: Your need to import the correct library
    """

    def __init__(self, label, xyz, cell, mesh=100., dmtol=0.001,
                 constraints=[], tdir="./", lunit="Ang", eunit="eV",
                 comm_type="socket"):
        self.md2ang = 0.06466
        self.meshcutoff = mesh
        self.dmtol = dmtol
        self.constraints = constraints
        self.label = label
        # 1D array gives the coordinates of each atom
        # [xi,yi,zi],i=1,na
        self.xyz = np.array([a[1:] for a in xyz], dtype='d').flatten()
        self.els = [a[0] for a in xyz]
        self.conv = self.md2ang*np.array([3*[1.0/np.sqrt(AtomicMassTable[a])]
                                          for a in self.els]).flatten()
        # 1D array gives the unit cell
        # x1,y1,z1,x2,y2,z2,x3,y3,z3
        self.cell = np.array(cell, dtype='d').flatten()
        self.lunit = lunit
        self.eunit = eunit
        self.genfdf(tdir, comm_type)
        if comm_type is "socket":
            try:
                import sclmd.pysiesta_socket as ps
            except ImportError:
                print("could not import pysiesta_socket")
        elif comm_type is "pipes":
            try:
                import sclmd.pysiesta_pipe as ps
            except ImportError:
                print("could not import pysiesta_pipe")
        else:
            raise ValueError("comm_type must be 'socket' or 'pipes'")

        self.ps = ps

    def genfdf(self, tdir="./", comm_type="pipe"):
        """
        generate the fdf files.
        It includes extra fdf files:
            STRUCT.fdf
            Default.fdf
        """
        fname = self.label+".fdf"
        fn = open(fname, "w")

        fn.write("#fdf generated by siesta:genfdf\n")
        fn.write("SystemName   "+self.label+"\n")
        fn.write("SystemLabel   "+self.label+"\n")
        fn.write("MD.TypeOfRUN   forces\n")

        if comm_type is "socket":
            fn.write("Master.code    fsiesta\n")
            fn.write("Master.interface    socket\n")
            fn.write("Master.address    localhost\n")
            fn.write("Master.port    10001\n")
            fn.write("Master.socketType    inet\n")

        fn.write("MeshCutoff    "+str(self.meshcutoff)+" Ry\n")
        fn.write("DM.Tolerance  "+str(self.dmtol)+"\n\n\n")
        for i in range(len(self.constraints)):
            if i == 0:
                fn.write("%block GeometryConstraints\n")
            fn.write("position from "+str(self.constraints[i][0])+" to\
                     "+str(self.constraints[i][1])+"\n")
            if i == len(self.constraints)-1:
                fn.write("%endblock GeometryConstraints\n")
        fn.write("%include STRUCT.fdf\n")
        #fn.write("%include "+tdir+"MD.fdf\n")
        fn.write("%include "+tdir+"Default.fdf\n")
        fn.close()

    def start(self, npc=1):
        """
        start siesta
        npc - number of cores
        """
        # set units to : Angstrom, eV
        self.ps.pysiestaunits("Ang", "eV")

        # 22Jan2022
        # simplify a bit, works also for new version of siesta
        # which has extra possibility of using socket to communicate
        launcher = 'mpirun -np ' + \
            str(npc) + ' siesta < ' + self.label + '.fdf > '+self.label+'.out&'
        print("my launcher:", launcher)
        self.ps.pysiestalaunch(self.label, command=launcher)
        print("siesta launched!")

        print("running test...")
        #energy,force = self.ps.pysiestaforce( self.label , self.xyz, self.cell )
        # equilibrium force
        self.initforce()

        # print "siesta equilibrium energy:", energy
        print("siesta equilibrium force:", self.f0)
        print("test finished!")

    def quit(self):
        """
        quit siesta
        """
        print("Quit siesta!")
        self.ps.pysiestaquit(self.label)
        print("Done!")

    def newx(self, q):
        """
        return the real coordinates from displacements got from MD

        performing unit conversion and remove the mass factor
        """
        return self.xyz + self.conv*q

    def absforce(self, q):
        """
        calculate the force from siesta
        q:  displacement list of all atoms, including those fixed
        """
        energy, force = self.ps.pysiestaforce(
            self.label, self.newx(q), self.cell)
        return self.conv*force

    def initforce(self):
        """
        """
        print("Calculate zero displacement force...")
        # equilibrium force
        extq = np.zeros(len(self.xyz))
        self.f0 = self.absforce(extq)

    def force(self, q):
        """
        calculate the relative force 
        q:  displacement list of all atoms, including those fixed
        """
        return self.absforce(q)-self.f0
