"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.mergeTsconfigOptions = exports.TypeScriptLibraryProject = exports.TypeScriptAppProject = exports.TypeScriptProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const semver = require("semver");
const common_1 = require("../common");
const component_1 = require("../component");
const javascript_1 = require("../javascript");
const sample_file_1 = require("../sample-file");
const textfile_1 = require("../textfile");
const typescript_1 = require("../typescript");
/**
 * TypeScript project
 * @pjid typescript
 */
class TypeScriptProject extends javascript_1.NodeProject {
    constructor(options) {
        super({
            ...options,
            // disable .projenrc.js if typescript is enabled
            projenrcJs: options.projenrcTs ? false : options.projenrcJs,
            jestOptions: {
                ...options.jestOptions,
                jestConfig: {
                    ...options.jestOptions?.jestConfig,
                    testMatch: [],
                },
            },
        });
        this.srcdir = options.srcdir ?? "src";
        this.libdir = options.libdir ?? "lib";
        this.docgen = options.docgen;
        this.docsDirectory = options.docsDirectory ?? "docs/";
        this.compileTask.exec("tsc --build");
        this.watchTask = this.addTask("watch", {
            description: "Watch & compile in the background",
            exec: "tsc --build -w",
        });
        this.testdir = options.testdir ?? "test";
        this.gitignore.include(`/${this.testdir}/`);
        this.npmignore?.exclude(`/${this.testdir}/`);
        // if the test directory is under `src/`, then we will run our tests against
        // the javascript files and not let jest compile it for us.
        const compiledTests = this.testdir.startsWith(this.srcdir + path.posix.sep);
        if (options.entrypointTypes || this.entrypoint !== "") {
            const entrypointTypes = options.entrypointTypes ??
                `${path
                    .join(path.dirname(this.entrypoint), path.basename(this.entrypoint, ".js"))
                    .replace(/\\/g, "/")}.d.ts`;
            this.package.addField("types", entrypointTypes);
        }
        const compilerOptionDefaults = {
            alwaysStrict: true,
            declaration: true,
            esModuleInterop: true,
            experimentalDecorators: true,
            inlineSourceMap: true,
            inlineSources: true,
            lib: ["es2019"],
            module: "CommonJS",
            noEmitOnError: false,
            noFallthroughCasesInSwitch: true,
            noImplicitAny: true,
            noImplicitReturns: true,
            noImplicitThis: true,
            noUnusedLocals: true,
            noUnusedParameters: true,
            resolveJsonModule: true,
            strict: true,
            strictNullChecks: true,
            strictPropertyInitialization: true,
            stripInternal: true,
            target: "ES2019",
        };
        if (!options.disableTsconfig) {
            this.tsconfig = new javascript_1.TypescriptConfig(this, mergeTsconfigOptions({
                include: [`${this.srcdir}/**/*.ts`],
                // exclude: ['node_modules'], // TODO: shouldn't we exclude node_modules?
                compilerOptions: {
                    rootDir: this.srcdir,
                    outDir: this.libdir,
                    ...compilerOptionDefaults,
                },
            }, options.tsconfig));
        }
        const tsconfigDevFile = options.tsconfigDevFile ?? "tsconfig.dev.json";
        this.tsconfigDev = new javascript_1.TypescriptConfig(this, mergeTsconfigOptions({
            fileName: tsconfigDevFile,
            include: [
                common_1.PROJEN_RC,
                `${this.srcdir}/**/*.ts`,
                `${this.testdir}/**/*.ts`,
            ],
            exclude: ["node_modules"],
            compilerOptions: compilerOptionDefaults,
        }, options.tsconfig, options.tsconfigDev));
        this.gitignore.include(`/${this.srcdir}/`);
        this.npmignore?.exclude(`/${this.srcdir}/`);
        if (this.srcdir !== this.libdir) {
            // separated, can ignore the entire libdir
            this.gitignore.exclude(`/${this.libdir}`);
        }
        else {
            // collocated, can only ignore the compiled output
            this.gitignore.exclude(`/${this.libdir}/**/*.js`);
            this.gitignore.exclude(`/${this.libdir}/**/*.d.ts`);
            this.gitignore.exclude(`/${this.libdir}/**/*.d.ts.map`);
        }
        this.npmignore?.include(`/${this.libdir}/`);
        this.npmignore?.include(`/${this.libdir}/**/*.js`);
        this.npmignore?.include(`/${this.libdir}/**/*.d.ts`);
        this.gitignore.exclude("/dist/");
        this.npmignore?.exclude("dist"); // jsii-pacmak expects this to be "dist" and not "/dist". otherwise it will tamper with it
        this.npmignore?.exclude("/tsconfig.json");
        this.npmignore?.exclude("/.github/");
        this.npmignore?.exclude("/.vscode/");
        this.npmignore?.exclude("/.idea/");
        this.npmignore?.exclude("/.projenrc.js");
        this.npmignore?.exclude("tsconfig.tsbuildinfo");
        if (this.jest) {
            if (compiledTests) {
                this.addJestCompiled(this.jest);
            }
            else {
                this.addJestNoCompile(this.jest);
            }
        }
        if (options.eslint ?? true) {
            this.eslint = new javascript_1.Eslint(this, {
                tsconfigPath: `./${this.tsconfigDev.fileName}`,
                dirs: [this.srcdir],
                devdirs: [this.testdir, "build-tools"],
                fileExtensions: [".ts", ".tsx"],
                lintProjenRc: false,
                ...options.eslintOptions,
            });
            this.tsconfigEslint = this.tsconfigDev;
        }
        if (!this.parent && options.projenrcTs) {
            new typescript_1.Projenrc(this, options.projenrcTsOptions);
        }
        const tsver = options.typescriptVersion
            ? `@${options.typescriptVersion}`
            : "";
        this.addDevDeps(`typescript${tsver}`, 
        // @types/node versions numbers match the node runtime versions' major.minor, however, new
        // releases are only created when API changes are included in a node release... We might for
        // example have dependencies that require `node >= 12.22`, but as 12.21 and 12.22 did not
        // include API changes, `@types/node@12.20.x` is the "correct" version to use. As it is not
        // possible to easily determine the correct version to use, we pick up the latest version.
        //
        // Additionally, we default to tracking the 12.x line, as the current earliest LTS release of
        // node is 12.x, so this is what corresponds to the broadest compatibility with supported node
        // runtimes.
        `@types/node@^${semver.major(this.package.minNodeVersion ?? "16.0.0")}`);
        // generate sample code in `src` and `lib` if these directories are empty or non-existent.
        if (options.sampleCode ?? true) {
            new SampleCode(this);
        }
        if (this.docgen) {
            new typescript_1.TypedocDocgen(this);
        }
    }
    /**
     * Tests are compiled to `lib/TESTDIR`, so we don't need jest to compile them
     * for us. just run them directly from javascript.
     */
    addJestCompiled(jest) {
        this.addDevDeps(`@types/jest${jest.jestVersion}`);
        const testout = path.posix.relative(this.srcdir, this.testdir);
        const libtest = path.posix.join(this.libdir, testout);
        const srctest = this.testdir;
        this.npmignore?.exclude(`/${libtest}/`);
        jest.addTestMatch(`**/${libtest}/**/?(*.)+(spec|test).js?(x)`);
        jest.addWatchIgnorePattern(`/${this.srcdir}/`);
        const resolveSnapshotPath = (test, ext) => {
            const fullpath = test.replace(libtest, srctest);
            return path.join(path.dirname(fullpath), "__snapshots__", path.basename(fullpath, ".js") + ".ts" + ext);
        };
        const resolveTestPath = (snap, ext) => {
            const filename = path.basename(snap, ".ts" + ext) + ".js";
            const dir = path.dirname(path.dirname(snap)).replace(srctest, libtest);
            return path.join(dir, filename);
        };
        const resolver = new textfile_1.TextFile(this, path.posix.join(common_1.PROJEN_DIR, "jest-snapshot-resolver.js"));
        if (!resolver.marker) {
            resolver.addLine(`// ${resolver.marker}`);
        }
        resolver.addLine('const path = require("path");');
        resolver.addLine(`const libtest = "${libtest}";`);
        resolver.addLine(`const srctest= "${srctest}";`);
        resolver.addLine("module.exports = {");
        resolver.addLine(`  resolveSnapshotPath: ${resolveSnapshotPath.toString()},`);
        resolver.addLine(`  resolveTestPath: ${resolveTestPath.toString()},`);
        resolver.addLine("  testPathForConsistencyCheck: path.join('some', '__tests__', 'example.test.js')");
        resolver.addLine("};");
        jest.addSnapshotResolver(`./${resolver.path}`);
    }
    addJestNoCompile(jest) {
        this.addDevDeps(`@types/jest${jest.jestVersion}`, `ts-jest${jest.jestVersion}`);
        jest.addTestMatch(`<rootDir>/${this.srcdir}/**/__tests__/**/*.ts?(x)`);
        jest.addTestMatch(`<rootDir>/(${this.testdir}|${this.srcdir})/**/*(*.)@(spec|test).ts?(x)`);
        // add relevant deps
        jest.config.preset = "ts-jest";
        jest.config.globals = {
            "ts-jest": {
                tsconfig: this.tsconfigDev.fileName,
            },
        };
    }
}
exports.TypeScriptProject = TypeScriptProject;
_a = JSII_RTTI_SYMBOL_1;
TypeScriptProject[_a] = { fqn: "projen.typescript.TypeScriptProject", version: "0.69.3" };
class SampleCode extends component_1.Component {
    constructor(project) {
        super(project);
        const srcCode = [
            "export class Hello {",
            "  public sayHello() {",
            "    return 'hello, world!';",
            "  }",
            "}",
        ].join("\n");
        const testCode = [
            "import { Hello } from '../src';",
            "",
            "test('hello', () => {",
            "  expect(new Hello().sayHello()).toBe('hello, world!');",
            "});",
        ].join("\n");
        new sample_file_1.SampleDir(project, project.srcdir, {
            files: {
                "index.ts": srcCode,
            },
        });
        if (project.jest) {
            new sample_file_1.SampleDir(project, project.testdir, {
                files: {
                    "hello.test.ts": testCode,
                },
            });
        }
    }
}
/**
 * TypeScript app.
 *
 * @pjid typescript-app
 */
class TypeScriptAppProject extends TypeScriptProject {
    constructor(options) {
        super({
            allowLibraryDependencies: false,
            releaseWorkflow: false,
            entrypoint: "",
            package: false,
            ...options,
        });
    }
}
exports.TypeScriptAppProject = TypeScriptAppProject;
_b = JSII_RTTI_SYMBOL_1;
TypeScriptAppProject[_b] = { fqn: "projen.typescript.TypeScriptAppProject", version: "0.69.3" };
/**
 * @deprecated use `TypeScriptProject`
 */
class TypeScriptLibraryProject extends TypeScriptProject {
}
exports.TypeScriptLibraryProject = TypeScriptLibraryProject;
_c = JSII_RTTI_SYMBOL_1;
TypeScriptLibraryProject[_c] = { fqn: "projen.typescript.TypeScriptLibraryProject", version: "0.69.3" };
/**
 * @internal
 */
function mergeTsconfigOptions(...options) {
    const definedOptions = options.filter(Boolean);
    return definedOptions.reduce((previous, current) => ({
        ...previous,
        ...current,
        include: [...(previous.include ?? []), ...(current.include ?? [])],
        exclude: [...(previous.exclude ?? []), ...(current.exclude ?? [])],
        compilerOptions: {
            ...previous.compilerOptions,
            ...current.compilerOptions,
        },
    }), { compilerOptions: {} });
}
exports.mergeTsconfigOptions = mergeTsconfigOptions;
//# sourceMappingURL=data:application/json;base64,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