"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Tasks = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const component_1 = require("./component");
const json_1 = require("./json");
const task_1 = require("./task");
const task_runtime_1 = require("./task-runtime");
/**
 * Defines project tasks.
 *
 * Tasks extend the projen CLI by adding subcommands to it. Task definitions are
 * synthesized into `.projen/tasks.json`.
 */
class Tasks extends component_1.Component {
    constructor(project) {
        super(project);
        this._tasks = {};
        this._env = {};
        new json_1.JsonFile(project, task_runtime_1.TaskRuntime.MANIFEST_FILE, {
            omitEmpty: true,
            obj: {
                tasks: (() => this.renderTasks()),
                env: (() => this._env),
            },
        });
    }
    /**
     * All tasks.
     */
    get all() {
        return Object.values(this._tasks);
    }
    /**
     * Adds a task to a project.
     * @param name The name of the task
     * @param options Task options.
     */
    addTask(name, options = {}) {
        const task = new task_1.Task(name, options);
        if (this._tasks[name]) {
            throw new Error(`A task with the name ${name} already exists. To override it, call removeTask first and then create the new task.`);
        }
        this._tasks[name] = task;
        return task;
    }
    /**
     * Removes a task from a project.
     *
     * @param name The name of the task to remove.
     *
     * @returns The `Task` that was removed, otherwise `undefined`.
     */
    removeTask(name) {
        const dependentTasks = this.all.filter((task) => task.steps.find((step) => step.spawn == name));
        if (dependentTasks.length > 0) {
            const errList = dependentTasks.map((depTask) => depTask.name).join(", ");
            throw new Error(`Unable to remove task "${name}" because the following tasks depend on it: ${errList}`);
        }
        const task = this._tasks[name];
        if (task) {
            delete this._tasks[name];
            return task;
        }
        else {
            return undefined;
        }
    }
    /**
     * Adds global environment.
     * @param name Environment variable name
     * @param value Value
     */
    addEnvironment(name, value) {
        if (typeof value !== "string" && value !== undefined) {
            this.project.logger.warn(`Received non-string value for environment variable ${name}. Value will be stringified.`);
            this._env[name] = String(value);
        }
        else {
            this._env[name] = value;
        }
    }
    /**
     * Returns a copy of the currently global environment for this project.
     */
    get env() {
        return {
            ...this._env,
        };
    }
    /**
     * Finds a task by name. Returns `undefined` if the task cannot be found.
     * @param name The name of the task
     */
    tryFind(name) {
        return this._tasks[name];
    }
    synthesize() {
        if (this.project.ejected) {
            // Insert a task-runner script so that tasks can be run after ejecting
            fs.mkdirSync(path.join(this.project.outdir, "scripts"), {
                recursive: true,
            });
            fs.copyFileSync(path.join(__dirname, "..", "lib", "run-task.js"), path.join(this.project.outdir, "scripts", "run-task"));
            fs.chmodSync(path.join(this.project.outdir, "scripts", "run-task"), "755");
        }
    }
    renderTasks() {
        const tasks = {};
        for (const task of Object.values(this._tasks).sort((x, y) => x.name.localeCompare(y.name))) {
            tasks[task.name] = task._renderSpec();
        }
        return tasks;
    }
}
exports.Tasks = Tasks;
_a = JSII_RTTI_SYMBOL_1;
Tasks[_a] = { fqn: "projen.Tasks", version: "0.69.3" };
//# sourceMappingURL=data:application/json;base64,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