"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.updateChangelog = void 0;
const fs_1 = require("fs");
const path_1 = require("path");
const logging = require("../logging");
const utils = require("../util");
/**
 * Prepends input changelog entry onto project-level changelog.
 *
 * Currently assumes a headerless changelog formatted according to
 * [conventional-changelog](https://github.com/conventional-changelog/conventional-changelog)
 * rules.
 *
 * @param cwd working directory (git repository)
 * @param options options
 */
async function updateChangelog(cwd, options) {
    const inputChangelog = path_1.join(cwd, options.inputChangelog);
    const outputChangelog = path_1.join(cwd, options.outputChangelog);
    const versionFile = path_1.join(cwd, options.versionFile);
    let version = (await utils.tryReadFile(versionFile)).trim();
    if (!version) {
        throw new Error(`Unable to determine version from ${versionFile}. Cannot proceed with changelog update. Did you run 'bump'?`);
    }
    const inputChangelogContent = await fs_1.promises.readFile(inputChangelog, "utf-8");
    const changelogVersionSearchPattern = `${version}`;
    if (!inputChangelogContent.includes(changelogVersionSearchPattern)) {
        throw new Error(`Supplied version ${version} was not found in input changelog. You may want to check it's content.`);
    }
    const outputChangelogContent = await fs_1.promises.readFile(outputChangelog, {
        encoding: "utf-8",
        flag: "a+",
    });
    if (outputChangelogContent.indexOf(changelogVersionSearchPattern) > -1) {
        logging.info(`Changelog already contains an entry for ${version}. Skipping changelog update.`);
        return;
    }
    const newChangelog = inputChangelogContent.trimEnd() +
        "\n\n" +
        outputChangelogContent.trimStart();
    await fs_1.promises.writeFile(outputChangelog, newChangelog);
    utils.exec(`git add ${outputChangelog} && git commit -m "chore(release): ${version}"`, { cwd });
}
exports.updateChangelog = updateChangelog;
//# sourceMappingURL=data:application/json;base64,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