"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.tag = void 0;
const fs_1 = require("fs");
const path_1 = require("path");
const utils = require("../util");
/**
 * Generate an annotated release tag using version and changelog files.
 *
 * The tag will be normalized to the format "v<version>" where version comes
 * from the provided version file.
 *
 * The tag annotation message will be set to the content of the provided release
 * changelog file.
 *
 * @param cwd working directory (git repository)
 * @param options options
 */
async function tag(cwd, options) {
    const git = (cmd) => utils.exec(`git ${cmd}`, { cwd: cwd });
    const releaseTagFilePath = path_1.join(cwd, options.releaseTagFile);
    const changelogFilePath = path_1.join(cwd, options.changelog);
    if (!fs_1.existsSync(releaseTagFilePath)) {
        throw new Error(`No release tag file present at ${releaseTagFilePath}`);
    }
    if (!fs_1.existsSync(changelogFilePath)) {
        throw new Error(`No changelog file present at ${changelogFilePath}`);
    }
    let releaseTag = (await utils.tryReadFile(releaseTagFilePath)).trim();
    if (!releaseTag) {
        throw new Error(`No version present in file at ${releaseTagFilePath}`);
    }
    git(`tag ${releaseTag} -a -F ${changelogFilePath}`);
}
exports.tag = tag;
//# sourceMappingURL=data:application/json;base64,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