import { Publisher } from "./publisher";
import { ReleaseTrigger } from "./release-trigger";
import { Component } from "../component";
import { GitHubProject } from "../github";
import { Job, JobPermissions, JobStep } from "../github/workflows-model";
import { Task } from "../task";
declare type BranchHook = (branch: string) => void;
/**
 * Project options for release.
 */
export interface ReleaseProjectOptions {
    /**
     * Automatically release new versions every commit to one of branches in `releaseBranches`.
     *
     * @default true
     *
     * @deprecated Use `releaseTrigger: ReleaseTrigger.continuous()` instead
     */
    readonly releaseEveryCommit?: boolean;
    /**
     * CRON schedule to trigger new releases.
     *
     * @default - no scheduled releases
     *
     * @deprecated Use `releaseTrigger: ReleaseTrigger.scheduled()` instead
     */
    readonly releaseSchedule?: string;
    /**
     * The release trigger to use.
     *
     * @default - Continuous releases (`ReleaseTrigger.continuous()`)
     */
    readonly releaseTrigger?: ReleaseTrigger;
    /**
     * A set of workflow steps to execute in order to setup the workflow
     * container.
     */
    readonly releaseWorkflowSetupSteps?: JobStep[];
    /**
     * Container image to use for GitHub workflows.
     *
     * @default - default image
     */
    readonly workflowContainerImage?: string;
    /**
     * Version requirement of `publib` which is used to publish modules to npm.
     * @default "latest"
     */
    readonly jsiiReleaseVersion?: string;
    /**
     * Steps to execute after build as part of the release workflow.
     * @default []
     */
    readonly postBuildSteps?: JobStep[];
    /**
     * Major version to release from the default branch.
     *
     * If this is specified, we bump the latest version of this major version line.
     * If not specified, we bump the global latest version.
     *
     * @default - Major version is not enforced.
     */
    readonly majorVersion?: number;
    /**
     * Minimal Major version to release
     *
     *
     * This can be useful to set to 1, as breaking changes before the 1.x major
     * release are not incrementing the major version number.
     *
     * Can not be set together with `majorVersion`.
     *
     * @default - No minimum version is being enforced
     */
    readonly minMajorVersion?: number;
    /**
     * Bump versions from the default branch as pre-releases (e.g. "beta",
     * "alpha", "pre").
     *
     * @default - normal semantic versions
     */
    readonly prerelease?: string;
    /**
     * The npmDistTag to use when publishing from the default branch.
     *
     * To set the npm dist-tag for release branches, set the `npmDistTag` property
     * for each branch.
     *
     * @default "latest"
     */
    readonly npmDistTag?: string;
    /**
     * The name of the default release workflow.
     *
     * @default "Release"
     */
    readonly releaseWorkflowName?: string;
    /**
     * Defines additional release branches. A workflow will be created for each
     * release branch which will publish releases from commits in this branch.
     * Each release branch _must_ be assigned a major version number which is used
     * to enforce that versions published from that branch always use that major
     * version. If multiple branches are used, the `majorVersion` field must also
     * be provided for the default branch.
     *
     * @default - no additional branches are used for release. you can use
     * `addBranch()` to add additional branches.
     */
    readonly releaseBranches?: {
        [name: string]: BranchOptions;
    };
    /**
     * Create a github issue on every failed publishing task.
     *
     * @default false
     */
    readonly releaseFailureIssue?: boolean;
    /**
     * The label to apply to issues indicating publish failures.
     * Only applies if `releaseFailureIssue` is true.
     *
     * @default "failed-release"
     */
    readonly releaseFailureIssueLabel?: string;
    /**
     * Automatically add the given prefix to release tags.
     * Useful if you are releasing on multiple branches with overlapping
     * version numbers.
     *
     * Note: this prefix is used to detect the latest tagged version
     * when bumping, so if you change this on a project with an existing version
     * history, you may need to manually tag your latest release
     * with the new prefix.
     *
     * @default "v"
     */
    readonly releaseTagPrefix?: string;
    /**
     * Custom configuration used when creating changelog with standard-version package.
     * Given values either append to default configuration or overwrite values in it.
     *
     * @default - standard configuration applicable for GitHub repositories
     */
    readonly versionrcOptions?: Record<string, any>;
    /**
     * Github Runner selection labels
     * @default ["ubuntu-latest"]
     */
    readonly workflowRunsOn?: string[];
    /**
     * Define publishing tasks that can be executed manually as well as workflows.
     *
     * Normally, publishing only happens within automated workflows. Enable this
     * in order to create a publishing task for each publishing activity.
     *
     * @default false
     */
    readonly publishTasks?: boolean;
    /**
     * Instead of actually publishing to package managers, just print the publishing command.
     *
     * @default false
     */
    readonly publishDryRun?: boolean;
}
/**
 * Options for `Release`.
 */
export interface ReleaseOptions extends ReleaseProjectOptions {
    /**
     * The task to execute in order to create the release artifacts. Artifacts are
     * expected to reside under `artifactsDirectory` (defaults to `dist/`) once
     * build is complete.
     */
    readonly task: Task;
    /**
     * A name of a .json file to set the `version` field in after a bump.
     *
     * @example "package.json"
     */
    readonly versionFile: string;
    /**
     * The default branch name to release from.
     *
     * Use `majorVersion` to restrict this branch to only publish releases with a
     * specific major version.
     *
     * You can add additional branches using `addBranch()`.
     */
    readonly branch: string;
    /**
     * Create a GitHub release for each release.
     *
     * @default true
     */
    readonly githubRelease?: boolean;
    /**
     * A directory which will contain build artifacts.
     *
     * @default "dist"
     */
    readonly artifactsDirectory: string;
    /**
     * Node version to setup in GitHub workflows if any node-based CLI utilities
     * are needed. For example `publib`, the CLI projen uses to publish releases,
     * is an npm library.
     *
     * @default 16.x
     */
    readonly workflowNodeVersion?: string;
    /**
     * Permissions granted to the release workflow job
     * @default `{ contents: JobPermission.WRITE }`
     */
    readonly workflowPermissions?: JobPermissions;
}
/**
 * Manages releases (currently through GitHub workflows).
 *
 * By default, no branches are released. To add branches, call `addBranch()`.
 */
export declare class Release extends Component {
    static readonly ANTI_TAMPER_CMD = "git diff --ignore-space-at-eol --exit-code";
    /**
     * Returns the `Release` component of a project or `undefined` if the project
     * does not have a Release component.
     */
    static of(project: GitHubProject): Release | undefined;
    /**
     * Package publisher.
     */
    readonly publisher: Publisher;
    private readonly buildTask;
    private readonly version;
    private readonly postBuildSteps;
    private readonly versionFile;
    private readonly releaseTrigger;
    private readonly preBuildSteps;
    private readonly containerImage?;
    private readonly _branches;
    private readonly jobs;
    private readonly defaultBranch;
    private readonly github?;
    private readonly workflowRunsOn?;
    private readonly workflowPermissions;
    private readonly _branchHooks;
    /**
     * Location of build artifacts.
     */
    readonly artifactsDirectory: string;
    constructor(project: GitHubProject, options: ReleaseOptions);
    /**
     * Add a hook that should be run for every branch (including those that will
     * be added by future `addBranch` calls).
     * @internal
     */
    _forEachBranch(hook: BranchHook): void;
    /**
     * Adds a release branch.
     *
     * It is a git branch from which releases are published. If a project has more than one release
     * branch, we require that `majorVersion` is also specified for the primary branch in order to
     * ensure branches always release the correct version.
     *
     * @param branch The branch to monitor (e.g. `main`, `v2.x`)
     * @param options Branch definition
     */
    addBranch(branch: string, options: BranchOptions): void;
    /**
     * Adds a release branch.
     *
     * It is a git branch from which releases are published. If a project has more than one release
     * branch, we require that `majorVersion` is also specified for the primary branch in order to
     * ensure branches always release the correct version.
     *
     * @param branch The branch to monitor (e.g. `main`, `v2.x`)
     * @param options Branch definition
     */
    private _addBranch;
    preSynthesize(): void;
    /**
     * Adds jobs to all release workflows.
     * @param jobs The jobs to add (name => job)
     */
    addJobs(jobs: Record<string, Job>): void;
    /**
     * Retrieve all release branch names
     */
    get branches(): string[];
    /**
     * @returns a workflow or `undefined` if github integration is disabled.
     */
    private createWorkflow;
}
/**
 * Options for a release branch.
 */
export interface BranchOptions {
    /**
     * The name of the release workflow.
     * @default "release-BRANCH"
     */
    readonly workflowName?: string;
    /**
     * The major versions released from this branch.
     */
    readonly majorVersion: number;
    /**
     * The minimum major version to release.
     */
    readonly minMajorVersion?: number;
    /**
     * Bump the version as a pre-release tag.
     *
     * @default - normal releases
     */
    readonly prerelease?: string;
    /**
     * Automatically add the given prefix to release tags.
     * Useful if you are releasing on multiple branches with overlapping
     * version numbers.
     *
     * Note: this prefix is used to detect the latest tagged version
     * when bumping, so if you change this on a project with an existing version
     * history, you may need to manually tag your latest release
     * with the new prefix.
     *
     * @default - no prefix
     */
    readonly tagPrefix?: string;
    /**
     * The npm distribution tag to use for this branch.
     *
     * @default "latest"
     */
    readonly npmDistTag?: string;
}
export {};
