"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.bump = void 0;
const fs_1 = require("fs");
const path_1 = require("path");
const logging = require("../logging");
const util_1 = require("../util");
/**
 * Resolves the latest version from git tags and uses `standard-version` to bump
 * to the next version based on commits.
 *
 * This expects `standard-version` to be installed in the path.
 *
 * @param cwd working directory (git repository)
 * @param options options
 */
async function bump(cwd, options) {
    const versionFile = path_1.join(cwd, options.versionFile);
    const prerelease = options.prerelease;
    const major = options.majorVersion;
    const minMajorVersion = options.minMajorVersion;
    const prefix = options.tagPrefix ?? "";
    const bumpFile = path_1.join(cwd, options.bumpFile);
    const changelogFile = path_1.join(cwd, options.changelog);
    const releaseTagFile = path_1.join(cwd, options.releaseTagFile);
    if (major && minMajorVersion) {
        throw new Error(`minMajorVersion and majorVersion cannot be used together.`);
    }
    await fs_1.promises.mkdir(path_1.dirname(bumpFile), { recursive: true });
    await fs_1.promises.mkdir(path_1.dirname(changelogFile), { recursive: true });
    await fs_1.promises.mkdir(path_1.dirname(releaseTagFile), { recursive: true });
    const { latestVersion, latestTag, isFirstRelease } = determineLatestTag({
        cwd,
        major,
        prerelease,
        prefix,
    });
    const content = await tryReadVersionFile(versionFile);
    // update version
    content.version = latestVersion;
    logging.info(`Update ${versionFile} to latest resolved version: ${latestVersion}`);
    await fs_1.promises.writeFile(versionFile, JSON.stringify(content, undefined, 2));
    // check if the latest commit already has a version tag
    const currentTags = util_1.execCapture("git tag --points-at HEAD", { cwd })
        .toString("utf8")
        .split("\n");
    logging.info(`Tags listed on current commit: ${currentTags}`);
    let skipBump = false;
    if (currentTags.includes(latestTag)) {
        logging.info("Skipping bump...");
        skipBump = true;
        // delete the existing tag (locally)
        // if we don't do this, standard-version generates an empty changelog
        util_1.exec(`git tag --delete ${latestTag}`, { cwd });
    }
    // create a standard-version configuration file
    const rcfile = path_1.join(cwd, ".versionrc.json");
    await generateVersionrcFile(rcfile, versionFile, changelogFile, skipBump, prerelease, options.versionrcOptions);
    const cmd = ["npx", "standard-version@^9"];
    if (isFirstRelease) {
        cmd.push("--first-release");
    }
    if (prefix) {
        cmd.push(`--tag-prefix ${prefix}v`);
    }
    if (minMajorVersion) {
        const [majorVersion] = latestVersion.split(".");
        const majorVersionNumber = parseInt(majorVersion, 10);
        if (majorVersionNumber < minMajorVersion) {
            cmd.push(`--release-as ${minMajorVersion}.0.0`);
        }
    }
    util_1.exec(cmd.join(" "), { cwd });
    // add the tag back if it was previously removed
    if (currentTags.includes(latestTag)) {
        util_1.exec(`git tag ${latestTag}`, { cwd });
    }
    await fs_1.promises.rm(rcfile, { force: true, recursive: true });
    const newVersion = (await tryReadVersionFile(versionFile)).version;
    if (!newVersion) {
        throw new Error(`bump failed: ${versionFile} does not have a version set`);
    }
    // if MAJOR is defined, ensure that the new version is within the same major version
    if (major) {
        if (!newVersion.startsWith(`${major}.`)) {
            throw new Error(`bump failed: this branch is configured to only publish v${major} releases - bump resulted in ${newVersion}`);
        }
    }
    await fs_1.promises.writeFile(bumpFile, newVersion);
    const newTag = `${prefix}v${newVersion}`;
    await fs_1.promises.writeFile(releaseTagFile, newTag);
}
exports.bump = bump;
async function tryReadVersionFile(versionFile) {
    if (!fs_1.existsSync(versionFile)) {
        return {};
    }
    return JSON.parse(await fs_1.promises.readFile(versionFile, "utf-8"));
}
function generateVersionrcFile(rcfile, versionFile, changelogFile, skipBump, prerelease, configOptions) {
    return fs_1.promises.writeFile(rcfile, JSON.stringify({
        ...{
            packageFiles: [
                {
                    filename: versionFile,
                    type: "json",
                },
            ],
            bumpFiles: [
                {
                    filename: versionFile,
                    type: "json",
                },
            ],
            commitAll: false,
            infile: changelogFile,
            prerelease: prerelease,
            header: "",
            skip: {
                commit: true,
                tag: true,
                bump: skipBump,
            },
            ...configOptions,
        },
    }, undefined, 2));
}
/**
 * Determines the latest release tag.
 * @param major (optional) A major version line to select from
 * @param prerelease (optional) A pre-release suffix.
 * @returns the latest tag, and whether it is the first release or not
 */
function determineLatestTag(options) {
    const { cwd, major, prerelease, prefix } = options;
    // filter only tags for this prefix and major version if specified (start with "vNN.").
    const prefixFilter = major !== undefined ? `${prefix}v${major}.*` : `${prefix}v*`;
    const listGitTags = [
        "git",
        '-c "versionsort.suffix=-"',
        "tag",
        '--sort="-version:refname"',
        "--list",
        `"${prefixFilter}"`,
    ].join(" ");
    const stdout = util_1.execCapture(listGitTags, { cwd }).toString("utf8");
    let tags = stdout?.split("\n");
    // if prerelease is set and there are existing prerelease tags, filter versions that end with "-PRE.ddd".
    const prereleaseTags = tags.filter((x) => new RegExp(`-${prerelease}\.[0-9]+$`).test(x));
    if (prerelease && prereleaseTags.length > 0) {
        tags = prereleaseTags;
    }
    tags = tags.filter((x) => x);
    // if a pre-release tag is used, then add it to the initial version
    let isFirstRelease = false;
    let latestTag;
    if (tags.length > 0) {
        latestTag = tags[0];
    }
    else {
        const initial = `${prefix}v${major ?? 0}.0.0`;
        latestTag = prerelease ? `${initial}-${prerelease}.0` : initial;
        isFirstRelease = true;
    }
    // remove tag prefix (if exists)
    let latestVersion = latestTag;
    if (prefix && latestVersion.startsWith(prefix)) {
        latestVersion = latestVersion.substr(prefix.length);
    }
    // remove "v" prefix (if exists)
    if (latestVersion.startsWith("v")) {
        latestVersion = latestVersion.substr(1);
    }
    return { latestVersion, latestTag, isFirstRelease };
}
//# sourceMappingURL=data:application/json;base64,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