"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PythonProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const pip_1 = require("./pip");
const poetry_1 = require("./poetry");
const projenrc_1 = require("./projenrc");
const pytest_1 = require("./pytest");
const pytest_sample_1 = require("./pytest-sample");
const python_sample_1 = require("./python-sample");
const setuptools_1 = require("./setuptools");
const venv_1 = require("./venv");
const github_1 = require("../github");
const projenrc_2 = require("../javascript/projenrc");
const project_1 = require("../project");
const typescript_1 = require("../typescript");
const util_1 = require("../util");
/** Allowed characters in python project names */
const PYTHON_PROJECT_NAME_REGEX = /^[A-Za-z0-9-_\.]+$/;
/**
 * Python project.
 *
 * @pjid python
 */
class PythonProject extends github_1.GitHubProject {
    constructor(options) {
        super(options);
        if (!PYTHON_PROJECT_NAME_REGEX.test(options.name)) {
            throw new Error("Python projects must only consist of alphanumeric characters, hyphens, and underscores.");
        }
        this.moduleName = options.moduleName;
        this.version = options.version;
        const rcFileTypeOptions = [
            options.projenrcPython,
            options.projenrcJs,
            options.projenrcJson,
            options.projenrcTs,
        ];
        if (util_1.multipleSelected(rcFileTypeOptions)) {
            throw new Error("Only one of projenrcPython, projenrcJs, projenrcTs, and projenrcJson can be selected.");
        }
        const poetry = options.poetry ?? false;
        const pip = options.pip ?? !poetry;
        const venv = options.venv ?? !poetry;
        const setuptools = options.setuptools ?? (!poetry && this.projectType === project_1.ProjectType.LIB);
        if (poetry && (pip || venv || setuptools)) {
            throw new Error("poetry is true - pip, venv, and setuptools must be undefined or false");
        }
        if (!this.parent) {
            // default to projenrc.py if no other projenrc type was elected
            if (options.projenrcPython ?? !util_1.anySelected(rcFileTypeOptions)) {
                new projenrc_1.Projenrc(this, options.projenrcPythonOptions);
            }
            if (options.projenrcJs ?? false) {
                new projenrc_2.Projenrc(this, options.projenrcJsOptions);
            }
            if (options.projenrcTs ?? false) {
                new typescript_1.ProjenrcTs(this, options.projenrcTsOptions);
            }
        }
        if (venv) {
            this.envManager = new venv_1.Venv(this, options.venvOptions);
        }
        if (pip) {
            this.depsManager = new pip_1.Pip(this);
        }
        if (setuptools) {
            this.packagingManager = new setuptools_1.Setuptools(this, {
                version: options.version,
                description: options.description,
                authorName: options.authorName,
                authorEmail: options.authorEmail,
                license: options.license,
                homepage: options.homepage,
                classifiers: options.classifiers,
                setupConfig: options.setupConfig,
            });
        }
        // if (options.conda ?? false) {
        //   this.depsManager = new Conda(this, options);
        //   this.envManager = this.depsManager;
        // }
        // if (options.pipenv ?? false) {
        //   this.depsManager = new Pipenv(this, options);
        //   this.envManager = this.depsManager;
        // }
        if (poetry) {
            const poetryProject = new poetry_1.Poetry(this, {
                version: options.version,
                description: options.description,
                authorName: options.authorName,
                authorEmail: options.authorEmail,
                license: options.license,
                homepage: options.homepage,
                classifiers: options.classifiers,
                poetryOptions: {
                    readme: options.readme?.filename ?? "README.md",
                    ...options.poetryOptions,
                },
            });
            this.depsManager = poetryProject;
            this.envManager = poetryProject;
            this.packagingManager = poetryProject;
        }
        if (!this.envManager) {
            throw new Error("At least one tool must be chosen for managing the environment (venv, conda, pipenv, or poetry).");
        }
        if (!this.depsManager) {
            throw new Error("At least one tool must be chosen for managing dependencies (pip, conda, pipenv, or poetry).");
        }
        if (!this.packagingManager && this.projectType === project_1.ProjectType.LIB) {
            throw new Error("At least one tool must be chosen for managing packaging (setuptools or poetry).");
        }
        if (Number(venv) + Number(poetry) > 1) {
            throw new Error("More than one component has been chosen for managing the environment (venv, conda, pipenv, or poetry)");
        }
        if (Number(pip) + Number(poetry) > 1) {
            throw new Error("More than one component has been chosen for managing dependencies (pip, conda, pipenv, or poetry)");
        }
        if (Number(setuptools) + Number(poetry) > 1) {
            throw new Error("More than one component has been chosen for managing packaging (setuptools or poetry)");
        }
        if (options.pytest ?? true) {
            this.pytest = new pytest_1.Pytest(this, options.pytestOptions);
            new pytest_sample_1.PytestSample(this, {
                moduleName: this.moduleName,
                testdir: this.pytest.testdir,
            });
        }
        if (options.sample ?? true) {
            new python_sample_1.PythonSample(this, {
                dir: this.moduleName,
            });
        }
        for (const dep of options.deps ?? []) {
            this.addDependency(dep);
        }
        for (const dep of options.devDeps ?? []) {
            this.addDevDependency(dep);
        }
        this.addDefaultGitIgnore();
    }
    /**
     * Adds default gitignore options for a Python project based on
     * https://github.com/github/gitignore/blob/master/Python.gitignore
     */
    addDefaultGitIgnore() {
        this.gitignore.exclude("# Byte-compiled / optimized / DLL files", "__pycache__/", "*.py[cod]", "*$py.class", "", "# C extensions", "*.so", "", "# Distribution / packaging", ".Python", "build/", "develop-eggs/", "dist/", "downloads/", "eggs/", ".eggs/", "lib/", "lib64/", "parts/", "sdist/", "var/", "wheels/", "share/python-wheels/", "*.egg-info/", ".installed.cfg", "*.egg", "MANIFEST", "", "# PyInstaller", "#  Usually these files are written by a python script from a template", "#  before PyInstaller builds the exe, so as to inject date/other infos into it.", "*.manifest", "*.spec", "", "# Installer logs", "pip-log.txt", "pip-delete-this-directory.txt", "", "# Unit test / coverage reports", "htmlcov/", ".tox/", ".nox/", ".coverage", ".coverage.*", ".cache", "nosetests.xml", "coverage.xml", "*.cover", "*.py,cover", ".hypothesis/", ".pytest_cache/", "cover/", "", "# Translations", "*.mo", "*.pot", "", "# Django stuff:", "*.log", "local_settings.py", "db.sqlite3", "db.sqlite3-journal", "", "# Flask stuff:", "instance/", ".webassets-cache", "", "# Scrapy stuff:", ".scrapy", "", "# Sphinx documentation", "docs/_build/", "", "# PyBuilder", ".pybuilder/", "target/", "", "# Jupyter Notebook", ".ipynb_checkpoints", "", "# IPython", "profile_default/", "ipython_config.py", "", "# PEP 582; used by e.g. github.com/David-OConnor/pyflow", "__pypackages__/", "", "# Celery stuff", "celerybeat-schedule", "celerybeat.pid", "", "# SageMath parsed files", "*.sage.py", "", "# Environments", ".env", ".venv", "env/", "venv/", "ENV/", "env.bak/", "venv.bak/", "", "# Spyder project settings", ".spyderproject", ".spyproject", "", "# Rope project settings", ".ropeproject", "", "# mkdocs documentation", "/site", "", "# mypy", ".mypy_cache/", ".dmypy.json", "dmypy.json", "", "# Pyre type checker", ".pyre/", "", "# pytype static type analyzer", ".pytype/", "", "# Cython debug symbols", "cython_debug/");
    }
    /**
     * Adds a runtime dependency.
     *
     * @param spec Format `<module>@<semver>`
     */
    addDependency(spec) {
        return this.depsManager.addDependency(spec);
    }
    /**
     * Adds a dev dependency.
     *
     * @param spec Format `<module>@<semver>`
     */
    addDevDependency(spec) {
        return this.depsManager.addDevDependency(spec);
    }
    postSynthesize() {
        super.postSynthesize();
        this.envManager.setupEnvironment();
        this.depsManager.installDependencies();
    }
}
exports.PythonProject = PythonProject;
_a = JSII_RTTI_SYMBOL_1;
PythonProject[_a] = { fqn: "projen.python.PythonProject", version: "0.69.3" };
//# sourceMappingURL=data:application/json;base64,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