"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.resolvePythonImportName = exports.Projenrc = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs_1 = require("fs");
const path_1 = require("path");
const case_1 = require("case");
const common_1 = require("../common");
const dependencies_1 = require("../dependencies");
const inventory_1 = require("../inventory");
const projenrc_1 = require("../projenrc");
/**
 * Allows writing projenrc files in python.
 *
 * This will install `projen` as a Python dependency and will add a
 * `synth` task which will run `.projenrc.py`.
 */
class Projenrc extends projenrc_1.ProjenrcFile {
    constructor(project, options = {}) {
        super(project);
        const projenVersion = options.projenVersion ?? common_1.PROJEN_VERSION;
        this.filePath = options.filename ?? ".projenrc.py";
        project.deps.addDependency(`projen@${projenVersion}`, dependencies_1.DependencyType.DEVENV);
        // set up the "default" task which is the task executed when `projen` is executed for this project.
        project.defaultTask?.exec("python .projenrc.py");
        // if this is a new project, generate a skeleton for projenrc.py
        this.generateProjenrc();
    }
    generateProjenrc() {
        const bootstrap = this.project.initProject;
        if (!bootstrap) {
            return;
        }
        const jsiiFqn = bootstrap.fqn;
        const jsiiManifest = inventory_1.readJsiiManifest(jsiiFqn);
        const jsiiType = jsiiManifest.types[jsiiFqn];
        const optionsTypeFqn = jsiiType.initializer?.parameters?.[0].type?.fqn;
        if (!optionsTypeFqn) {
            this.project.logger.warn("cannot determine jsii type for project options");
            return;
        }
        const pythonFile = path_1.join(this.project.outdir, this.filePath);
        // skip if file exists
        if (fs_1.existsSync(pythonFile)) {
            return;
        }
        const lines = new Array();
        let indent = 0;
        const emit = (line = "") => lines.push(" ".repeat(indent * 4) + line);
        // const openBlock = (line: string = '') => { emit(line + ' {'); indent++; };
        // const closeBlock = () => { indent--; emit('}'); };
        const optionFqns = {};
        for (const option of bootstrap.type.options) {
            if (option.fqn) {
                optionFqns[option.name] = option.fqn;
            }
        }
        const { renderedOptions, imports } = renderPythonOptions(indent, optionFqns, bootstrap.args);
        const importName = resolvePythonImportName(jsiiFqn, jsiiManifest);
        emit(toPythonImport(importName));
        for (const fqn of imports) {
            emit(toPythonImport(fqn));
        }
        emit();
        emit(`project = ${jsiiType.name}(${renderedOptions})`);
        emit();
        emit("project.synth()");
        fs_1.mkdirSync(path_1.dirname(pythonFile), { recursive: true });
        fs_1.writeFileSync(pythonFile, lines.join("\n"));
        this.project.logger.info(`Project definition file was created at ${pythonFile}`);
    }
}
exports.Projenrc = Projenrc;
_a = JSII_RTTI_SYMBOL_1;
Projenrc[_a] = { fqn: "projen.python.Projenrc", version: "0.69.3" };
function resolvePythonImportName(jsiiFqn, jsiiManifest) {
    const moduleName = jsiiManifest?.targets?.python?.module;
    // Module name prefix should take precedence in the event moduleName !== fqn prefix
    return moduleName
        ? [moduleName, ...jsiiFqn.split(".").slice(1)].join(".")
        : jsiiFqn;
}
exports.resolvePythonImportName = resolvePythonImportName;
function renderPythonOptions(indent, optionFqns, initOptions) {
    const imports = new Set();
    if (!initOptions || Object.keys(initOptions).length === 0) {
        // no options
        return { renderedOptions: "", imports };
    }
    const lines = [""];
    for (const [name, value] of Object.entries(initOptions)) {
        const { pythonValue, importName } = toPythonValue(value, name, optionFqns);
        if (importName)
            imports.add(importName);
        lines.push(`${toPythonProperty(name)}=${pythonValue},`);
    }
    const renderedOptions = lines
        .join(`\n${" ".repeat((indent + 1) * 4)}`)
        .concat("\n");
    return { renderedOptions, imports };
}
function toPythonProperty(prop) {
    return case_1.snake(prop);
}
function toPythonValue(value, name, optionFqns) {
    if (typeof value === "boolean") {
        return { pythonValue: value ? "True" : "False" };
    }
    else if (typeof value === "number") {
        return { pythonValue: JSON.stringify(value) };
    }
    else if (typeof value === "string") {
        if (optionFqns[name] !== undefined) {
            const parts = optionFqns[name].split(".");
            const base = parts[parts.length - 1];
            const choice = String(value).toUpperCase().replace(/-/g, "_");
            return { pythonValue: `${base}.${choice}`, importName: optionFqns[name] };
        }
        else {
            return { pythonValue: JSON.stringify(value) };
        }
    }
    else if (value === undefined || value === null) {
        return { pythonValue: "None" };
    }
    else {
        return { pythonValue: JSON.stringify(value) };
    }
}
function toPythonImport(fqn) {
    const parts = fqn.split(".");
    if (parts.length === 1) {
        return `import ${parts[0]}`;
    }
    else {
        return `from ${parts.slice(0, -1).join(".")} import ${parts[parts.length - 1]}`;
    }
}
//# sourceMappingURL=data:application/json;base64,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