"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProjectType = exports.Project = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs_1 = require("fs");
const os_1 = require("os");
const path = require("path");
const glob = require("glob");
const cleanup_1 = require("./cleanup");
const common_1 = require("./common");
const dependencies_1 = require("./dependencies");
const file_1 = require("./file");
const gitattributes_1 = require("./gitattributes");
const ignore_file_1 = require("./ignore-file");
const render_options_1 = require("./javascript/render-options");
const json_1 = require("./json");
const logger_1 = require("./logger");
const object_file_1 = require("./object-file");
const project_build_1 = require("./project-build");
const projenrc_json_1 = require("./projenrc-json");
const renovatebot_1 = require("./renovatebot");
const tasks_1 = require("./tasks");
const util_1 = require("./util");
/**
 * Base project
 */
class Project {
    constructor(options) {
        this._components = new Array();
        this.subprojects = new Array();
        this.tips = new Array();
        this.initProject = render_options_1.resolveInitProject(options);
        this.name = options.name;
        this.parent = options.parent;
        this.excludeFromCleanup = [];
        this._ejected = util_1.isTruthy(process.env.PROJEN_EJECTING);
        if (this.ejected) {
            this.projenCommand = "scripts/run-task";
        }
        else {
            this.projenCommand = options.projenCommand ?? "npx projen";
        }
        this.outdir = this.determineOutdir(options.outdir);
        this.root = this.parent ? this.parent.root : this;
        // must happen after this.outdir, this.parent and this.root are initialized
        this.parent?._addSubProject(this);
        // ------------------------------------------------------------------------
        this.gitattributes = new gitattributes_1.GitAttributesFile(this);
        this.annotateGenerated("/.projen/**"); // contents  of the .projen/ directory are generated by projen
        this.annotateGenerated(`/${this.gitattributes.path}`); // the .gitattributes file itself is generated
        if (options.gitOptions?.lfsPatterns) {
            for (const pattern of options.gitOptions.lfsPatterns) {
                this.gitattributes.addAttributes(pattern, "filter=lfs", "diff=lfs", "merge=lfs", "-text");
            }
        }
        this.gitignore = new ignore_file_1.IgnoreFile(this, ".gitignore", options.gitIgnoreOptions);
        this.gitignore.exclude("node_modules/"); // created by running `npx projen`
        this.gitignore.include(`/${this.gitattributes.path}`);
        // oh no: tasks depends on gitignore so it has to be initialized after
        // smells like dep injection but god forbid.
        this.tasks = new tasks_1.Tasks(this);
        if (!this.ejected) {
            this.defaultTask = this.tasks.addTask(Project.DEFAULT_TASK, {
                description: "Synthesize project files",
            });
            // Subtasks should call the root task for synth
            if (this.parent) {
                this.defaultTask.exec(`${this.projenCommand} ${Project.DEFAULT_TASK}`, {
                    cwd: path.relative(this.outdir, this.root.outdir),
                });
            }
            if (!this.parent) {
                this.ejectTask = this.tasks.addTask("eject", {
                    description: "Remove projen from the project",
                    env: {
                        PROJEN_EJECTING: "true",
                    },
                });
                this.ejectTask.spawn(this.defaultTask);
            }
        }
        this.projectBuild = new project_build_1.ProjectBuild(this);
        this.deps = new dependencies_1.Dependencies(this);
        this.logger = new logger_1.Logger(this, options.logging);
        const projenrcJson = options.projenrcJson ?? false;
        if (!this.parent && projenrcJson) {
            new projenrc_json_1.ProjenrcJson(this, options.projenrcJsonOptions);
        }
        if (options.renovatebot) {
            new renovatebot_1.Renovatebot(this, options.renovatebotOptions);
        }
        this.commitGenerated = options.commitGenerated ?? true;
        if (!this.ejected) {
            new json_1.JsonFile(this, cleanup_1.FILE_MANIFEST, {
                omitEmpty: true,
                obj: () => ({
                    // replace `\` with `/` to ensure paths match across platforms
                    files: this.files
                        .filter((f) => f.readonly)
                        .map((f) => f.path.replace(/\\/g, "/")),
                }),
                // This file is used by projen to track the generated files, so must be committed.
                committed: true,
            });
        }
    }
    /**
     * Returns all the components within this project.
     */
    get components() {
        return [...this._components];
    }
    /**
     * All files in this project.
     */
    get files() {
        const isFile = (c) => c instanceof file_1.FileBase;
        return this._components
            .filter(isFile)
            .sort((f1, f2) => f1.path.localeCompare(f2.path));
    }
    /**
     * Adds a new task to this project. This will fail if the project already has
     * a task with this name.
     *
     * @param name The task name to add
     * @param props Task properties
     */
    addTask(name, props = {}) {
        return this.tasks.addTask(name, props);
    }
    /**
     * Removes a task from a project.
     *
     * @param name The name of the task to remove.
     *
     * @returns The `Task` that was removed, otherwise `undefined`.
     */
    removeTask(name) {
        return this.tasks.removeTask(name);
    }
    get buildTask() {
        return this.projectBuild.buildTask;
    }
    get compileTask() {
        return this.projectBuild.compileTask;
    }
    get testTask() {
        return this.projectBuild.testTask;
    }
    get preCompileTask() {
        return this.projectBuild.preCompileTask;
    }
    get postCompileTask() {
        return this.projectBuild.postCompileTask;
    }
    get packageTask() {
        return this.projectBuild.packageTask;
    }
    /**
     * Finds a file at the specified relative path within this project and all
     * its subprojects.
     *
     * @param filePath The file path. If this path is relative, it will be resolved
     * from the root of _this_ project.
     * @returns a `FileBase` or undefined if there is no file in that path
     */
    tryFindFile(filePath) {
        const absolute = path.isAbsolute(filePath)
            ? filePath
            : path.resolve(this.outdir, filePath);
        for (const file of this.files) {
            if (absolute === file.absolutePath) {
                return file;
            }
        }
        for (const child of this.subprojects) {
            const file = child.tryFindFile(absolute);
            if (file) {
                return file;
            }
        }
        return undefined;
    }
    /**
     * Finds a json file by name.
     * @param filePath The file path.
     * @deprecated use `tryFindObjectFile`
     */
    tryFindJsonFile(filePath) {
        const file = this.tryFindObjectFile(filePath);
        if (!file) {
            return undefined;
        }
        if (!(file instanceof json_1.JsonFile)) {
            throw new Error(`found file ${filePath} but it is not a JsonFile. got: ${file.constructor.name}`);
        }
        return file;
    }
    /**
     * Finds an object file (like JsonFile, YamlFile, etc.) by name.
     * @param filePath The file path.
     */
    tryFindObjectFile(filePath) {
        const file = this.tryFindFile(filePath);
        if (!file) {
            return undefined;
        }
        if (!(file instanceof object_file_1.ObjectFile)) {
            throw new Error(`found file ${filePath} but it is not a ObjectFile. got: ${file.constructor.name}`);
        }
        return file;
    }
    /**
     * Finds a file at the specified relative path within this project and removes
     * it.
     *
     * @param filePath The file path. If this path is relative, it will be
     * resolved from the root of _this_ project.
     * @returns a `FileBase` if the file was found and removed, or undefined if
     * the file was not found.
     */
    tryRemoveFile(filePath) {
        const absolute = path.isAbsolute(filePath)
            ? filePath
            : path.resolve(this.outdir, filePath);
        const isFile = (c) => c instanceof file_1.FileBase;
        const index = this._components.findIndex((c) => isFile(c) && c.absolutePath === absolute);
        if (index !== -1) {
            return this._components.splice(index, 1)[0];
        }
        for (const child of this.subprojects) {
            const file = child.tryRemoveFile(absolute);
            if (file) {
                return file;
            }
        }
        return undefined;
    }
    /**
     * Prints a "tip" message during synthesis.
     * @param message The message
     * @deprecated - use `project.logger.info(message)` to show messages during synthesis
     */
    addTip(message) {
        this.tips.push(message);
    }
    /**
     * Exclude the matching files from pre-synth cleanup. Can be used when, for example, some
     * source files include the projen marker and we don't want them to be erased during synth.
     *
     * @param globs The glob patterns to match
     */
    addExcludeFromCleanup(...globs) {
        this.excludeFromCleanup.push(...globs);
    }
    /**
     * Returns the shell command to execute in order to run a task.
     *
     * By default, this is `npx projen@<version> <task>`
     *
     * @param task The task for which the command is required
     */
    runTaskCommand(task) {
        return `npx projen@${common_1.PROJEN_VERSION} ${task.name}`;
    }
    /**
     * Exclude these files from the bundled package. Implemented by project types based on the
     * packaging mechanism. For example, `NodeProject` delegates this to `.npmignore`.
     *
     * @param _pattern The glob pattern to exclude
     */
    addPackageIgnore(_pattern) {
        // nothing to do at the abstract level
    }
    /**
     * Adds a .gitignore pattern.
     * @param pattern The glob pattern to ignore.
     */
    addGitIgnore(pattern) {
        this.gitignore.addPatterns(pattern);
    }
    /**
     * Consider a set of files as "generated". This method is implemented by
     * derived classes and used for example, to add git attributes to tell GitHub
     * that certain files are generated.
     *
     * @param _glob the glob pattern to match (could be a file path).
     */
    annotateGenerated(_glob) {
        // nothing to do at the abstract level
    }
    /**
     * Synthesize all project files into `outdir`.
     *
     * 1. Call "this.preSynthesize()"
     * 2. Delete all generated files
     * 3. Synthesize all sub-projects
     * 4. Synthesize all components of this project
     * 5. Call "postSynthesize()" for all components of this project
     * 6. Call "this.postSynthesize()"
     */
    synth() {
        const outdir = this.outdir;
        this.logger.debug("Synthesizing project...");
        this.preSynthesize();
        for (const comp of this._components) {
            comp.preSynthesize();
        }
        // we exclude all subproject directories to ensure that when subproject.synth()
        // gets called below after cleanup(), subproject generated files are left intact
        for (const subproject of this.subprojects) {
            this.addExcludeFromCleanup(subproject.outdir + "/**");
        }
        // delete orphaned files before we start synthesizing new ones
        cleanup_1.cleanup(outdir, this.files.map((f) => f.path.replace(/\\/g, "/")), this.excludeFromCleanup);
        for (const subproject of this.subprojects) {
            subproject.synth();
        }
        for (const comp of this._components) {
            comp.synthesize();
        }
        if (!util_1.isTruthy(process.env.PROJEN_DISABLE_POST)) {
            for (const comp of this._components) {
                comp.postSynthesize();
            }
            // project-level hook
            this.postSynthesize();
        }
        if (this.ejected) {
            this.logger.debug("Ejecting project...");
            // Backup projenrc files
            const files = glob.sync(".projenrc.*", {
                cwd: this.outdir,
                dot: true,
                nodir: true,
                absolute: true,
            });
            for (const file of files) {
                fs_1.renameSync(file, `${file}.bak`);
            }
        }
        this.logger.debug("Synthesis complete");
    }
    /**
     * Whether or not the project is being ejected.
     */
    get ejected() {
        return this._ejected;
    }
    /**
     * Called before all components are synthesized.
     */
    preSynthesize() { }
    /**
     * Called after all components are synthesized. Order is *not* guaranteed.
     */
    postSynthesize() { }
    /**
     * Adds a component to the project.
     * @internal
     */
    _addComponent(component) {
        this._components.push(component);
    }
    /**
     * Adds a sub-project to this project.
     *
     * This is automatically called when a new project is created with `parent`
     * pointing to this project, so there is no real need to call this manually.
     *
     * @param sub-project The child project to add.
     * @internal
     */
    _addSubProject(subproject) {
        if (subproject.parent !== this) {
            throw new Error('"parent" of child project must be this project');
        }
        // check that `outdir` is exclusive
        for (const p of this.subprojects) {
            if (path.resolve(p.outdir) === path.resolve(subproject.outdir)) {
                throw new Error(`there is already a sub-project with "outdir": ${subproject.outdir}`);
            }
        }
        this.subprojects.push(subproject);
    }
    /**
     * Resolves the project's output directory.
     */
    determineOutdir(outdirOption) {
        if (this.parent && outdirOption && path.isAbsolute(outdirOption)) {
            throw new Error('"outdir" must be a relative path');
        }
        // if this is a subproject, it is relative to the parent
        if (this.parent) {
            if (!outdirOption) {
                throw new Error('"outdir" must be specified for subprojects');
            }
            return path.resolve(this.parent.outdir, outdirOption);
        }
        // if this is running inside a test and outdir is not explicitly set
        // use a temp directory (unless cwd is aleady under tmp)
        if (common_1.IS_TEST_RUN && !outdirOption) {
            const realCwd = fs_1.realpathSync(process.cwd());
            const realTmp = fs_1.realpathSync(os_1.tmpdir());
            if (realCwd.startsWith(realTmp)) {
                return path.resolve(realCwd, outdirOption ?? ".");
            }
            return fs_1.mkdtempSync(path.join(os_1.tmpdir(), "projen."));
        }
        return path.resolve(outdirOption ?? ".");
    }
}
exports.Project = Project;
_a = JSII_RTTI_SYMBOL_1;
Project[_a] = { fqn: "projen.Project", version: "0.69.3" };
/**
 * The name of the default task (the task executed when `projen` is run without arguments). Normally
 * this task should synthesize the project files.
 */
Project.DEFAULT_TASK = "default";
/**
 * Which type of project this is.
 *
 * @deprecated no longer supported at the base project level
 */
var ProjectType;
(function (ProjectType) {
    /**
     * This module may be a either a library or an app.
     */
    ProjectType["UNKNOWN"] = "unknown";
    /**
     * This is a library, intended to be published to a package manager and
     * consumed by other projects.
     */
    ProjectType["LIB"] = "lib";
    /**
     * This is an app (service, tool, website, etc). Its artifacts are intended to
     * be deployed or published for end-user consumption.
     */
    ProjectType["APP"] = "app";
})(ProjectType = exports.ProjectType || (exports.ProjectType = {}));
//# sourceMappingURL=data:application/json;base64,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