"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.UpgradeDependenciesSchedule = exports.UpgradeDependencies = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const component_1 = require("../component");
const dependencies_1 = require("../dependencies");
const github_1 = require("../github");
const constants_1 = require("../github/constants");
const workflow_actions_1 = require("../github/workflow-actions");
const workflows_model_1 = require("../github/workflows-model");
const release_1 = require("../release");
function context(value) {
    return `\${{ ${value} }}`;
}
const REPO = context("github.repository");
const RUN_ID = context("github.run_id");
const RUN_URL = `https://github.com/${REPO}/actions/runs/${RUN_ID}`;
const CREATE_PATCH_STEP_ID = "create_patch";
const PATCH_CREATED_OUTPUT = "patch_created";
/**
 * Upgrade node project dependencies.
 */
class UpgradeDependencies extends component_1.Component {
    constructor(project, options = {}) {
        super(project);
        /**
         * The workflows that execute the upgrades. One workflow per branch.
         */
        this.workflows = [];
        this._project = project;
        this.options = options;
        this.pullRequestTitle = options.pullRequestTitle ?? "upgrade dependencies";
        this.gitIdentity =
            options.workflowOptions?.gitIdentity ?? constants_1.DEFAULT_GITHUB_ACTIONS_USER;
        this.permissions = {
            contents: workflows_model_1.JobPermission.READ,
            ...options.workflowOptions?.permissions,
        };
        this.postBuildSteps = [];
        this.containerOptions = options.workflowOptions?.container;
        project.addDevDeps("npm-check-updates@^16");
        this.postUpgradeTask =
            project.tasks.tryFind("post-upgrade") ??
                project.tasks.addTask("post-upgrade", {
                    description: "Runs after upgrading dependencies",
                });
        this.upgradeTask = project.addTask(options.taskName ?? "upgrade", {
            // this task should not run in CI mode because its designed to
            // update package.json and lock files.
            env: { CI: "0" },
            description: this.pullRequestTitle,
            steps: { toJSON: () => this.renderTaskSteps() },
        });
        this.upgradeTask.lock(); // this task is a lazy value, so make it readonly
        if (this.upgradeTask && project.github && (options.workflow ?? true)) {
            if (options.workflowOptions?.branches) {
                for (const branch of options.workflowOptions.branches) {
                    this.workflows.push(this.createWorkflow(this.upgradeTask, project.github, branch));
                }
            }
            else if (release_1.Release.of(project)) {
                const release = release_1.Release.of(project);
                release._forEachBranch((branch) => {
                    this.workflows.push(this.createWorkflow(this.upgradeTask, project.github, branch));
                });
            }
            else {
                // represents the default repository branch.
                // just like not specifying anything.
                const defaultBranch = undefined;
                this.workflows.push(this.createWorkflow(this.upgradeTask, project.github, defaultBranch));
            }
        }
    }
    /**
     * Add steps to execute a successful build.
     * @param steps workflow steps
     */
    addPostBuildSteps(...steps) {
        this.postBuildSteps.push(...steps);
    }
    renderTaskSteps() {
        const exclude = this.options.exclude ?? [];
        // exclude depedencies that has already version pinned (fully or with patch version) by Projen with ncu (but not package manager upgrade)
        // Getting only unique values through set
        const ncuExcludes = [
            ...new Set(this.project.deps.all
                .filter((dep) => dep.version &&
                dep.version[0] !== "^" &&
                dep.type !== dependencies_1.DependencyType.OVERRIDE)
                .map((dep) => dep.name)
                .concat(exclude)),
        ];
        const ncuIncludes = this.options.include?.filter((item) => !ncuExcludes.includes(item));
        const includeLength = this.options.include?.length ?? 0;
        const ncuIncludesLength = ncuIncludes?.length ?? 0;
        // If all explicit includes already have version pinned, don't add task.
        // Note that without explicit includes task gets added
        if (includeLength > 0 && ncuIncludesLength === 0) {
            return [{ exec: "echo No dependencies to upgrade." }];
        }
        const steps = new Array();
        // update npm-check-updates before everything else, in case there is a bug
        // in it or one of its dependencies. This will make upgrade workflows
        // slightly more stable and resilient to upstream changes.
        steps.push({
            exec: this._project.package.renderUpgradePackagesCommand([], ["npm-check-updates"]),
        });
        for (const dep of ["dev", "optional", "peer", "prod", "bundle"]) {
            const ncuCommand = [
                "npm-check-updates",
                "--dep",
                dep,
                "--upgrade",
                "--target=minor",
            ];
            // Don't add includes and excludes same time
            if (ncuIncludes) {
                ncuCommand.push(`--filter='${ncuIncludes.join(",")}'`);
            }
            else if (ncuExcludes.length > 0) {
                ncuCommand.push(`--reject='${ncuExcludes.join(",")}'`);
            }
            steps.push({ exec: ncuCommand.join(" ") });
        }
        // run "yarn/npm install" to update the lockfile and install any deps (such as projen)
        steps.push({ exec: this._project.package.installAndUpdateLockfileCommand });
        // run upgrade command to upgrade transitive deps as well
        steps.push({
            exec: this._project.package.renderUpgradePackagesCommand(exclude, this.options.include),
        });
        // run "projen" to give projen a chance to update dependencies (it will also run "yarn install")
        steps.push({ exec: this._project.projenCommand });
        steps.push({ spawn: this.postUpgradeTask.name });
        return steps;
    }
    createWorkflow(task, github, branch) {
        const schedule = this.options.workflowOptions?.schedule ??
            UpgradeDependenciesSchedule.DAILY;
        const workflowName = `${task.name}${branch ? `-${branch.replace(/\//g, "-")}` : ""}`;
        const workflow = github.addWorkflow(workflowName);
        const triggers = {
            workflowDispatch: {},
            schedule: schedule.cron.length > 0
                ? schedule.cron.map((e) => ({ cron: e }))
                : undefined,
        };
        workflow.on(triggers);
        const upgrade = this.createUpgrade(task, github, branch);
        const pr = this.createPr(workflow, upgrade);
        const jobs = {};
        jobs[upgrade.jobId] = upgrade.job;
        jobs[pr.jobId] = pr.job;
        workflow.addJobs(jobs);
        return workflow;
    }
    createUpgrade(task, github, branch) {
        const runsOn = this.options.workflowOptions?.runsOn ?? ["ubuntu-latest"];
        const with_ = {
            ...(branch ? { ref: branch } : {}),
            ...(github.downloadLfs ? { lfs: true } : {}),
        };
        const steps = [
            {
                name: "Checkout",
                uses: "actions/checkout@v3",
                with: Object.keys(with_).length > 0 ? with_ : undefined,
            },
            ...this._project.renderWorkflowSetup({ mutable: false }),
            {
                name: "Upgrade dependencies",
                run: this._project.runTaskCommand(task),
            },
        ];
        steps.push(...this.postBuildSteps);
        steps.push(...workflow_actions_1.WorkflowActions.createUploadGitPatch({
            stepId: CREATE_PATCH_STEP_ID,
            outputName: PATCH_CREATED_OUTPUT,
        }));
        return {
            job: {
                name: "Upgrade",
                container: this.containerOptions,
                permissions: this.permissions,
                runsOn: runsOn ?? ["ubuntu-latest"],
                steps: steps,
                outputs: {
                    [PATCH_CREATED_OUTPUT]: {
                        stepId: CREATE_PATCH_STEP_ID,
                        outputName: PATCH_CREATED_OUTPUT,
                    },
                },
            },
            jobId: "upgrade",
            ref: branch,
        };
    }
    createPr(workflow, upgrade) {
        // default to API access method used by all GitHub workflows, unless a
        // custom one is specified
        const apiAccess = this.options.workflowOptions?.projenCredentials ??
            workflow.projenCredentials;
        const token = apiAccess.tokenRef;
        const runsOn = this.options.workflowOptions?.runsOn ?? ["ubuntu-latest"];
        const workflowName = workflow.name;
        const branchName = `github-actions/${workflowName}`;
        const prStepId = "create-pr";
        const title = `chore(deps): ${this.pullRequestTitle}`;
        const description = [
            "Upgrades project dependencies. See details in [workflow run].",
            "",
            `[Workflow Run]: ${RUN_URL}`,
            "",
            "------",
            "",
            `*Automatically created by projen via the "${workflow.name}" workflow*`,
        ].join("\n");
        const committer = `${this.gitIdentity.name} <${this.gitIdentity.email}>`;
        const steps = [
            ...apiAccess.setupSteps,
            ...workflow_actions_1.WorkflowActions.checkoutWithPatch({
                ref: upgrade.ref,
            }),
            ...workflow_actions_1.WorkflowActions.setGitIdentity(this.gitIdentity),
            {
                name: "Create Pull Request",
                id: prStepId,
                uses: "peter-evans/create-pull-request@v4",
                with: {
                    // the pr can modify workflow files, so we need to use the custom
                    // secret if one is configured.
                    token: token,
                    "commit-message": `${title}\n\n${description}`,
                    branch: branchName,
                    title: title,
                    labels: this.options.workflowOptions?.labels?.join(",") || undefined,
                    assignees: this.options.workflowOptions?.assignees?.join(",") || undefined,
                    body: description,
                    author: committer,
                    committer: committer,
                    signoff: this.options.signoff ?? true,
                },
            },
        ];
        return {
            job: {
                name: "Create Pull Request",
                if: `\${{ needs.${upgrade.jobId}.outputs.${PATCH_CREATED_OUTPUT} }}`,
                needs: [upgrade.jobId],
                permissions: {
                    contents: github_1.workflows.JobPermission.READ,
                },
                runsOn: runsOn ?? ["ubuntu-latest"],
                steps: steps,
            },
            jobId: "pr",
        };
    }
}
exports.UpgradeDependencies = UpgradeDependencies;
_a = JSII_RTTI_SYMBOL_1;
UpgradeDependencies[_a] = { fqn: "projen.javascript.UpgradeDependencies", version: "0.69.3" };
/**
 * How often to check for new versions and raise pull requests for version upgrades.
 */
class UpgradeDependenciesSchedule {
    constructor(cron) {
        this.cron = cron;
    }
    /**
     * Create a schedule from a raw cron expression.
     */
    static expressions(cron) {
        return new UpgradeDependenciesSchedule(cron);
    }
}
exports.UpgradeDependenciesSchedule = UpgradeDependenciesSchedule;
_b = JSII_RTTI_SYMBOL_1;
UpgradeDependenciesSchedule[_b] = { fqn: "projen.javascript.UpgradeDependenciesSchedule", version: "0.69.3" };
/**
 * Disables automatic upgrades.
 */
UpgradeDependenciesSchedule.NEVER = new UpgradeDependenciesSchedule([]);
/**
 * At 00:00.
 */
UpgradeDependenciesSchedule.DAILY = new UpgradeDependenciesSchedule(["0 0 * * *"]);
/**
 * At 00:00 on every day-of-week from Monday through Friday.
 */
UpgradeDependenciesSchedule.WEEKDAY = new UpgradeDependenciesSchedule([
    "0 0 * * 1-5",
]);
/**
 * At 00:00 on Monday.
 */
UpgradeDependenciesSchedule.WEEKLY = new UpgradeDependenciesSchedule([
    "0 0 * * 1",
]);
/**
 * At 00:00 on day-of-month 1.
 */
UpgradeDependenciesSchedule.MONTHLY = new UpgradeDependenciesSchedule([
    "0 0 1 * *",
]);
//# sourceMappingURL=data:application/json;base64,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