"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NodeProject = exports.AutoRelease = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path_1 = require("path");
const bundler_1 = require("./bundler");
const jest_1 = require("./jest");
const node_package_1 = require("./node-package");
const projenrc_1 = require("./projenrc");
const build_1 = require("../build");
const common_1 = require("../common");
const github_1 = require("../github");
const constants_1 = require("../github/constants");
const util_1 = require("../github/util");
const workflows_model_1 = require("../github/workflows-model");
const ignore_file_1 = require("../ignore-file");
const javascript_1 = require("../javascript");
const license_1 = require("../license");
const release_1 = require("../release");
const util_2 = require("../util");
const version_1 = require("../version");
const PROJEN_SCRIPT = "projen";
/**
 * Automatic bump modes
 */
var AutoRelease;
(function (AutoRelease) {
    /**
     * Automatically bump & release a new version for every commit to "main"
     */
    AutoRelease[AutoRelease["EVERY_COMMIT"] = 0] = "EVERY_COMMIT";
    /**
     * Automatically bump & release a new version on a daily basis.
     */
    AutoRelease[AutoRelease["DAILY"] = 1] = "DAILY";
})(AutoRelease = exports.AutoRelease || (exports.AutoRelease = {}));
/**
 * Node.js project
 *
 * @pjid node
 */
class NodeProject extends github_1.GitHubProject {
    constructor(options) {
        super(options);
        this.package = new node_package_1.NodePackage(this, options);
        this.workflowBootstrapSteps = options.workflowBootstrapSteps ?? [];
        this.workflowGitIdentity =
            options.workflowGitIdentity ?? constants_1.DEFAULT_GITHUB_ACTIONS_USER;
        this.artifactsDirectory = options.artifactsDirectory ?? "dist";
        this.artifactsJavascriptDirectory = path_1.join(this.artifactsDirectory, "js");
        this.runScriptCommand = (() => {
            switch (this.packageManager) {
                case node_package_1.NodePackageManager.NPM:
                    return "npm run";
                case node_package_1.NodePackageManager.YARN:
                case node_package_1.NodePackageManager.YARN2:
                    return "yarn run";
                case node_package_1.NodePackageManager.PNPM:
                    return "pnpm run";
                default:
                    throw new Error(`unexpected package manager ${this.packageManager}`);
            }
        })();
        this.nodeVersion =
            options.workflowNodeVersion ?? this.package.minNodeVersion;
        // add PATH for all tasks which includes the project's npm .bin list
        this.tasks.addEnvironment("PATH", '$(npx -c "node -e \\"console.log(process.env.PATH)\\"")');
        this.addLicense(options);
        if (options.npmignoreEnabled ?? true) {
            this.npmignore = new ignore_file_1.IgnoreFile(this, ".npmignore", options.npmIgnoreOptions);
        }
        this.addDefaultGitIgnore();
        if (options.gitignore?.length) {
            for (const i of options.gitignore) {
                this.gitignore.exclude(i);
            }
        }
        if (options.npmignore?.length) {
            if (!this.npmignore) {
                throw new Error('.npmignore is not defined for an APP project type. Add "npmIgnore: true" to override this');
            }
            for (const i of options.npmignore) {
                this.npmignore.exclude(i);
            }
        }
        if (!this.ejected) {
            this.setScript(PROJEN_SCRIPT, this.package.projenCommand);
        }
        this.npmignore?.exclude(`/${common_1.PROJEN_RC}`);
        this.npmignore?.exclude(`/${common_1.PROJEN_DIR}/`);
        this.gitignore.include(`/${common_1.PROJEN_RC}`);
        const projen = options.projenDevDependency ?? true;
        if (projen && !this.ejected) {
            const postfix = options.projenVersion ? `@${options.projenVersion}` : "";
            this.addDevDeps(`projen${postfix}`);
        }
        if (!options.defaultReleaseBranch) {
            throw new Error('"defaultReleaseBranch" is temporarily a required option while we migrate its default value from "master" to "main"');
        }
        const buildEnabled = options.buildWorkflow ?? (this.parent ? false : true);
        // configure jest if enabled
        // must be before the build/release workflows
        if (options.jest ?? true) {
            this.jest = new jest_1.Jest(this, options.jestOptions);
        }
        const requiresIdTokenPermission = (options.scopedPackagesOptions ?? []).length > 0 &&
            options.codeArtifactOptions?.authProvider ===
                release_1.CodeArtifactAuthProvider.GITHUB_OIDC;
        const workflowPermissions = {
            idToken: requiresIdTokenPermission ? workflows_model_1.JobPermission.WRITE : undefined,
        };
        if (buildEnabled && this.github) {
            this.buildWorkflow = new build_1.BuildWorkflow(this, {
                buildTask: this.buildTask,
                artifactsDirectory: this.artifactsDirectory,
                containerImage: options.workflowContainerImage,
                gitIdentity: this.workflowGitIdentity,
                mutableBuild: options.mutableBuild,
                preBuildSteps: this.renderWorkflowSetup({
                    mutable: options.mutableBuild ?? true,
                }),
                postBuildSteps: options.postBuildSteps,
                runsOn: options.workflowRunsOn,
                workflowTriggers: options.buildWorkflowTriggers,
                permissions: workflowPermissions,
            });
            this.buildWorkflow.addPostBuildSteps(...this.renderUploadCoverageJobStep(options));
        }
        const release = options.release ??
            options.releaseWorkflow ??
            (this.parent ? false : true);
        if (release) {
            this.addDevDeps(version_1.Version.STANDARD_VERSION);
            this.release = new release_1.Release(this, {
                versionFile: "package.json",
                task: this.buildTask,
                branch: options.defaultReleaseBranch ?? "main",
                artifactsDirectory: this.artifactsDirectory,
                ...options,
                releaseWorkflowSetupSteps: [
                    ...this.renderWorkflowSetup({ mutable: false }),
                    ...(options.releaseWorkflowSetupSteps ?? []),
                ],
                postBuildSteps: [
                    ...(options.postBuildSteps ?? []),
                    ...this.renderUploadCoverageJobStep(options),
                ],
                workflowNodeVersion: this.nodeVersion,
                workflowPermissions,
            });
            this.publisher = this.release.publisher;
            const nodePackageToReleaseCodeArtifactAuthProviderMapping = {
                [node_package_1.CodeArtifactAuthProvider.ACCESS_AND_SECRET_KEY_PAIR]: release_1.CodeArtifactAuthProvider.ACCESS_AND_SECRET_KEY_PAIR,
                [node_package_1.CodeArtifactAuthProvider.GITHUB_OIDC]: release_1.CodeArtifactAuthProvider.GITHUB_OIDC,
            };
            if (options.releaseToNpm ?? false) {
                const codeArtifactOptions = release_1.isAwsCodeArtifactRegistry(this.package.npmRegistry)
                    ? {
                        accessKeyIdSecret: options.codeArtifactOptions?.accessKeyIdSecret,
                        secretAccessKeySecret: options.codeArtifactOptions?.secretAccessKeySecret,
                        roleToAssume: options.codeArtifactOptions?.roleToAssume,
                        authProvider: options.codeArtifactOptions?.authProvider
                            ? nodePackageToReleaseCodeArtifactAuthProviderMapping[options.codeArtifactOptions.authProvider]
                            : release_1.CodeArtifactAuthProvider.ACCESS_AND_SECRET_KEY_PAIR,
                    }
                    : {};
                this.release.publisher.publishToNpm({
                    registry: this.package.npmRegistry,
                    npmTokenSecret: this.package.npmTokenSecret,
                    codeArtifactOptions,
                });
            }
        }
        else {
            // validate that no release options are selected if the release workflow is disabled.
            if (options.releaseToNpm) {
                throw new Error('"releaseToNpm" is not supported if "release" is not set');
            }
            if (options.releaseEveryCommit) {
                throw new Error('"releaseEveryCommit" is not supported if "release" is not set');
            }
            if (options.releaseSchedule) {
                throw new Error('"releaseSchedule" is not supported if "release" is not set');
            }
        }
        if ((options.autoMerge ?? true) &&
            this.github?.mergify &&
            this.buildWorkflow?.buildJobIds) {
            this.autoMerge = new github_1.AutoMerge(this.github, options.autoMergeOptions);
            this.autoMerge.addConditionsLater({
                render: () => this.buildWorkflow?.buildJobIds.map((id) => `status-success=${id}`) ??
                    [],
            });
        }
        const dependabot = options.dependabot ?? false;
        const depsUpgrade = options.depsUpgrade ?? !dependabot;
        if (dependabot && depsUpgrade) {
            throw new Error("'dependabot' cannot be configured together with 'depsUpgrade'");
        }
        const depsAutoApprove = options.autoApproveUpgrades ?? false;
        if (depsAutoApprove && !this.autoApprove && this.github) {
            throw new Error("Automatic approval of dependencies upgrades requires configuring `autoApproveOptions`");
        }
        const autoApproveLabel = (condition) => condition && this.autoApprove?.label
            ? [this.autoApprove.label]
            : undefined;
        if (dependabot) {
            const defaultOptions = {
                labels: autoApproveLabel(depsAutoApprove),
            };
            this.github?.addDependabot(util_2.deepMerge([defaultOptions, options.dependabotOptions ?? {}]));
        }
        if (depsUpgrade) {
            const defaultOptions = {
                workflowOptions: {
                    container: options.workflowContainerImage
                        ? {
                            image: options.workflowContainerImage,
                        }
                        : undefined,
                    labels: autoApproveLabel(depsAutoApprove),
                    gitIdentity: this.workflowGitIdentity,
                    permissions: workflowPermissions,
                },
            };
            this.upgradeWorkflow = new javascript_1.UpgradeDependencies(this, util_2.deepMerge([defaultOptions, options.depsUpgradeOptions ?? {}]));
        }
        if (options.pullRequestTemplate ?? true) {
            this.github?.addPullRequestTemplate(...(options.pullRequestTemplateContents ?? []));
        }
        const projenrcJs = options.projenrcJs ?? !options.projenrcJson;
        if (!this.parent && projenrcJs) {
            new projenrc_1.Projenrc(this, options.projenrcJsOptions);
        }
        // add a bundler component - this enables things like Lambda bundling and in the future web bundling.
        this.bundler = new bundler_1.Bundler(this, options.bundlerOptions);
        if (options.package ?? true) {
            this.packageTask.exec(`mkdir -p ${this.artifactsJavascriptDirectory}`);
            // always use npm here - yarn doesn't add much value
            // sadly we cannot use --pack-destination because it is not supported by older npm
            this.packageTask.exec(`mv $(npm pack) ${this.artifactsJavascriptDirectory}/`);
        }
        if (options.prettier ?? false) {
            this.prettier = new javascript_1.Prettier(this, { ...options.prettierOptions });
        }
    }
    /**
     * @deprecated use `package.allowLibraryDependencies`
     */
    get allowLibraryDependencies() {
        return this.package.allowLibraryDependencies;
    }
    /**
     * @deprecated use `package.entrypoint`
     */
    get entrypoint() {
        return this.package.entrypoint;
    }
    /**
     * Minimum node.js version required by this package.
     */
    get minNodeVersion() {
        return this.package.minNodeVersion;
    }
    /**
     * Maximum node version required by this package.
     */
    get maxNodeVersion() {
        return this.package.maxNodeVersion;
    }
    /**
     * The package manager to use.
     *
     * @deprecated use `package.packageManager`
     */
    get packageManager() {
        return this.package.packageManager;
    }
    /**
     * @deprecated use `package.addField(x, y)`
     */
    get manifest() {
        return this.package.manifest;
    }
    renderUploadCoverageJobStep(options) {
        // run codecov if enabled or a secret token name is passed in
        // AND jest must be configured
        if ((options.codeCov || options.codeCovTokenSecret) && this.jest?.config) {
            return [
                {
                    name: "Upload coverage to Codecov",
                    uses: "codecov/codecov-action@v3",
                    with: options.codeCovTokenSecret
                        ? {
                            token: `\${{ secrets.${options.codeCovTokenSecret} }}`,
                            directory: this.jest.config.coverageDirectory,
                        }
                        : {
                            directory: this.jest.config.coverageDirectory,
                        },
                },
            ];
        }
        else {
            return [];
        }
    }
    addBins(bins) {
        this.package.addBin(bins);
    }
    /**
     * Replaces the contents of an npm package.json script.
     *
     * @param name The script name
     * @param command The command to execute
     */
    setScript(name, command) {
        this.package.setScript(name, command);
    }
    /**
     * Removes the npm script (always successful).
     * @param name The name of the script.
     */
    removeScript(name) {
        this.package.removeScript(name);
    }
    /**
     * Indicates if a script by the name name is defined.
     * @param name The name of the script
     */
    hasScript(name) {
        return this.package.hasScript(name);
    }
    /**
     * DEPRECATED
     * @deprecated use `project.compileTask.exec()`
     */
    addCompileCommand(...commands) {
        for (const c of commands) {
            this.compileTask.exec(c);
        }
    }
    /**
     * DEPRECATED
     * @deprecated use `project.testTask.exec()`
     */
    addTestCommand(...commands) {
        for (const c of commands) {
            this.testTask.exec(c);
        }
    }
    /**
     * Directly set fields in `package.json`.
     * @param fields The fields to set
     */
    addFields(fields) {
        for (const [name, value] of Object.entries(fields)) {
            this.package.addField(name, value);
        }
    }
    /**
     * Adds keywords to package.json (deduplicated)
     * @param keywords The keywords to add
     */
    addKeywords(...keywords) {
        this.package.addKeywords(...keywords);
    }
    /**
     * Get steps for scoped package access
     *
     * @param codeArtifactOptions Details of logging in to AWS
     * @returns array of job steps required for each private scoped packages
     */
    getScopedPackageSteps(codeArtifactOptions) {
        const parsedCodeArtifactOptions = {
            accessKeyIdSecret: codeArtifactOptions?.accessKeyIdSecret ?? "AWS_ACCESS_KEY_ID",
            secretAccessKeySecret: codeArtifactOptions?.secretAccessKeySecret ?? "AWS_SECRET_ACCESS_KEY",
            roleToAssume: codeArtifactOptions?.roleToAssume,
            authProvider: codeArtifactOptions?.authProvider,
        };
        if (parsedCodeArtifactOptions.authProvider ===
            node_package_1.CodeArtifactAuthProvider.GITHUB_OIDC) {
            return [
                {
                    name: "Configure AWS Credentials",
                    uses: "aws-actions/configure-aws-credentials@v2",
                    with: {
                        "aws-region": "us-east-2",
                        "role-to-assume": parsedCodeArtifactOptions.roleToAssume,
                        "role-duration-seconds": 900,
                    },
                },
                {
                    name: "AWS CodeArtifact Login",
                    run: `${this.runScriptCommand} ca:login`,
                },
            ];
        }
        if (parsedCodeArtifactOptions.roleToAssume) {
            return [
                {
                    name: "Configure AWS Credentials",
                    uses: "aws-actions/configure-aws-credentials@v2",
                    with: {
                        "aws-access-key-id": util_1.secretToString(parsedCodeArtifactOptions.accessKeyIdSecret),
                        "aws-secret-access-key": util_1.secretToString(parsedCodeArtifactOptions.secretAccessKeySecret),
                        "aws-region": "us-east-2",
                        "role-to-assume": parsedCodeArtifactOptions.roleToAssume,
                        "role-duration-seconds": 900,
                    },
                },
                {
                    name: "AWS CodeArtifact Login",
                    run: `${this.runScriptCommand} ca:login`,
                },
            ];
        }
        return [
            {
                name: "AWS CodeArtifact Login",
                run: `${this.runScriptCommand} ca:login`,
                env: {
                    AWS_ACCESS_KEY_ID: util_1.secretToString(parsedCodeArtifactOptions.accessKeyIdSecret),
                    AWS_SECRET_ACCESS_KEY: util_1.secretToString(parsedCodeArtifactOptions.secretAccessKeySecret),
                },
            },
        ];
    }
    /**
     * Returns the set of workflow steps which should be executed to bootstrap a
     * workflow.
     *
     * @param options Options.
     * @returns Job steps
     */
    renderWorkflowSetup(options = {}) {
        const install = new Array();
        // first run the workflow bootstrap steps
        install.push(...this.workflowBootstrapSteps);
        if (this.nodeVersion) {
            install.push({
                name: "Setup Node.js",
                uses: "actions/setup-node@v3",
                with: { "node-version": this.nodeVersion },
            });
        }
        if (this.package.packageManager === node_package_1.NodePackageManager.PNPM) {
            install.push({
                name: "Setup pnpm",
                uses: "pnpm/action-setup@v2.2.4",
                with: { version: this.package.pnpmVersion },
            });
        }
        const mutable = options.mutable ?? false;
        if (this.package.scopedPackagesOptions) {
            install.push(...this.getScopedPackageSteps(this.package.codeArtifactOptions));
        }
        install.push({
            name: "Install dependencies",
            run: mutable
                ? this.package.installAndUpdateLockfileCommand
                : this.package.installCommand,
        });
        return install;
    }
    /**
     * Defines normal dependencies.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addDeps(...deps) {
        return this.package.addDeps(...deps);
    }
    /**
     * Defines development/test dependencies.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addDevDeps(...deps) {
        return this.package.addDevDeps(...deps);
    }
    /**
     * Defines peer dependencies.
     *
     * When adding peer dependencies, a devDependency will also be added on the
     * pinned version of the declared peer. This will ensure that you are testing
     * your code against the minimum version required from your consumers.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addPeerDeps(...deps) {
        return this.package.addPeerDeps(...deps);
    }
    /**
     * Defines bundled dependencies.
     *
     * Bundled dependencies will be added as normal dependencies as well as to the
     * `bundledDependencies` section of your `package.json`.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addBundledDeps(...deps) {
        return this.package.addBundledDeps(...deps);
    }
    addPackageIgnore(pattern) {
        this.npmignore?.addPatterns(pattern);
    }
    addLicense(options) {
        if (this.package.license) {
            new license_1.License(this, {
                spdx: this.package.license,
                copyrightOwner: options.copyrightOwner ?? options.authorName,
                copyrightPeriod: options.copyrightPeriod,
            });
        }
    }
    addDefaultGitIgnore() {
        this.gitignore.exclude("# Logs", "logs", "*.log", "npm-debug.log*", "yarn-debug.log*", "yarn-error.log*", "lerna-debug.log*", "# Diagnostic reports (https://nodejs.org/api/report.html)", "report.[0-9]*.[0-9]*.[0-9]*.[0-9]*.json", "# Runtime data", "pids", "*.pid", "*.seed", "*.pid.lock", "# Directory for instrumented libs generated by jscoverage/JSCover", "lib-cov", "# Coverage directory used by tools like istanbul", "coverage", "*.lcov", "# nyc test coverage", ".nyc_output", "# Compiled binary addons (https://nodejs.org/api/addons.html)", "build/Release", "# Dependency directories", "node_modules/", "jspm_packages/", "# TypeScript cache", "*.tsbuildinfo", "# Optional eslint cache", ".eslintcache", "# Output of 'npm pack'", "*.tgz", "# Yarn Integrity file", ".yarn-integrity", "# parcel-bundler cache (https://parceljs.org/)", ".cache");
    }
    /**
     * Returns the shell command to execute in order to run a task. This will
     * typically be `npx projen TASK`.
     *
     * @param task The task for which the command is required
     */
    runTaskCommand(task) {
        return `${this.package.projenCommand} ${task.name}`;
    }
    /**
     * The job ID of the build workflow.
     */
    get buildWorkflowJobId() {
        return this.buildWorkflow?.buildJobIds[0];
    }
}
exports.NodeProject = NodeProject;
_a = JSII_RTTI_SYMBOL_1;
NodeProject[_a] = { fqn: "projen.javascript.NodeProject", version: "0.69.3" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibm9kZS1wcm9qZWN0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vc3JjL2phdmFzY3JpcHQvbm9kZS1wcm9qZWN0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUEsK0JBQTRCO0FBQzVCLHVDQUFvRDtBQUNwRCxpQ0FBMkM7QUFDM0MsaURBTXdCO0FBQ3hCLHlDQUF1RDtBQUN2RCxvQ0FBeUM7QUFDekMsc0NBQWtEO0FBQ2xELHNDQU1tQjtBQUNuQixtREFBa0U7QUFDbEUseUNBQWdEO0FBQ2hELCtEQUttQztBQUNuQyxnREFBK0Q7QUFDL0QsOENBS3VCO0FBQ3ZCLHdDQUFxQztBQUNyQyx3Q0FRb0I7QUFFcEIsa0NBQW9DO0FBQ3BDLHdDQUFxQztBQUVyQyxNQUFNLGFBQWEsR0FBRyxRQUFRLENBQUM7QUFtUS9COztHQUVHO0FBQ0gsSUFBWSxXQVVYO0FBVkQsV0FBWSxXQUFXO0lBQ3JCOztPQUVHO0lBQ0gsNkRBQVksQ0FBQTtJQUVaOztPQUVHO0lBQ0gsK0NBQUssQ0FBQTtBQUNQLENBQUMsRUFWVyxXQUFXLEdBQVgsbUJBQVcsS0FBWCxtQkFBVyxRQVV0QjtBQUVEOzs7O0dBSUc7QUFDSCxNQUFhLFdBQVksU0FBUSxzQkFBYTtJQWlINUMsWUFBWSxPQUEyQjtRQUNyQyxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUM7UUFFZixJQUFJLENBQUMsT0FBTyxHQUFHLElBQUksMEJBQVcsQ0FBQyxJQUFJLEVBQUUsT0FBTyxDQUFDLENBQUM7UUFDOUMsSUFBSSxDQUFDLHNCQUFzQixHQUFHLE9BQU8sQ0FBQyxzQkFBc0IsSUFBSSxFQUFFLENBQUM7UUFDbkUsSUFBSSxDQUFDLG1CQUFtQjtZQUN0QixPQUFPLENBQUMsbUJBQW1CLElBQUksdUNBQTJCLENBQUM7UUFDN0QsSUFBSSxDQUFDLGtCQUFrQixHQUFHLE9BQU8sQ0FBQyxrQkFBa0IsSUFBSSxNQUFNLENBQUM7UUFDL0QsSUFBSSxDQUFDLDRCQUE0QixHQUFHLFdBQUksQ0FBQyxJQUFJLENBQUMsa0JBQWtCLEVBQUUsSUFBSSxDQUFDLENBQUM7UUFFeEUsSUFBSSxDQUFDLGdCQUFnQixHQUFHLENBQUMsR0FBRyxFQUFFO1lBQzVCLFFBQVEsSUFBSSxDQUFDLGNBQWMsRUFBRTtnQkFDM0IsS0FBSyxpQ0FBa0IsQ0FBQyxHQUFHO29CQUN6QixPQUFPLFNBQVMsQ0FBQztnQkFDbkIsS0FBSyxpQ0FBa0IsQ0FBQyxJQUFJLENBQUM7Z0JBQzdCLEtBQUssaUNBQWtCLENBQUMsS0FBSztvQkFDM0IsT0FBTyxVQUFVLENBQUM7Z0JBQ3BCLEtBQUssaUNBQWtCLENBQUMsSUFBSTtvQkFDMUIsT0FBTyxVQUFVLENBQUM7Z0JBQ3BCO29CQUNFLE1BQU0sSUFBSSxLQUFLLENBQUMsOEJBQThCLElBQUksQ0FBQyxjQUFjLEVBQUUsQ0FBQyxDQUFDO2FBQ3hFO1FBQ0gsQ0FBQyxDQUFDLEVBQUUsQ0FBQztRQUVMLElBQUksQ0FBQyxXQUFXO1lBQ2QsT0FBTyxDQUFDLG1CQUFtQixJQUFJLElBQUksQ0FBQyxPQUFPLENBQUMsY0FBYyxDQUFDO1FBRTdELG9FQUFvRTtRQUNwRSxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQWMsQ0FDdkIsTUFBTSxFQUNOLHlEQUF5RCxDQUMxRCxDQUFDO1FBRUYsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUV6QixJQUFJLE9BQU8sQ0FBQyxnQkFBZ0IsSUFBSSxJQUFJLEVBQUU7WUFDcEMsSUFBSSxDQUFDLFNBQVMsR0FBRyxJQUFJLHdCQUFVLENBQzdCLElBQUksRUFDSixZQUFZLEVBQ1osT0FBTyxDQUFDLGdCQUFnQixDQUN6QixDQUFDO1NBQ0g7UUFFRCxJQUFJLENBQUMsbUJBQW1CLEVBQUUsQ0FBQztRQUUzQixJQUFJLE9BQU8sQ0FBQyxTQUFTLEVBQUUsTUFBTSxFQUFFO1lBQzdCLEtBQUssTUFBTSxDQUFDLElBQUksT0FBTyxDQUFDLFNBQVMsRUFBRTtnQkFDakMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUM7YUFDM0I7U0FDRjtRQUVELElBQUksT0FBTyxDQUFDLFNBQVMsRUFBRSxNQUFNLEVBQUU7WUFDN0IsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUU7Z0JBQ25CLE1BQU0sSUFBSSxLQUFLLENBQ2IsMkZBQTJGLENBQzVGLENBQUM7YUFDSDtZQUVELEtBQUssTUFBTSxDQUFDLElBQUksT0FBTyxDQUFDLFNBQVMsRUFBRTtnQkFDakMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUM7YUFDM0I7U0FDRjtRQUVELElBQUksQ0FBQyxJQUFJLENBQUMsT0FBTyxFQUFFO1lBQ2pCLElBQUksQ0FBQyxTQUFTLENBQUMsYUFBYSxFQUFFLElBQUksQ0FBQyxPQUFPLENBQUMsYUFBYSxDQUFDLENBQUM7U0FDM0Q7UUFFRCxJQUFJLENBQUMsU0FBUyxFQUFFLE9BQU8sQ0FBQyxJQUFJLGtCQUFTLEVBQUUsQ0FBQyxDQUFDO1FBQ3pDLElBQUksQ0FBQyxTQUFTLEVBQUUsT0FBTyxDQUFDLElBQUksbUJBQVUsR0FBRyxDQUFDLENBQUM7UUFDM0MsSUFBSSxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUMsSUFBSSxrQkFBUyxFQUFFLENBQUMsQ0FBQztRQUV4QyxNQUFNLE1BQU0sR0FBRyxPQUFPLENBQUMsbUJBQW1CLElBQUksSUFBSSxDQUFDO1FBQ25ELElBQUksTUFBTSxJQUFJLENBQUMsSUFBSSxDQUFDLE9BQU8sRUFBRTtZQUMzQixNQUFNLE9BQU8sR0FBRyxPQUFPLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQyxJQUFJLE9BQU8sQ0FBQyxhQUFhLEVBQUUsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDO1lBQ3pFLElBQUksQ0FBQyxVQUFVLENBQUMsU0FBUyxPQUFPLEVBQUUsQ0FBQyxDQUFDO1NBQ3JDO1FBRUQsSUFBSSxDQUFDLE9BQU8sQ0FBQyxvQkFBb0IsRUFBRTtZQUNqQyxNQUFNLElBQUksS0FBSyxDQUNiLG9IQUFvSCxDQUNySCxDQUFDO1NBQ0g7UUFFRCxNQUFNLFlBQVksR0FBRyxPQUFPLENBQUMsYUFBYSxJQUFJLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUUzRSw0QkFBNEI7UUFDNUIsNkNBQTZDO1FBQzdDLElBQUksT0FBTyxDQUFDLElBQUksSUFBSSxJQUFJLEVBQUU7WUFDeEIsSUFBSSxDQUFDLElBQUksR0FBRyxJQUFJLFdBQUksQ0FBQyxJQUFJLEVBQUUsT0FBTyxDQUFDLFdBQVcsQ0FBQyxDQUFDO1NBQ2pEO1FBRUQsTUFBTSx5QkFBeUIsR0FDN0IsQ0FBQyxPQUFPLENBQUMscUJBQXFCLElBQUksRUFBRSxDQUFDLENBQUMsTUFBTSxHQUFHLENBQUM7WUFDaEQsT0FBTyxDQUFDLG1CQUFtQixFQUFFLFlBQVk7Z0JBQ3ZDLGtDQUF3QixDQUFDLFdBQVcsQ0FBQztRQUV6QyxNQUFNLG1CQUFtQixHQUFtQjtZQUMxQyxPQUFPLEVBQUUseUJBQXlCLENBQUMsQ0FBQyxDQUFDLCtCQUFhLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxTQUFTO1NBQ3JFLENBQUM7UUFFRixJQUFJLFlBQVksSUFBSSxJQUFJLENBQUMsTUFBTSxFQUFFO1lBQy9CLElBQUksQ0FBQyxhQUFhLEdBQUcsSUFBSSxxQkFBYSxDQUFDLElBQUksRUFBRTtnQkFDM0MsU0FBUyxFQUFFLElBQUksQ0FBQyxTQUFTO2dCQUN6QixrQkFBa0IsRUFBRSxJQUFJLENBQUMsa0JBQWtCO2dCQUMzQyxjQUFjLEVBQUUsT0FBTyxDQUFDLHNCQUFzQjtnQkFDOUMsV0FBVyxFQUFFLElBQUksQ0FBQyxtQkFBbUI7Z0JBQ3JDLFlBQVksRUFBRSxPQUFPLENBQUMsWUFBWTtnQkFDbEMsYUFBYSxFQUFFLElBQUksQ0FBQyxtQkFBbUIsQ0FBQztvQkFDdEMsT0FBTyxFQUFFLE9BQU8sQ0FBQyxZQUFZLElBQUksSUFBSTtpQkFDdEMsQ0FBQztnQkFDRixjQUFjLEVBQUUsT0FBTyxDQUFDLGNBQWM7Z0JBQ3RDLE1BQU0sRUFBRSxPQUFPLENBQUMsY0FBYztnQkFDOUIsZ0JBQWdCLEVBQUUsT0FBTyxDQUFDLHFCQUFxQjtnQkFDL0MsV0FBVyxFQUFFLG1CQUFtQjthQUNqQyxDQUFDLENBQUM7WUFFSCxJQUFJLENBQUMsYUFBYSxDQUFDLGlCQUFpQixDQUNsQyxHQUFHLElBQUksQ0FBQywyQkFBMkIsQ0FBQyxPQUFPLENBQUMsQ0FDN0MsQ0FBQztTQUNIO1FBRUQsTUFBTSxPQUFPLEdBQ1gsT0FBTyxDQUFDLE9BQU87WUFDZixPQUFPLENBQUMsZUFBZTtZQUN2QixDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDL0IsSUFBSSxPQUFPLEVBQUU7WUFDWCxJQUFJLENBQUMsVUFBVSxDQUFDLGlCQUFPLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztZQUUxQyxJQUFJLENBQUMsT0FBTyxHQUFHLElBQUksaUJBQU8sQ0FBQyxJQUFJLEVBQUU7Z0JBQy9CLFdBQVcsRUFBRSxjQUFjO2dCQUMzQixJQUFJLEVBQUUsSUFBSSxDQUFDLFNBQVM7Z0JBQ3BCLE1BQU0sRUFBRSxPQUFPLENBQUMsb0JBQW9CLElBQUksTUFBTTtnQkFDOUMsa0JBQWtCLEVBQUUsSUFBSSxDQUFDLGtCQUFrQjtnQkFDM0MsR0FBRyxPQUFPO2dCQUVWLHlCQUF5QixFQUFFO29CQUN6QixHQUFHLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxFQUFFLE9BQU8sRUFBRSxLQUFLLEVBQUUsQ0FBQztvQkFDL0MsR0FBRyxDQUFDLE9BQU8sQ0FBQyx5QkFBeUIsSUFBSSxFQUFFLENBQUM7aUJBQzdDO2dCQUNELGNBQWMsRUFBRTtvQkFDZCxHQUFHLENBQUMsT0FBTyxDQUFDLGNBQWMsSUFBSSxFQUFFLENBQUM7b0JBQ2pDLEdBQUcsSUFBSSxDQUFDLDJCQUEyQixDQUFDLE9BQU8sQ0FBQztpQkFDN0M7Z0JBRUQsbUJBQW1CLEVBQUUsSUFBSSxDQUFDLFdBQVc7Z0JBQ3JDLG1CQUFtQjthQUNwQixDQUFDLENBQUM7WUFFSCxJQUFJLENBQUMsU0FBUyxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsU0FBUyxDQUFDO1lBRXhDLE1BQU0sbURBQW1ELEdBR3JEO2dCQUNGLENBQUMsdUNBQW1DLENBQUMsMEJBQTBCLENBQUMsRUFDOUQsa0NBQStCLENBQUMsMEJBQTBCO2dCQUM1RCxDQUFDLHVDQUFtQyxDQUFDLFdBQVcsQ0FBQyxFQUMvQyxrQ0FBK0IsQ0FBQyxXQUFXO2FBQzlDLENBQUM7WUFFRixJQUFJLE9BQU8sQ0FBQyxZQUFZLElBQUksS0FBSyxFQUFFO2dCQUNqQyxNQUFNLG1CQUFtQixHQUN2QixtQ0FBeUIsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLFdBQVcsQ0FBQztvQkFDakQsQ0FBQyxDQUFDO3dCQUNFLGlCQUFpQixFQUNmLE9BQU8sQ0FBQyxtQkFBbUIsRUFBRSxpQkFBaUI7d0JBQ2hELHFCQUFxQixFQUNuQixPQUFPLENBQUMsbUJBQW1CLEVBQUUscUJBQXFCO3dCQUNwRCxZQUFZLEVBQUUsT0FBTyxDQUFDLG1CQUFtQixFQUFFLFlBQVk7d0JBQ3ZELFlBQVksRUFBRSxPQUFPLENBQUMsbUJBQW1CLEVBQUUsWUFBWTs0QkFDckQsQ0FBQyxDQUFDLG1EQUFtRCxDQUNqRCxPQUFPLENBQUMsbUJBQW1CLENBQUMsWUFBWSxDQUN6Qzs0QkFDSCxDQUFDLENBQUMsa0NBQStCLENBQUMsMEJBQTBCO3FCQUMvRDtvQkFDSCxDQUFDLENBQUMsRUFBRSxDQUFDO2dCQUNULElBQUksQ0FBQyxPQUFPLENBQUMsU0FBUyxDQUFDLFlBQVksQ0FBQztvQkFDbEMsUUFBUSxFQUFFLElBQUksQ0FBQyxPQUFPLENBQUMsV0FBVztvQkFDbEMsY0FBYyxFQUFFLElBQUksQ0FBQyxPQUFPLENBQUMsY0FBYztvQkFDM0MsbUJBQW1CO2lCQUNwQixDQUFDLENBQUM7YUFDSjtTQUNGO2FBQU07WUFDTCxxRkFBcUY7WUFDckYsSUFBSSxPQUFPLENBQUMsWUFBWSxFQUFFO2dCQUN4QixNQUFNLElBQUksS0FBSyxDQUNiLHlEQUF5RCxDQUMxRCxDQUFDO2FBQ0g7WUFFRCxJQUFJLE9BQU8sQ0FBQyxrQkFBa0IsRUFBRTtnQkFDOUIsTUFBTSxJQUFJLEtBQUssQ0FDYiwrREFBK0QsQ0FDaEUsQ0FBQzthQUNIO1lBRUQsSUFBSSxPQUFPLENBQUMsZUFBZSxFQUFFO2dCQUMzQixNQUFNLElBQUksS0FBSyxDQUNiLDREQUE0RCxDQUM3RCxDQUFDO2FBQ0g7U0FDRjtRQUVELElBQ0UsQ0FBQyxPQUFPLENBQUMsU0FBUyxJQUFJLElBQUksQ0FBQztZQUMzQixJQUFJLENBQUMsTUFBTSxFQUFFLE9BQU87WUFDcEIsSUFBSSxDQUFDLGFBQWEsRUFBRSxXQUFXLEVBQy9CO1lBQ0EsSUFBSSxDQUFDLFNBQVMsR0FBRyxJQUFJLGtCQUFTLENBQUMsSUFBSSxDQUFDLE1BQU0sRUFBRSxPQUFPLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztZQUN0RSxJQUFJLENBQUMsU0FBUyxDQUFDLGtCQUFrQixDQUFDO2dCQUNoQyxNQUFNLEVBQUUsR0FBRyxFQUFFLENBQ1gsSUFBSSxDQUFDLGFBQWEsRUFBRSxXQUFXLENBQUMsR0FBRyxDQUFDLENBQUMsRUFBRSxFQUFFLEVBQUUsQ0FBQyxrQkFBa0IsRUFBRSxFQUFFLENBQUM7b0JBQ25FLEVBQUU7YUFDTCxDQUFDLENBQUM7U0FDSjtRQUVELE1BQU0sVUFBVSxHQUFHLE9BQU8sQ0FBQyxVQUFVLElBQUksS0FBSyxDQUFDO1FBQy9DLE1BQU0sV0FBVyxHQUFHLE9BQU8sQ0FBQyxXQUFXLElBQUksQ0FBQyxVQUFVLENBQUM7UUFFdkQsSUFBSSxVQUFVLElBQUksV0FBVyxFQUFFO1lBQzdCLE1BQU0sSUFBSSxLQUFLLENBQ2IsK0RBQStELENBQ2hFLENBQUM7U0FDSDtRQUVELE1BQU0sZUFBZSxHQUFHLE9BQU8sQ0FBQyxtQkFBbUIsSUFBSSxLQUFLLENBQUM7UUFFN0QsSUFBSSxlQUFlLElBQUksQ0FBQyxJQUFJLENBQUMsV0FBVyxJQUFJLElBQUksQ0FBQyxNQUFNLEVBQUU7WUFDdkQsTUFBTSxJQUFJLEtBQUssQ0FDYix1RkFBdUYsQ0FDeEYsQ0FBQztTQUNIO1FBRUQsTUFBTSxnQkFBZ0IsR0FBRyxDQUFDLFNBQWtCLEVBQUUsRUFBRSxDQUM5QyxTQUFTLElBQUksSUFBSSxDQUFDLFdBQVcsRUFBRSxLQUFLO1lBQ2xDLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsS0FBSyxDQUFDO1lBQzFCLENBQUMsQ0FBQyxTQUFTLENBQUM7UUFFaEIsSUFBSSxVQUFVLEVBQUU7WUFDZCxNQUFNLGNBQWMsR0FBRztnQkFDckIsTUFBTSxFQUFFLGdCQUFnQixDQUFDLGVBQWUsQ0FBQzthQUMxQyxDQUFDO1lBQ0YsSUFBSSxDQUFDLE1BQU0sRUFBRSxhQUFhLENBQ3hCLGdCQUFTLENBQUMsQ0FBQyxjQUFjLEVBQUUsT0FBTyxDQUFDLGlCQUFpQixJQUFJLEVBQUUsQ0FBQyxDQUFDLENBQzdELENBQUM7U0FDSDtRQUVELElBQUksV0FBVyxFQUFFO1lBQ2YsTUFBTSxjQUFjLEdBQStCO2dCQUNqRCxlQUFlLEVBQUU7b0JBQ2YsU0FBUyxFQUFFLE9BQU8sQ0FBQyxzQkFBc0I7d0JBQ3ZDLENBQUMsQ0FBQzs0QkFDRSxLQUFLLEVBQUUsT0FBTyxDQUFDLHNCQUFzQjt5QkFDdEM7d0JBQ0gsQ0FBQyxDQUFDLFNBQVM7b0JBQ2IsTUFBTSxFQUFFLGdCQUFnQixDQUFDLGVBQWUsQ0FBQztvQkFDekMsV0FBVyxFQUFFLElBQUksQ0FBQyxtQkFBbUI7b0JBQ3JDLFdBQVcsRUFBRSxtQkFBbUI7aUJBQ2pDO2FBQ0YsQ0FBQztZQUNGLElBQUksQ0FBQyxlQUFlLEdBQUcsSUFBSSxnQ0FBbUIsQ0FDNUMsSUFBSSxFQUNKLGdCQUFTLENBQUMsQ0FBQyxjQUFjLEVBQUUsT0FBTyxDQUFDLGtCQUFrQixJQUFJLEVBQUUsQ0FBQyxDQUFDLENBQzlELENBQUM7U0FDSDtRQUVELElBQUksT0FBTyxDQUFDLG1CQUFtQixJQUFJLElBQUksRUFBRTtZQUN2QyxJQUFJLENBQUMsTUFBTSxFQUFFLHNCQUFzQixDQUNqQyxHQUFHLENBQUMsT0FBTyxDQUFDLDJCQUEyQixJQUFJLEVBQUUsQ0FBQyxDQUMvQyxDQUFDO1NBQ0g7UUFFRCxNQUFNLFVBQVUsR0FBRyxPQUFPLENBQUMsVUFBVSxJQUFJLENBQUMsT0FBTyxDQUFDLFlBQVksQ0FBQztRQUMvRCxJQUFJLENBQUMsSUFBSSxDQUFDLE1BQU0sSUFBSSxVQUFVLEVBQUU7WUFDOUIsSUFBSSxtQkFBUSxDQUFDLElBQUksRUFBRSxPQUFPLENBQUMsaUJBQWlCLENBQUMsQ0FBQztTQUMvQztRQUVELHFHQUFxRztRQUNyRyxJQUFJLENBQUMsT0FBTyxHQUFHLElBQUksaUJBQU8sQ0FBQyxJQUFJLEVBQUUsT0FBTyxDQUFDLGNBQWMsQ0FBQyxDQUFDO1FBRXpELElBQUksT0FBTyxDQUFDLE9BQU8sSUFBSSxJQUFJLEVBQUU7WUFDM0IsSUFBSSxDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQUMsWUFBWSxJQUFJLENBQUMsNEJBQTRCLEVBQUUsQ0FBQyxDQUFDO1lBRXZFLG9EQUFvRDtZQUNwRCxrRkFBa0Y7WUFDbEYsSUFBSSxDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQ25CLGtCQUFrQixJQUFJLENBQUMsNEJBQTRCLEdBQUcsQ0FDdkQsQ0FBQztTQUNIO1FBRUQsSUFBSSxPQUFPLENBQUMsUUFBUSxJQUFJLEtBQUssRUFBRTtZQUM3QixJQUFJLENBQUMsUUFBUSxHQUFHLElBQUkscUJBQVEsQ0FBQyxJQUFJLEVBQUUsRUFBRSxHQUFHLE9BQU8sQ0FBQyxlQUFlLEVBQUUsQ0FBQyxDQUFDO1NBQ3BFO0lBQ0gsQ0FBQztJQTNZRDs7T0FFRztJQUNILElBQVcsd0JBQXdCO1FBQ2pDLE9BQU8sSUFBSSxDQUFDLE9BQU8sQ0FBQyx3QkFBd0IsQ0FBQztJQUMvQyxDQUFDO0lBRUQ7O09BRUc7SUFDSCxJQUFXLFVBQVU7UUFDbkIsT0FBTyxJQUFJLENBQUMsT0FBTyxDQUFDLFVBQVUsQ0FBQztJQUNqQyxDQUFDO0lBeUJEOztPQUVHO0lBQ0gsSUFBVyxjQUFjO1FBQ3ZCLE9BQU8sSUFBSSxDQUFDLE9BQU8sQ0FBQyxjQUFjLENBQUM7SUFDckMsQ0FBQztJQUVEOztPQUVHO0lBQ0gsSUFBVyxjQUFjO1FBQ3ZCLE9BQU8sSUFBSSxDQUFDLE9BQU8sQ0FBQyxjQUFjLENBQUM7SUFDckMsQ0FBQztJQUlEOzs7O09BSUc7SUFDSCxJQUFXLGNBQWM7UUFDdkIsT0FBTyxJQUFJLENBQUMsT0FBTyxDQUFDLGNBQWMsQ0FBQztJQUNyQyxDQUFDO0lBWUQ7O09BRUc7SUFDSCxJQUFXLFFBQVE7UUFDakIsT0FBTyxJQUFJLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQztJQUMvQixDQUFDO0lBZ1VPLDJCQUEyQixDQUFDLE9BQTJCO1FBQzdELDZEQUE2RDtRQUM3RCw4QkFBOEI7UUFDOUIsSUFBSSxDQUFDLE9BQU8sQ0FBQyxPQUFPLElBQUksT0FBTyxDQUFDLGtCQUFrQixDQUFDLElBQUksSUFBSSxDQUFDLElBQUksRUFBRSxNQUFNLEVBQUU7WUFDeEUsT0FBTztnQkFDTDtvQkFDRSxJQUFJLEVBQUUsNEJBQTRCO29CQUNsQyxJQUFJLEVBQUUsMkJBQTJCO29CQUNqQyxJQUFJLEVBQUUsT0FBTyxDQUFDLGtCQUFrQjt3QkFDOUIsQ0FBQyxDQUFDOzRCQUNFLEtBQUssRUFBRSxnQkFBZ0IsT0FBTyxDQUFDLGtCQUFrQixLQUFLOzRCQUN0RCxTQUFTLEVBQUUsSUFBSSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsaUJBQWlCO3lCQUM5Qzt3QkFDSCxDQUFDLENBQUM7NEJBQ0UsU0FBUyxFQUFFLElBQUksQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLGlCQUFpQjt5QkFDOUM7aUJBQ047YUFDRixDQUFDO1NBQ0g7YUFBTTtZQUNMLE9BQU8sRUFBRSxDQUFDO1NBQ1g7SUFDSCxDQUFDO0lBRU0sT0FBTyxDQUFDLElBQTRCO1FBQ3pDLElBQUksQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQzVCLENBQUM7SUFFRDs7Ozs7T0FLRztJQUNJLFNBQVMsQ0FBQyxJQUFZLEVBQUUsT0FBZTtRQUM1QyxJQUFJLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxJQUFJLEVBQUUsT0FBTyxDQUFDLENBQUM7SUFDeEMsQ0FBQztJQUVEOzs7T0FHRztJQUNJLFlBQVksQ0FBQyxJQUFZO1FBQzlCLElBQUksQ0FBQyxPQUFPLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQ2xDLENBQUM7SUFFRDs7O09BR0c7SUFDSSxTQUFTLENBQUMsSUFBWTtRQUMzQixPQUFPLElBQUksQ0FBQyxPQUFPLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQ3RDLENBQUM7SUFFRDs7O09BR0c7SUFDSSxpQkFBaUIsQ0FBQyxHQUFHLFFBQWtCO1FBQzVDLEtBQUssTUFBTSxDQUFDLElBQUksUUFBUSxFQUFFO1lBQ3hCLElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDO1NBQzFCO0lBQ0gsQ0FBQztJQUVEOzs7T0FHRztJQUNJLGNBQWMsQ0FBQyxHQUFHLFFBQWtCO1FBQ3pDLEtBQUssTUFBTSxDQUFDLElBQUksUUFBUSxFQUFFO1lBQ3hCLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDO1NBQ3ZCO0lBQ0gsQ0FBQztJQUVEOzs7T0FHRztJQUNJLFNBQVMsQ0FBQyxNQUErQjtRQUM5QyxLQUFLLE1BQU0sQ0FBQyxJQUFJLEVBQUUsS0FBSyxDQUFDLElBQUksTUFBTSxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsRUFBRTtZQUNsRCxJQUFJLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQyxJQUFJLEVBQUUsS0FBSyxDQUFDLENBQUM7U0FDcEM7SUFDSCxDQUFDO0lBRUQ7OztPQUdHO0lBQ0ksV0FBVyxDQUFDLEdBQUcsUUFBa0I7UUFDdEMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxXQUFXLENBQUMsR0FBRyxRQUFRLENBQUMsQ0FBQztJQUN4QyxDQUFDO0lBRUQ7Ozs7O09BS0c7SUFDSyxxQkFBcUIsQ0FDM0IsbUJBQW9EO1FBRXBELE1BQU0seUJBQXlCLEdBQUc7WUFDaEMsaUJBQWlCLEVBQ2YsbUJBQW1CLEVBQUUsaUJBQWlCLElBQUksbUJBQW1CO1lBQy9ELHFCQUFxQixFQUNuQixtQkFBbUIsRUFBRSxxQkFBcUIsSUFBSSx1QkFBdUI7WUFDdkUsWUFBWSxFQUFFLG1CQUFtQixFQUFFLFlBQVk7WUFDL0MsWUFBWSxFQUFFLG1CQUFtQixFQUFFLFlBQVk7U0FDaEQsQ0FBQztRQUVGLElBQ0UseUJBQXlCLENBQUMsWUFBWTtZQUN0Qyx1Q0FBbUMsQ0FBQyxXQUFXLEVBQy9DO1lBQ0EsT0FBTztnQkFDTDtvQkFDRSxJQUFJLEVBQUUsMkJBQTJCO29CQUNqQyxJQUFJLEVBQUUsMENBQTBDO29CQUNoRCxJQUFJLEVBQUU7d0JBQ0osWUFBWSxFQUFFLFdBQVc7d0JBQ3pCLGdCQUFnQixFQUFFLHlCQUF5QixDQUFDLFlBQVk7d0JBQ3hELHVCQUF1QixFQUFFLEdBQUc7cUJBQzdCO2lCQUNGO2dCQUNEO29CQUNFLElBQUksRUFBRSx3QkFBd0I7b0JBQzlCLEdBQUcsRUFBRSxHQUFHLElBQUksQ0FBQyxnQkFBZ0IsV0FBVztpQkFDekM7YUFDRixDQUFDO1NBQ0g7UUFFRCxJQUFJLHlCQUF5QixDQUFDLFlBQVksRUFBRTtZQUMxQyxPQUFPO2dCQUNMO29CQUNFLElBQUksRUFBRSwyQkFBMkI7b0JBQ2pDLElBQUksRUFBRSwwQ0FBMEM7b0JBQ2hELElBQUksRUFBRTt3QkFDSixtQkFBbUIsRUFBRSxxQkFBYyxDQUNqQyx5QkFBeUIsQ0FBQyxpQkFBaUIsQ0FDNUM7d0JBQ0QsdUJBQXVCLEVBQUUscUJBQWMsQ0FDckMseUJBQXlCLENBQUMscUJBQXFCLENBQ2hEO3dCQUNELFlBQVksRUFBRSxXQUFXO3dCQUN6QixnQkFBZ0IsRUFBRSx5QkFBeUIsQ0FBQyxZQUFZO3dCQUN4RCx1QkFBdUIsRUFBRSxHQUFHO3FCQUM3QjtpQkFDRjtnQkFDRDtvQkFDRSxJQUFJLEVBQUUsd0JBQXdCO29CQUM5QixHQUFHLEVBQUUsR0FBRyxJQUFJLENBQUMsZ0JBQWdCLFdBQVc7aUJBQ3pDO2FBQ0YsQ0FBQztTQUNIO1FBRUQsT0FBTztZQUNMO2dCQUNFLElBQUksRUFBRSx3QkFBd0I7Z0JBQzlCLEdBQUcsRUFBRSxHQUFHLElBQUksQ0FBQyxnQkFBZ0IsV0FBVztnQkFDeEMsR0FBRyxFQUFFO29CQUNILGlCQUFpQixFQUFFLHFCQUFjLENBQy9CLHlCQUF5QixDQUFDLGlCQUFpQixDQUM1QztvQkFDRCxxQkFBcUIsRUFBRSxxQkFBYyxDQUNuQyx5QkFBeUIsQ0FBQyxxQkFBcUIsQ0FDaEQ7aUJBQ0Y7YUFDRjtTQUNGLENBQUM7SUFDSixDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUJBQW1CLENBQ3hCLFVBQXNDLEVBQUU7UUFFeEMsTUFBTSxPQUFPLEdBQUcsSUFBSSxLQUFLLEVBQVcsQ0FBQztRQUVyQyx5Q0FBeUM7UUFDekMsT0FBTyxDQUFDLElBQUksQ0FBQyxHQUFHLElBQUksQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO1FBRTdDLElBQUksSUFBSSxDQUFDLFdBQVcsRUFBRTtZQUNwQixPQUFPLENBQUMsSUFBSSxDQUFDO2dCQUNYLElBQUksRUFBRSxlQUFlO2dCQUNyQixJQUFJLEVBQUUsdUJBQXVCO2dCQUM3QixJQUFJLEVBQUUsRUFBRSxjQUFjLEVBQUUsSUFBSSxDQUFDLFdBQVcsRUFBRTthQUMzQyxDQUFDLENBQUM7U0FDSjtRQUVELElBQUksSUFBSSxDQUFDLE9BQU8sQ0FBQyxjQUFjLEtBQUssaUNBQWtCLENBQUMsSUFBSSxFQUFFO1lBQzNELE9BQU8sQ0FBQyxJQUFJLENBQUM7Z0JBQ1gsSUFBSSxFQUFFLFlBQVk7Z0JBQ2xCLElBQUksRUFBRSwwQkFBMEI7Z0JBQ2hDLElBQUksRUFBRSxFQUFFLE9BQU8sRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRTthQUM1QyxDQUFDLENBQUM7U0FDSjtRQUVELE1BQU0sT0FBTyxHQUFHLE9BQU8sQ0FBQyxPQUFPLElBQUksS0FBSyxDQUFDO1FBRXpDLElBQUksSUFBSSxDQUFDLE9BQU8sQ0FBQyxxQkFBcUIsRUFBRTtZQUN0QyxPQUFPLENBQUMsSUFBSSxDQUNWLEdBQUcsSUFBSSxDQUFDLHFCQUFxQixDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsbUJBQW1CLENBQUMsQ0FDaEUsQ0FBQztTQUNIO1FBRUQsT0FBTyxDQUFDLElBQUksQ0FBQztZQUNYLElBQUksRUFBRSxzQkFBc0I7WUFDNUIsR0FBRyxFQUFFLE9BQU87Z0JBQ1YsQ0FBQyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsK0JBQStCO2dCQUM5QyxDQUFDLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxjQUFjO1NBQ2hDLENBQUMsQ0FBQztRQUVILE9BQU8sT0FBTyxDQUFDO0lBQ2pCLENBQUM7SUFFRDs7Ozs7Ozs7T0FRRztJQUNJLE9BQU8sQ0FBQyxHQUFHLElBQWM7UUFDOUIsT0FBTyxJQUFJLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxHQUFHLElBQUksQ0FBQyxDQUFDO0lBQ3ZDLENBQUM7SUFFRDs7Ozs7Ozs7T0FRRztJQUNJLFVBQVUsQ0FBQyxHQUFHLElBQWM7UUFDakMsT0FBTyxJQUFJLENBQUMsT0FBTyxDQUFDLFVBQVUsQ0FBQyxHQUFHLElBQUksQ0FBQyxDQUFDO0lBQzFDLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxXQUFXLENBQUMsR0FBRyxJQUFjO1FBQ2xDLE9BQU8sSUFBSSxDQUFDLE9BQU8sQ0FBQyxXQUFXLENBQUMsR0FBRyxJQUFJLENBQUMsQ0FBQztJQUMzQyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7O09BV0c7SUFDSSxjQUFjLENBQUMsR0FBRyxJQUFjO1FBQ3JDLE9BQU8sSUFBSSxDQUFDLE9BQU8sQ0FBQyxjQUFjLENBQUMsR0FBRyxJQUFJLENBQUMsQ0FBQztJQUM5QyxDQUFDO0lBRU0sZ0JBQWdCLENBQUMsT0FBZTtRQUNyQyxJQUFJLENBQUMsU0FBUyxFQUFFLFdBQVcsQ0FBQyxPQUFPLENBQUMsQ0FBQztJQUN2QyxDQUFDO0lBRU8sVUFBVSxDQUFDLE9BQTJCO1FBQzVDLElBQUksSUFBSSxDQUFDLE9BQU8sQ0FBQyxPQUFPLEVBQUU7WUFDeEIsSUFBSSxpQkFBTyxDQUFDLElBQUksRUFBRTtnQkFDaEIsSUFBSSxFQUFFLElBQUksQ0FBQyxPQUFPLENBQUMsT0FBTztnQkFDMUIsY0FBYyxFQUFFLE9BQU8sQ0FBQyxjQUFjLElBQUksT0FBTyxDQUFDLFVBQVU7Z0JBQzVELGVBQWUsRUFBRSxPQUFPLENBQUMsZUFBZTthQUN6QyxDQUFDLENBQUM7U0FDSjtJQUNILENBQUM7SUFFTyxtQkFBbUI7UUFDekIsSUFBSSxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQ3BCLFFBQVEsRUFDUixNQUFNLEVBQ04sT0FBTyxFQUNQLGdCQUFnQixFQUNoQixpQkFBaUIsRUFDakIsaUJBQWlCLEVBQ2pCLGtCQUFrQixFQUVsQiwyREFBMkQsRUFDM0QseUNBQXlDLEVBRXpDLGdCQUFnQixFQUNoQixNQUFNLEVBQ04sT0FBTyxFQUNQLFFBQVEsRUFDUixZQUFZLEVBRVosbUVBQW1FLEVBQ25FLFNBQVMsRUFFVCxrREFBa0QsRUFDbEQsVUFBVSxFQUNWLFFBQVEsRUFFUixxQkFBcUIsRUFDckIsYUFBYSxFQUViLCtEQUErRCxFQUMvRCxlQUFlLEVBRWYsMEJBQTBCLEVBQzFCLGVBQWUsRUFDZixnQkFBZ0IsRUFFaEIsb0JBQW9CLEVBQ3BCLGVBQWUsRUFFZix5QkFBeUIsRUFDekIsY0FBYyxFQUVkLHdCQUF3QixFQUN4QixPQUFPLEVBRVAsdUJBQXVCLEVBQ3ZCLGlCQUFpQixFQUVqQixnREFBZ0QsRUFDaEQsUUFBUSxDQUNULENBQUM7SUFDSixDQUFDO0lBRUQ7Ozs7O09BS0c7SUFDSSxjQUFjLENBQUMsSUFBVTtRQUM5QixPQUFPLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxhQUFhLElBQUksSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ3RELENBQUM7SUFFRDs7T0FFRztJQUNILElBQVcsa0JBQWtCO1FBQzNCLE9BQU8sSUFBSSxDQUFDLGFBQWEsRUFBRSxXQUFXLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDNUMsQ0FBQzs7QUFod0JILGtDQWl3QkMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBqb2luIH0gZnJvbSBcInBhdGhcIjtcbmltcG9ydCB7IEJ1bmRsZXIsIEJ1bmRsZXJPcHRpb25zIH0gZnJvbSBcIi4vYnVuZGxlclwiO1xuaW1wb3J0IHsgSmVzdCwgSmVzdE9wdGlvbnMgfSBmcm9tIFwiLi9qZXN0XCI7XG5pbXBvcnQge1xuICBDb2RlQXJ0aWZhY3RBdXRoUHJvdmlkZXIgYXMgTm9kZVBhY2thZ2VDb2RlQXJ0aWZhY3RBdXRoUHJvdmlkZXIsXG4gIENvZGVBcnRpZmFjdE9wdGlvbnMsXG4gIE5vZGVQYWNrYWdlLFxuICBOb2RlUGFja2FnZU1hbmFnZXIsXG4gIE5vZGVQYWNrYWdlT3B0aW9ucyxcbn0gZnJvbSBcIi4vbm9kZS1wYWNrYWdlXCI7XG5pbXBvcnQgeyBQcm9qZW5yYywgUHJvamVucmNPcHRpb25zIH0gZnJvbSBcIi4vcHJvamVucmNcIjtcbmltcG9ydCB7IEJ1aWxkV29ya2Zsb3cgfSBmcm9tIFwiLi4vYnVpbGRcIjtcbmltcG9ydCB7IFBST0pFTl9ESVIsIFBST0pFTl9SQyB9IGZyb20gXCIuLi9jb21tb25cIjtcbmltcG9ydCB7XG4gIEF1dG9NZXJnZSxcbiAgRGVwZW5kYWJvdE9wdGlvbnMsXG4gIEdpdEh1YlByb2plY3QsXG4gIEdpdEh1YlByb2plY3RPcHRpb25zLFxuICBHaXRJZGVudGl0eSxcbn0gZnJvbSBcIi4uL2dpdGh1YlwiO1xuaW1wb3J0IHsgREVGQVVMVF9HSVRIVUJfQUNUSU9OU19VU0VSIH0gZnJvbSBcIi4uL2dpdGh1Yi9jb25zdGFudHNcIjtcbmltcG9ydCB7IHNlY3JldFRvU3RyaW5nIH0gZnJvbSBcIi4uL2dpdGh1Yi91dGlsXCI7XG5pbXBvcnQge1xuICBKb2JQZXJtaXNzaW9uLFxuICBKb2JQZXJtaXNzaW9ucyxcbiAgSm9iU3RlcCxcbiAgVHJpZ2dlcnMsXG59IGZyb20gXCIuLi9naXRodWIvd29ya2Zsb3dzLW1vZGVsXCI7XG5pbXBvcnQgeyBJZ25vcmVGaWxlLCBJZ25vcmVGaWxlT3B0aW9ucyB9IGZyb20gXCIuLi9pZ25vcmUtZmlsZVwiO1xuaW1wb3J0IHtcbiAgUHJldHRpZXIsXG4gIFByZXR0aWVyT3B0aW9ucyxcbiAgVXBncmFkZURlcGVuZGVuY2llcyxcbiAgVXBncmFkZURlcGVuZGVuY2llc09wdGlvbnMsXG59IGZyb20gXCIuLi9qYXZhc2NyaXB0XCI7XG5pbXBvcnQgeyBMaWNlbnNlIH0gZnJvbSBcIi4uL2xpY2Vuc2VcIjtcbmltcG9ydCB7XG4gIGlzQXdzQ29kZUFydGlmYWN0UmVnaXN0cnksXG4gIFB1Ymxpc2hlcixcbiAgUmVsZWFzZSxcbiAgUmVsZWFzZVByb2plY3RPcHRpb25zLFxuICBOcG1QdWJsaXNoT3B0aW9ucyxcbiAgQ29kZUFydGlmYWN0QXV0aFByb3ZpZGVyIGFzIFJlbGVhc2VDb2RlQXJ0aWZhY3RBdXRoUHJvdmlkZXIsXG4gIENvZGVBcnRpZmFjdEF1dGhQcm92aWRlcixcbn0gZnJvbSBcIi4uL3JlbGVhc2VcIjtcbmltcG9ydCB7IFRhc2sgfSBmcm9tIFwiLi4vdGFza1wiO1xuaW1wb3J0IHsgZGVlcE1lcmdlIH0gZnJvbSBcIi4uL3V0aWxcIjtcbmltcG9ydCB7IFZlcnNpb24gfSBmcm9tIFwiLi4vdmVyc2lvblwiO1xuXG5jb25zdCBQUk9KRU5fU0NSSVBUID0gXCJwcm9qZW5cIjtcblxuZXhwb3J0IGludGVyZmFjZSBOb2RlUHJvamVjdE9wdGlvbnNcbiAgZXh0ZW5kcyBHaXRIdWJQcm9qZWN0T3B0aW9ucyxcbiAgICBOb2RlUGFja2FnZU9wdGlvbnMsXG4gICAgUmVsZWFzZVByb2plY3RPcHRpb25zIHtcbiAgLyoqXG4gICAqIExpY2Vuc2UgY29weXJpZ2h0IG93bmVyLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGRlZmF1bHRzIHRvIHRoZSB2YWx1ZSBvZiBhdXRob3JOYW1lIG9yIFwiXCIgaWYgYGF1dGhvck5hbWVgIGlzIHVuZGVmaW5lZC5cbiAgICovXG4gIHJlYWRvbmx5IGNvcHlyaWdodE93bmVyPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgY29weXJpZ2h0IHllYXJzIHRvIHB1dCBpbiB0aGUgTElDRU5TRSBmaWxlLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGN1cnJlbnQgeWVhclxuICAgKi9cbiAgcmVhZG9ubHkgY29weXJpZ2h0UGVyaW9kPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBWZXJzaW9uIG9mIHByb2plbiB0byBpbnN0YWxsLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIERlZmF1bHRzIHRvIHRoZSBsYXRlc3QgdmVyc2lvbi5cbiAgICovXG4gIHJlYWRvbmx5IHByb2plblZlcnNpb24/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEluZGljYXRlcyBvZiBcInByb2plblwiIHNob3VsZCBiZSBpbnN0YWxsZWQgYXMgYSBkZXZEZXBlbmRlbmN5LlxuICAgKlxuICAgKiBAZGVmYXVsdCB0cnVlXG4gICAqL1xuICByZWFkb25seSBwcm9qZW5EZXZEZXBlbmRlbmN5PzogYm9vbGVhbjtcblxuICAvKipcbiAgICogRGVmaW5lIGEgR2l0SHViIHdvcmtmbG93IGZvciBidWlsZGluZyBQUnMuXG4gICAqIEBkZWZhdWx0IC0gdHJ1ZSBpZiBub3QgYSBzdWJwcm9qZWN0XG4gICAqL1xuICByZWFkb25seSBidWlsZFdvcmtmbG93PzogYm9vbGVhbjtcblxuICAvKipcbiAgICogQXV0b21hdGljYWxseSB1cGRhdGUgZmlsZXMgbW9kaWZpZWQgZHVyaW5nIGJ1aWxkcyB0byBwdWxsLXJlcXVlc3QgYnJhbmNoZXMuIFRoaXMgbWVhbnNcbiAgICogdGhhdCBhbnkgZmlsZXMgc3ludGhlc2l6ZWQgYnkgcHJvamVuIG9yIGUuZy4gdGVzdCBzbmFwc2hvdHMgd2lsbCBhbHdheXMgYmUgdXAtdG8tZGF0ZVxuICAgKiBiZWZvcmUgYSBQUiBpcyBtZXJnZWQuXG4gICAqXG4gICAqIEltcGxpZXMgdGhhdCBQUiBidWlsZHMgZG8gbm90IGhhdmUgYW50aS10YW1wZXIgY2hlY2tzLlxuICAgKlxuICAgKiBAZGVmYXVsdCB0cnVlXG4gICAqL1xuICByZWFkb25seSBtdXRhYmxlQnVpbGQ/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBEZWZpbmUgYSBHaXRIdWIgd29ya2Zsb3cgc3RlcCBmb3Igc2VuZGluZyBjb2RlIGNvdmVyYWdlIG1ldHJpY3MgdG8gaHR0cHM6Ly9jb2RlY292LmlvL1xuICAgKiBVc2VzIGNvZGVjb3YvY29kZWNvdi1hY3Rpb25AdjNcbiAgICogQSBzZWNyZXQgaXMgcmVxdWlyZWQgZm9yIHByaXZhdGUgcmVwb3MuIENvbmZpZ3VyZWQgd2l0aCBAY29kZUNvdlRva2VuU2VjcmV0XG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSBjb2RlQ292PzogYm9vbGVhbjtcblxuICAvKipcbiAgICogRGVmaW5lIHRoZSBzZWNyZXQgbmFtZSBmb3IgYSBzcGVjaWZpZWQgaHR0cHM6Ly9jb2RlY292LmlvLyB0b2tlblxuICAgKiBBIHNlY3JldCBpcyByZXF1aXJlZCB0byBzZW5kIGNvdmVyYWdlIGZvciBwcml2YXRlIHJlcG9zaXRvcmllc1xuICAgKiBAZGVmYXVsdCAtIGlmIHRoaXMgb3B0aW9uIGlzIG5vdCBzcGVjaWZpZWQsIG9ubHkgcHVibGljIHJlcG9zaXRvcmllcyBhcmUgc3VwcG9ydGVkXG4gICAqL1xuICByZWFkb25seSBjb2RlQ292VG9rZW5TZWNyZXQ/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIERFUFJFQ0FURUQ6IHJlbmFtZWQgdG8gYHJlbGVhc2VgLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIHRydWUgaWYgbm90IGEgc3VicHJvamVjdFxuICAgKiBAZGVwcmVjYXRlZCBzZWUgYHJlbGVhc2VgLlxuICAgKi9cbiAgcmVhZG9ubHkgcmVsZWFzZVdvcmtmbG93PzogYm9vbGVhbjtcblxuICAvKipcbiAgICogQWRkIHJlbGVhc2UgbWFuYWdlbWVudCB0byB0aGlzIHByb2plY3QuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gdHJ1ZSAoZmFsc2UgZm9yIHN1YnByb2plY3RzKVxuICAgKi9cbiAgcmVhZG9ubHkgcmVsZWFzZT86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSBtYWluIHJlbGVhc2UgYnJhbmNoLlxuICAgKlxuICAgKiBAZGVmYXVsdCBcIm1haW5cIlxuICAgKi9cbiAgcmVhZG9ubHkgZGVmYXVsdFJlbGVhc2VCcmFuY2g6IHN0cmluZztcblxuICAvKipcbiAgICogV29ya2Zsb3cgc3RlcHMgdG8gdXNlIGluIG9yZGVyIHRvIGJvb3RzdHJhcCB0aGlzIHJlcG8uXG4gICAqXG4gICAqIEBkZWZhdWx0IFwieWFybiBpbnN0YWxsIC0tZnJvemVuLWxvY2tmaWxlICYmIHlhcm4gcHJvamVuXCJcbiAgICovXG4gIHJlYWRvbmx5IHdvcmtmbG93Qm9vdHN0cmFwU3RlcHM/OiBKb2JTdGVwW107XG5cbiAgLyoqXG4gICAqIFRoZSBnaXQgaWRlbnRpdHkgdG8gdXNlIGluIHdvcmtmbG93cy5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBHaXRIdWIgQWN0aW9uc1xuICAgKi9cbiAgcmVhZG9ubHkgd29ya2Zsb3dHaXRJZGVudGl0eT86IEdpdElkZW50aXR5O1xuXG4gIC8qKlxuICAgKiBBdXRvbWF0aWNhbGx5IHJlbGVhc2UgdG8gbnBtIHdoZW4gbmV3IHZlcnNpb25zIGFyZSBpbnRyb2R1Y2VkLlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgcmVsZWFzZVRvTnBtPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogVGhlIG5vZGUgdmVyc2lvbiB0byB1c2UgaW4gR2l0SHViIHdvcmtmbG93cy5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBzYW1lIGFzIGBtaW5Ob2RlVmVyc2lvbmBcbiAgICovXG4gIHJlYWRvbmx5IHdvcmtmbG93Tm9kZVZlcnNpb24/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFVzZSBkZXBlbmRhYm90IHRvIGhhbmRsZSBkZXBlbmRlbmN5IHVwZ3JhZGVzLlxuICAgKiBDYW5ub3QgYmUgdXNlZCBpbiBjb25qdW5jdGlvbiB3aXRoIGBkZXBzVXBncmFkZWAuXG4gICAqXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSBkZXBlbmRhYm90PzogYm9vbGVhbjtcblxuICAvKipcbiAgICogT3B0aW9ucyBmb3IgZGVwZW5kYWJvdC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBkZWZhdWx0IG9wdGlvbnNcbiAgICovXG4gIHJlYWRvbmx5IGRlcGVuZGFib3RPcHRpb25zPzogRGVwZW5kYWJvdE9wdGlvbnM7XG5cbiAgLyoqXG4gICAqIFVzZSBnaXRodWIgd29ya2Zsb3dzIHRvIGhhbmRsZSBkZXBlbmRlbmN5IHVwZ3JhZGVzLlxuICAgKiBDYW5ub3QgYmUgdXNlZCBpbiBjb25qdW5jdGlvbiB3aXRoIGBkZXBlbmRhYm90YC5cbiAgICpcbiAgICogQGRlZmF1bHQgdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgZGVwc1VwZ3JhZGU/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBPcHRpb25zIGZvciBgVXBncmFkZURlcGVuZGVuY2llc2AuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gZGVmYXVsdCBvcHRpb25zXG4gICAqL1xuICByZWFkb25seSBkZXBzVXBncmFkZU9wdGlvbnM/OiBVcGdyYWRlRGVwZW5kZW5jaWVzT3B0aW9ucztcblxuICAvKipcbiAgICogQXV0b21hdGljYWxseSBhcHByb3ZlIGRlcHMgdXBncmFkZSBQUnMsIGFsbG93aW5nIHRoZW0gdG8gYmVcbiAgICogbWVyZ2VkIGJ5IG1lcmdpZnkgKGlmIGNvbmZpZ3VlZCkuXG4gICAqXG4gICAqIFRocm93IGlmIHNldCB0byB0cnVlIGJ1dCBgYXV0b0FwcHJvdmVPcHRpb25zYCBhcmUgbm90IGRlZmluZWQuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgYXV0b0FwcHJvdmVVcGdyYWRlcz86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIERlZmluZXMgYW4gLm5wbWlnbm9yZSBmaWxlLiBOb3JtYWxseSB0aGlzIGlzIG9ubHkgbmVlZGVkIGZvciBsaWJyYXJpZXMgdGhhdFxuICAgKiBhcmUgcGFja2FnZWQgYXMgdGFyYmFsbHMuXG4gICAqXG4gICAqIEBkZWZhdWx0IHRydWVcbiAgICovXG4gIHJlYWRvbmx5IG5wbWlnbm9yZUVuYWJsZWQ/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBDb25maWd1cmF0aW9uIG9wdGlvbnMgZm9yIC5ucG1pZ25vcmUgZmlsZVxuICAgKi9cbiAgcmVhZG9ubHkgbnBtSWdub3JlT3B0aW9ucz86IElnbm9yZUZpbGVPcHRpb25zO1xuXG4gIC8qKlxuICAgKiBBZGRpdGlvbmFsIGVudHJpZXMgdG8gLm5wbWlnbm9yZS5cbiAgICogQGRlcHJlY2F0ZWQgLSB1c2UgYHByb2plY3QuYWRkUGFja2FnZUlnbm9yZWBcbiAgICovXG4gIHJlYWRvbmx5IG5wbWlnbm9yZT86IHN0cmluZ1tdO1xuXG4gIC8qKlxuICAgKiBJbmNsdWRlIGEgR2l0SHViIHB1bGwgcmVxdWVzdCB0ZW1wbGF0ZS5cbiAgICpcbiAgICogQGRlZmF1bHQgdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgcHVsbFJlcXVlc3RUZW1wbGF0ZT86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFRoZSBjb250ZW50cyBvZiB0aGUgcHVsbCByZXF1ZXN0IHRlbXBsYXRlLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGRlZmF1bHQgY29udGVudFxuICAgKi9cbiAgcmVhZG9ubHkgcHVsbFJlcXVlc3RUZW1wbGF0ZUNvbnRlbnRzPzogc3RyaW5nW107XG5cbiAgLyoqXG4gICAqIFNldHVwIHByZXR0aWVyLlxuICAgKlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgcHJldHRpZXI/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBQcmV0dGllciBvcHRpb25zXG4gICAqIEBkZWZhdWx0IC0gZGVmYXVsdCBvcHRpb25zXG4gICAqL1xuICByZWFkb25seSBwcmV0dGllck9wdGlvbnM/OiBQcmV0dGllck9wdGlvbnM7XG5cbiAgLyoqXG4gICAqIEFkZGl0aW9uYWwgZW50cmllcyB0byAuZ2l0aWdub3JlXG4gICAqL1xuICByZWFkb25seSBnaXRpZ25vcmU/OiBzdHJpbmdbXTtcblxuICAvKipcbiAgICogU2V0dXAgamVzdCB1bml0IHRlc3RzXG4gICAqIEBkZWZhdWx0IHRydWVcbiAgICovXG4gIHJlYWRvbmx5IGplc3Q/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBKZXN0IG9wdGlvbnNcbiAgICogQGRlZmF1bHQgLSBkZWZhdWx0IG9wdGlvbnNcbiAgICovXG4gIHJlYWRvbmx5IGplc3RPcHRpb25zPzogSmVzdE9wdGlvbnM7XG5cbiAgLyoqXG4gICAqIEdlbmVyYXRlIChvbmNlKSAucHJvamVucmMuanMgKGluIEphdmFTY3JpcHQpLiBTZXQgdG8gYGZhbHNlYCBpbiBvcmRlciB0byBkaXNhYmxlXG4gICAqIC5wcm9qZW5yYy5qcyBnZW5lcmF0aW9uLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIHRydWUgaWYgcHJvamVucmNKc29uIGlzIGZhbHNlXG4gICAqL1xuICByZWFkb25seSBwcm9qZW5yY0pzPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogT3B0aW9ucyBmb3IgLnByb2plbnJjLmpzXG4gICAqIEBkZWZhdWx0IC0gZGVmYXVsdCBvcHRpb25zXG4gICAqL1xuICByZWFkb25seSBwcm9qZW5yY0pzT3B0aW9ucz86IFByb2plbnJjT3B0aW9ucztcblxuICAvKipcbiAgICogT3B0aW9ucyBmb3IgYEJ1bmRsZXJgLlxuICAgKi9cbiAgcmVhZG9ubHkgYnVuZGxlck9wdGlvbnM/OiBCdW5kbGVyT3B0aW9ucztcblxuICAvKipcbiAgICogQSBkaXJlY3Rvcnkgd2hpY2ggd2lsbCBjb250YWluIGJ1aWxkIGFydGlmYWN0cy5cbiAgICpcbiAgICogQGRlZmF1bHQgXCJkaXN0XCJcbiAgICovXG4gIHJlYWRvbmx5IGFydGlmYWN0c0RpcmVjdG9yeT86IHN0cmluZztcblxuICAvKipcbiAgICogRGVmaW5lcyBhIGBwYWNrYWdlYCB0YXNrIHRoYXQgd2lsbCBwcm9kdWNlIGFuIG5wbSB0YXJiYWxsIHVuZGVyIHRoZVxuICAgKiBhcnRpZmFjdHMgZGlyZWN0b3J5IChlLmcuIGBkaXN0YCkuXG4gICAqXG4gICAqIEBkZWZhdWx0IHRydWVcbiAgICovXG4gIHJlYWRvbmx5IHBhY2thZ2U/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBCdWlsZCB3b3JrZmxvdyB0cmlnZ2Vyc1xuICAgKiBAZGVmYXVsdCBcInsgcHVsbFJlcXVlc3Q6IHt9LCB3b3JrZmxvd0Rpc3BhdGNoOiB7fSB9XCJcbiAgICovXG4gIHJlYWRvbmx5IGJ1aWxkV29ya2Zsb3dUcmlnZ2Vycz86IFRyaWdnZXJzO1xufVxuXG4vKipcbiAqIEF1dG9tYXRpYyBidW1wIG1vZGVzXG4gKi9cbmV4cG9ydCBlbnVtIEF1dG9SZWxlYXNlIHtcbiAgLyoqXG4gICAqIEF1dG9tYXRpY2FsbHkgYnVtcCAmIHJlbGVhc2UgYSBuZXcgdmVyc2lvbiBmb3IgZXZlcnkgY29tbWl0IHRvIFwibWFpblwiXG4gICAqL1xuICBFVkVSWV9DT01NSVQsXG5cbiAgLyoqXG4gICAqIEF1dG9tYXRpY2FsbHkgYnVtcCAmIHJlbGVhc2UgYSBuZXcgdmVyc2lvbiBvbiBhIGRhaWx5IGJhc2lzLlxuICAgKi9cbiAgREFJTFksXG59XG5cbi8qKlxuICogTm9kZS5qcyBwcm9qZWN0XG4gKlxuICogQHBqaWQgbm9kZVxuICovXG5leHBvcnQgY2xhc3MgTm9kZVByb2plY3QgZXh0ZW5kcyBHaXRIdWJQcm9qZWN0IHtcbiAgLyoqXG4gICAqIEFQSSBmb3IgbWFuYWdpbmcgdGhlIG5vZGUgcGFja2FnZS5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBwYWNrYWdlOiBOb2RlUGFja2FnZTtcblxuICAvKipcbiAgICogVGhlIC5ucG1pZ25vcmUgZmlsZS5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBucG1pZ25vcmU/OiBJZ25vcmVGaWxlO1xuXG4gIC8qKlxuICAgKiBAZGVwcmVjYXRlZCB1c2UgYHBhY2thZ2UuYWxsb3dMaWJyYXJ5RGVwZW5kZW5jaWVzYFxuICAgKi9cbiAgcHVibGljIGdldCBhbGxvd0xpYnJhcnlEZXBlbmRlbmNpZXMoKTogYm9vbGVhbiB7XG4gICAgcmV0dXJuIHRoaXMucGFja2FnZS5hbGxvd0xpYnJhcnlEZXBlbmRlbmNpZXM7XG4gIH1cblxuICAvKipcbiAgICogQGRlcHJlY2F0ZWQgdXNlIGBwYWNrYWdlLmVudHJ5cG9pbnRgXG4gICAqL1xuICBwdWJsaWMgZ2V0IGVudHJ5cG9pbnQoKTogc3RyaW5nIHtcbiAgICByZXR1cm4gdGhpcy5wYWNrYWdlLmVudHJ5cG9pbnQ7XG4gIH1cblxuICAvKipcbiAgICogQ29tcG9uZW50IHRoYXQgc2V0cyB1cCBtZXJnaWZ5IGZvciBtZXJnaW5nIGFwcHJvdmVkIHB1bGwgcmVxdWVzdHMuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgYXV0b01lcmdlPzogQXV0b01lcmdlO1xuXG4gIC8qKlxuICAgKiBUaGUgUFIgYnVpbGQgR2l0SHViIHdvcmtmbG93LiBgdW5kZWZpbmVkYCBpZiBgYnVpbGRXb3JrZmxvd2AgaXMgZGlzYWJsZWQuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgYnVpbGRXb3JrZmxvdz86IEJ1aWxkV29ya2Zsb3c7XG5cbiAgLyoqXG4gICAqIFBhY2thZ2UgcHVibGlzaGVyLiBUaGlzIHdpbGwgYmUgYHVuZGVmaW5lZGAgaWYgdGhlIHByb2plY3QgZG9lcyBub3QgaGF2ZSBhXG4gICAqIHJlbGVhc2Ugd29ya2Zsb3cuXG4gICAqXG4gICAqIEBkZXByZWNhdGVkIHVzZSBgcmVsZWFzZS5wdWJsaXNoZXJgLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHB1Ymxpc2hlcj86IFB1Ymxpc2hlcjtcblxuICAvKipcbiAgICogUmVsZWFzZSBtYW5hZ2VtZW50LlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHJlbGVhc2U/OiBSZWxlYXNlO1xuXG4gIC8qKlxuICAgKiBNaW5pbXVtIG5vZGUuanMgdmVyc2lvbiByZXF1aXJlZCBieSB0aGlzIHBhY2thZ2UuXG4gICAqL1xuICBwdWJsaWMgZ2V0IG1pbk5vZGVWZXJzaW9uKCk6IHN0cmluZyB8IHVuZGVmaW5lZCB7XG4gICAgcmV0dXJuIHRoaXMucGFja2FnZS5taW5Ob2RlVmVyc2lvbjtcbiAgfVxuXG4gIC8qKlxuICAgKiBNYXhpbXVtIG5vZGUgdmVyc2lvbiByZXF1aXJlZCBieSB0aGlzIHBhY2thZ2UuXG4gICAqL1xuICBwdWJsaWMgZ2V0IG1heE5vZGVWZXJzaW9uKCk6IHN0cmluZyB8IHVuZGVmaW5lZCB7XG4gICAgcmV0dXJuIHRoaXMucGFja2FnZS5tYXhOb2RlVmVyc2lvbjtcbiAgfVxuXG4gIHByb3RlY3RlZCByZWFkb25seSBub2RlVmVyc2lvbj86IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIHBhY2thZ2UgbWFuYWdlciB0byB1c2UuXG4gICAqXG4gICAqIEBkZXByZWNhdGVkIHVzZSBgcGFja2FnZS5wYWNrYWdlTWFuYWdlcmBcbiAgICovXG4gIHB1YmxpYyBnZXQgcGFja2FnZU1hbmFnZXIoKTogTm9kZVBhY2thZ2VNYW5hZ2VyIHtcbiAgICByZXR1cm4gdGhpcy5wYWNrYWdlLnBhY2thZ2VNYW5hZ2VyO1xuICB9XG5cbiAgLyoqXG4gICAqIFRoZSBjb21tYW5kIHRvIHVzZSB0byBydW4gc2NyaXB0cyAoZS5nLiBgeWFybiBydW5gIG9yIGBucG0gcnVuYCBkZXBlbmRzIG9uIHRoZSBwYWNrYWdlIG1hbmFnZXIpLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHJ1blNjcmlwdENvbW1hbmQ6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIEplc3QgY29uZmlndXJhdGlvbiAoaWYgZW5hYmxlZClcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBqZXN0PzogSmVzdDtcblxuICAvKipcbiAgICogQGRlcHJlY2F0ZWQgdXNlIGBwYWNrYWdlLmFkZEZpZWxkKHgsIHkpYFxuICAgKi9cbiAgcHVibGljIGdldCBtYW5pZmVzdCgpIHtcbiAgICByZXR1cm4gdGhpcy5wYWNrYWdlLm1hbmlmZXN0O1xuICB9XG5cbiAgcHVibGljIHJlYWRvbmx5IGJ1bmRsZXI6IEJ1bmRsZXI7XG5cbiAgLyoqXG4gICAqIFRoZSBidWlsZCBvdXRwdXQgZGlyZWN0b3J5LiBBbiBucG0gdGFyYmFsbCB3aWxsIGJlIGNyZWF0ZWQgdW5kZXIgdGhlIGBqc2BcbiAgICogc3ViZGlyZWN0b3J5LiBGb3IgZXhhbXBsZSwgaWYgdGhpcyBpcyBzZXQgdG8gYGRpc3RgICh0aGUgZGVmYXVsdCksIHRoZSBucG1cbiAgICogdGFyYmFsbCB3aWxsIGJlIHBsYWNlZCB1bmRlciBgZGlzdC9qcy9ib29tLWJvb20tMS4yLjMudGdgLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGFydGlmYWN0c0RpcmVjdG9yeTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgbG9jYXRpb24gb2YgdGhlIG5wbSB0YXJiYWxsIGFmdGVyIGJ1aWxkIChgJHthcnRpZmFjdHNEaXJlY3Rvcnl9L2pzYCkuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgYXJ0aWZhY3RzSmF2YXNjcmlwdERpcmVjdG9yeTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgdXBncmFkZSB3b3JrZmxvdy5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSB1cGdyYWRlV29ya2Zsb3c/OiBVcGdyYWRlRGVwZW5kZW5jaWVzO1xuXG4gIHByb3RlY3RlZCByZWFkb25seSB3b3JrZmxvd0Jvb3RzdHJhcFN0ZXBzOiBKb2JTdGVwW107XG4gIHByaXZhdGUgcmVhZG9ubHkgd29ya2Zsb3dHaXRJZGVudGl0eTogR2l0SWRlbnRpdHk7XG4gIHB1YmxpYyByZWFkb25seSBwcmV0dGllcj86IFByZXR0aWVyO1xuXG4gIGNvbnN0cnVjdG9yKG9wdGlvbnM6IE5vZGVQcm9qZWN0T3B0aW9ucykge1xuICAgIHN1cGVyKG9wdGlvbnMpO1xuXG4gICAgdGhpcy5wYWNrYWdlID0gbmV3IE5vZGVQYWNrYWdlKHRoaXMsIG9wdGlvbnMpO1xuICAgIHRoaXMud29ya2Zsb3dCb290c3RyYXBTdGVwcyA9IG9wdGlvbnMud29ya2Zsb3dCb290c3RyYXBTdGVwcyA/PyBbXTtcbiAgICB0aGlzLndvcmtmbG93R2l0SWRlbnRpdHkgPVxuICAgICAgb3B0aW9ucy53b3JrZmxvd0dpdElkZW50aXR5ID8/IERFRkFVTFRfR0lUSFVCX0FDVElPTlNfVVNFUjtcbiAgICB0aGlzLmFydGlmYWN0c0RpcmVjdG9yeSA9IG9wdGlvbnMuYXJ0aWZhY3RzRGlyZWN0b3J5ID8/IFwiZGlzdFwiO1xuICAgIHRoaXMuYXJ0aWZhY3RzSmF2YXNjcmlwdERpcmVjdG9yeSA9IGpvaW4odGhpcy5hcnRpZmFjdHNEaXJlY3RvcnksIFwianNcIik7XG5cbiAgICB0aGlzLnJ1blNjcmlwdENvbW1hbmQgPSAoKCkgPT4ge1xuICAgICAgc3dpdGNoICh0aGlzLnBhY2thZ2VNYW5hZ2VyKSB7XG4gICAgICAgIGNhc2UgTm9kZVBhY2thZ2VNYW5hZ2VyLk5QTTpcbiAgICAgICAgICByZXR1cm4gXCJucG0gcnVuXCI7XG4gICAgICAgIGNhc2UgTm9kZVBhY2thZ2VNYW5hZ2VyLllBUk46XG4gICAgICAgIGNhc2UgTm9kZVBhY2thZ2VNYW5hZ2VyLllBUk4yOlxuICAgICAgICAgIHJldHVybiBcInlhcm4gcnVuXCI7XG4gICAgICAgIGNhc2UgTm9kZVBhY2thZ2VNYW5hZ2VyLlBOUE06XG4gICAgICAgICAgcmV0dXJuIFwicG5wbSBydW5cIjtcbiAgICAgICAgZGVmYXVsdDpcbiAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYHVuZXhwZWN0ZWQgcGFja2FnZSBtYW5hZ2VyICR7dGhpcy5wYWNrYWdlTWFuYWdlcn1gKTtcbiAgICAgIH1cbiAgICB9KSgpO1xuXG4gICAgdGhpcy5ub2RlVmVyc2lvbiA9XG4gICAgICBvcHRpb25zLndvcmtmbG93Tm9kZVZlcnNpb24gPz8gdGhpcy5wYWNrYWdlLm1pbk5vZGVWZXJzaW9uO1xuXG4gICAgLy8gYWRkIFBBVEggZm9yIGFsbCB0YXNrcyB3aGljaCBpbmNsdWRlcyB0aGUgcHJvamVjdCdzIG5wbSAuYmluIGxpc3RcbiAgICB0aGlzLnRhc2tzLmFkZEVudmlyb25tZW50KFxuICAgICAgXCJQQVRIXCIsXG4gICAgICAnJChucHggLWMgXCJub2RlIC1lIFxcXFxcImNvbnNvbGUubG9nKHByb2Nlc3MuZW52LlBBVEgpXFxcXFwiXCIpJ1xuICAgICk7XG5cbiAgICB0aGlzLmFkZExpY2Vuc2Uob3B0aW9ucyk7XG5cbiAgICBpZiAob3B0aW9ucy5ucG1pZ25vcmVFbmFibGVkID8/IHRydWUpIHtcbiAgICAgIHRoaXMubnBtaWdub3JlID0gbmV3IElnbm9yZUZpbGUoXG4gICAgICAgIHRoaXMsXG4gICAgICAgIFwiLm5wbWlnbm9yZVwiLFxuICAgICAgICBvcHRpb25zLm5wbUlnbm9yZU9wdGlvbnNcbiAgICAgICk7XG4gICAgfVxuXG4gICAgdGhpcy5hZGREZWZhdWx0R2l0SWdub3JlKCk7XG5cbiAgICBpZiAob3B0aW9ucy5naXRpZ25vcmU/Lmxlbmd0aCkge1xuICAgICAgZm9yIChjb25zdCBpIG9mIG9wdGlvbnMuZ2l0aWdub3JlKSB7XG4gICAgICAgIHRoaXMuZ2l0aWdub3JlLmV4Y2x1ZGUoaSk7XG4gICAgICB9XG4gICAgfVxuXG4gICAgaWYgKG9wdGlvbnMubnBtaWdub3JlPy5sZW5ndGgpIHtcbiAgICAgIGlmICghdGhpcy5ucG1pZ25vcmUpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICAgICcubnBtaWdub3JlIGlzIG5vdCBkZWZpbmVkIGZvciBhbiBBUFAgcHJvamVjdCB0eXBlLiBBZGQgXCJucG1JZ25vcmU6IHRydWVcIiB0byBvdmVycmlkZSB0aGlzJ1xuICAgICAgICApO1xuICAgICAgfVxuXG4gICAgICBmb3IgKGNvbnN0IGkgb2Ygb3B0aW9ucy5ucG1pZ25vcmUpIHtcbiAgICAgICAgdGhpcy5ucG1pZ25vcmUuZXhjbHVkZShpKTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICBpZiAoIXRoaXMuZWplY3RlZCkge1xuICAgICAgdGhpcy5zZXRTY3JpcHQoUFJPSkVOX1NDUklQVCwgdGhpcy5wYWNrYWdlLnByb2plbkNvbW1hbmQpO1xuICAgIH1cblxuICAgIHRoaXMubnBtaWdub3JlPy5leGNsdWRlKGAvJHtQUk9KRU5fUkN9YCk7XG4gICAgdGhpcy5ucG1pZ25vcmU/LmV4Y2x1ZGUoYC8ke1BST0pFTl9ESVJ9L2ApO1xuICAgIHRoaXMuZ2l0aWdub3JlLmluY2x1ZGUoYC8ke1BST0pFTl9SQ31gKTtcblxuICAgIGNvbnN0IHByb2plbiA9IG9wdGlvbnMucHJvamVuRGV2RGVwZW5kZW5jeSA/PyB0cnVlO1xuICAgIGlmIChwcm9qZW4gJiYgIXRoaXMuZWplY3RlZCkge1xuICAgICAgY29uc3QgcG9zdGZpeCA9IG9wdGlvbnMucHJvamVuVmVyc2lvbiA/IGBAJHtvcHRpb25zLnByb2plblZlcnNpb259YCA6IFwiXCI7XG4gICAgICB0aGlzLmFkZERldkRlcHMoYHByb2plbiR7cG9zdGZpeH1gKTtcbiAgICB9XG5cbiAgICBpZiAoIW9wdGlvbnMuZGVmYXVsdFJlbGVhc2VCcmFuY2gpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihcbiAgICAgICAgJ1wiZGVmYXVsdFJlbGVhc2VCcmFuY2hcIiBpcyB0ZW1wb3JhcmlseSBhIHJlcXVpcmVkIG9wdGlvbiB3aGlsZSB3ZSBtaWdyYXRlIGl0cyBkZWZhdWx0IHZhbHVlIGZyb20gXCJtYXN0ZXJcIiB0byBcIm1haW5cIidcbiAgICAgICk7XG4gICAgfVxuXG4gICAgY29uc3QgYnVpbGRFbmFibGVkID0gb3B0aW9ucy5idWlsZFdvcmtmbG93ID8/ICh0aGlzLnBhcmVudCA/IGZhbHNlIDogdHJ1ZSk7XG5cbiAgICAvLyBjb25maWd1cmUgamVzdCBpZiBlbmFibGVkXG4gICAgLy8gbXVzdCBiZSBiZWZvcmUgdGhlIGJ1aWxkL3JlbGVhc2Ugd29ya2Zsb3dzXG4gICAgaWYgKG9wdGlvbnMuamVzdCA/PyB0cnVlKSB7XG4gICAgICB0aGlzLmplc3QgPSBuZXcgSmVzdCh0aGlzLCBvcHRpb25zLmplc3RPcHRpb25zKTtcbiAgICB9XG5cbiAgICBjb25zdCByZXF1aXJlc0lkVG9rZW5QZXJtaXNzaW9uID1cbiAgICAgIChvcHRpb25zLnNjb3BlZFBhY2thZ2VzT3B0aW9ucyA/PyBbXSkubGVuZ3RoID4gMCAmJlxuICAgICAgb3B0aW9ucy5jb2RlQXJ0aWZhY3RPcHRpb25zPy5hdXRoUHJvdmlkZXIgPT09XG4gICAgICAgIENvZGVBcnRpZmFjdEF1dGhQcm92aWRlci5HSVRIVUJfT0lEQztcblxuICAgIGNvbnN0IHdvcmtmbG93UGVybWlzc2lvbnM6IEpvYlBlcm1pc3Npb25zID0ge1xuICAgICAgaWRUb2tlbjogcmVxdWlyZXNJZFRva2VuUGVybWlzc2lvbiA/IEpvYlBlcm1pc3Npb24uV1JJVEUgOiB1bmRlZmluZWQsXG4gICAgfTtcblxuICAgIGlmIChidWlsZEVuYWJsZWQgJiYgdGhpcy5naXRodWIpIHtcbiAgICAgIHRoaXMuYnVpbGRXb3JrZmxvdyA9IG5ldyBCdWlsZFdvcmtmbG93KHRoaXMsIHtcbiAgICAgICAgYnVpbGRUYXNrOiB0aGlzLmJ1aWxkVGFzayxcbiAgICAgICAgYXJ0aWZhY3RzRGlyZWN0b3J5OiB0aGlzLmFydGlmYWN0c0RpcmVjdG9yeSxcbiAgICAgICAgY29udGFpbmVySW1hZ2U6IG9wdGlvbnMud29ya2Zsb3dDb250YWluZXJJbWFnZSxcbiAgICAgICAgZ2l0SWRlbnRpdHk6IHRoaXMud29ya2Zsb3dHaXRJZGVudGl0eSxcbiAgICAgICAgbXV0YWJsZUJ1aWxkOiBvcHRpb25zLm11dGFibGVCdWlsZCxcbiAgICAgICAgcHJlQnVpbGRTdGVwczogdGhpcy5yZW5kZXJXb3JrZmxvd1NldHVwKHtcbiAgICAgICAgICBtdXRhYmxlOiBvcHRpb25zLm11dGFibGVCdWlsZCA/PyB0cnVlLFxuICAgICAgICB9KSxcbiAgICAgICAgcG9zdEJ1aWxkU3RlcHM6IG9wdGlvbnMucG9zdEJ1aWxkU3RlcHMsXG4gICAgICAgIHJ1bnNPbjogb3B0aW9ucy53b3JrZmxvd1J1bnNPbixcbiAgICAgICAgd29ya2Zsb3dUcmlnZ2Vyczogb3B0aW9ucy5idWlsZFdvcmtmbG93VHJpZ2dlcnMsXG4gICAgICAgIHBlcm1pc3Npb25zOiB3b3JrZmxvd1Blcm1pc3Npb25zLFxuICAgICAgfSk7XG5cbiAgICAgIHRoaXMuYnVpbGRXb3JrZmxvdy5hZGRQb3N0QnVpbGRTdGVwcyhcbiAgICAgICAgLi4udGhpcy5yZW5kZXJVcGxvYWRDb3ZlcmFnZUpvYlN0ZXAob3B0aW9ucylcbiAgICAgICk7XG4gICAgfVxuXG4gICAgY29uc3QgcmVsZWFzZSA9XG4gICAgICBvcHRpb25zLnJlbGVhc2UgPz9cbiAgICAgIG9wdGlvbnMucmVsZWFzZVdvcmtmbG93ID8/XG4gICAgICAodGhpcy5wYXJlbnQgPyBmYWxzZSA6IHRydWUpO1xuICAgIGlmIChyZWxlYXNlKSB7XG4gICAgICB0aGlzLmFkZERldkRlcHMoVmVyc2lvbi5TVEFOREFSRF9WRVJTSU9OKTtcblxuICAgICAgdGhpcy5yZWxlYXNlID0gbmV3IFJlbGVhc2UodGhpcywge1xuICAgICAgICB2ZXJzaW9uRmlsZTogXCJwYWNrYWdlLmpzb25cIiwgLy8gdGhpcyBpcyB3aGVyZSBcInZlcnNpb25cIiBpcyBzZXQgYWZ0ZXIgYnVtcFxuICAgICAgICB0YXNrOiB0aGlzLmJ1aWxkVGFzayxcbiAgICAgICAgYnJhbmNoOiBvcHRpb25zLmRlZmF1bHRSZWxlYXNlQnJhbmNoID8/IFwibWFpblwiLFxuICAgICAgICBhcnRpZmFjdHNEaXJlY3Rvcnk6IHRoaXMuYXJ0aWZhY3RzRGlyZWN0b3J5LFxuICAgICAgICAuLi5vcHRpb25zLFxuXG4gICAgICAgIHJlbGVhc2VXb3JrZmxvd1NldHVwU3RlcHM6IFtcbiAgICAgICAgICAuLi50aGlzLnJlbmRlcldvcmtmbG93U2V0dXAoeyBtdXRhYmxlOiBmYWxzZSB9KSxcbiAgICAgICAgICAuLi4ob3B0aW9ucy5yZWxlYXNlV29ya2Zsb3dTZXR1cFN0ZXBzID8/IFtdKSxcbiAgICAgICAgXSxcbiAgICAgICAgcG9zdEJ1aWxkU3RlcHM6IFtcbiAgICAgICAgICAuLi4ob3B0aW9ucy5wb3N0QnVpbGRTdGVwcyA/PyBbXSksXG4gICAgICAgICAgLi4udGhpcy5yZW5kZXJVcGxvYWRDb3ZlcmFnZUpvYlN0ZXAob3B0aW9ucyksXG4gICAgICAgIF0sXG5cbiAgICAgICAgd29ya2Zsb3dOb2RlVmVyc2lvbjogdGhpcy5ub2RlVmVyc2lvbixcbiAgICAgICAgd29ya2Zsb3dQZXJtaXNzaW9ucyxcbiAgICAgIH0pO1xuXG4gICAgICB0aGlzLnB1Ymxpc2hlciA9IHRoaXMucmVsZWFzZS5wdWJsaXNoZXI7XG5cbiAgICAgIGNvbnN0IG5vZGVQYWNrYWdlVG9SZWxlYXNlQ29kZUFydGlmYWN0QXV0aFByb3ZpZGVyTWFwcGluZzogUmVjb3JkPFxuICAgICAgICBOb2RlUGFja2FnZUNvZGVBcnRpZmFjdEF1dGhQcm92aWRlcixcbiAgICAgICAgUmVsZWFzZUNvZGVBcnRpZmFjdEF1dGhQcm92aWRlclxuICAgICAgPiA9IHtcbiAgICAgICAgW05vZGVQYWNrYWdlQ29kZUFydGlmYWN0QXV0aFByb3ZpZGVyLkFDQ0VTU19BTkRfU0VDUkVUX0tFWV9QQUlSXTpcbiAgICAgICAgICBSZWxlYXNlQ29kZUFydGlmYWN0QXV0aFByb3ZpZGVyLkFDQ0VTU19BTkRfU0VDUkVUX0tFWV9QQUlSLFxuICAgICAgICBbTm9kZVBhY2thZ2VDb2RlQXJ0aWZhY3RBdXRoUHJvdmlkZXIuR0lUSFVCX09JRENdOlxuICAgICAgICAgIFJlbGVhc2VDb2RlQXJ0aWZhY3RBdXRoUHJvdmlkZXIuR0lUSFVCX09JREMsXG4gICAgICB9O1xuXG4gICAgICBpZiAob3B0aW9ucy5yZWxlYXNlVG9OcG0gPz8gZmFsc2UpIHtcbiAgICAgICAgY29uc3QgY29kZUFydGlmYWN0T3B0aW9uczogTnBtUHVibGlzaE9wdGlvbnNbXCJjb2RlQXJ0aWZhY3RPcHRpb25zXCJdID1cbiAgICAgICAgICBpc0F3c0NvZGVBcnRpZmFjdFJlZ2lzdHJ5KHRoaXMucGFja2FnZS5ucG1SZWdpc3RyeSlcbiAgICAgICAgICAgID8ge1xuICAgICAgICAgICAgICAgIGFjY2Vzc0tleUlkU2VjcmV0OlxuICAgICAgICAgICAgICAgICAgb3B0aW9ucy5jb2RlQXJ0aWZhY3RPcHRpb25zPy5hY2Nlc3NLZXlJZFNlY3JldCxcbiAgICAgICAgICAgICAgICBzZWNyZXRBY2Nlc3NLZXlTZWNyZXQ6XG4gICAgICAgICAgICAgICAgICBvcHRpb25zLmNvZGVBcnRpZmFjdE9wdGlvbnM/LnNlY3JldEFjY2Vzc0tleVNlY3JldCxcbiAgICAgICAgICAgICAgICByb2xlVG9Bc3N1bWU6IG9wdGlvbnMuY29kZUFydGlmYWN0T3B0aW9ucz8ucm9sZVRvQXNzdW1lLFxuICAgICAgICAgICAgICAgIGF1dGhQcm92aWRlcjogb3B0aW9ucy5jb2RlQXJ0aWZhY3RPcHRpb25zPy5hdXRoUHJvdmlkZXJcbiAgICAgICAgICAgICAgICAgID8gbm9kZVBhY2thZ2VUb1JlbGVhc2VDb2RlQXJ0aWZhY3RBdXRoUHJvdmlkZXJNYXBwaW5nW1xuICAgICAgICAgICAgICAgICAgICAgIG9wdGlvbnMuY29kZUFydGlmYWN0T3B0aW9ucy5hdXRoUHJvdmlkZXJcbiAgICAgICAgICAgICAgICAgICAgXVxuICAgICAgICAgICAgICAgICAgOiBSZWxlYXNlQ29kZUFydGlmYWN0QXV0aFByb3ZpZGVyLkFDQ0VTU19BTkRfU0VDUkVUX0tFWV9QQUlSLFxuICAgICAgICAgICAgICB9XG4gICAgICAgICAgICA6IHt9O1xuICAgICAgICB0aGlzLnJlbGVhc2UucHVibGlzaGVyLnB1Ymxpc2hUb05wbSh7XG4gICAgICAgICAgcmVnaXN0cnk6IHRoaXMucGFja2FnZS5ucG1SZWdpc3RyeSxcbiAgICAgICAgICBucG1Ub2tlblNlY3JldDogdGhpcy5wYWNrYWdlLm5wbVRva2VuU2VjcmV0LFxuICAgICAgICAgIGNvZGVBcnRpZmFjdE9wdGlvbnMsXG4gICAgICAgIH0pO1xuICAgICAgfVxuICAgIH0gZWxzZSB7XG4gICAgICAvLyB2YWxpZGF0ZSB0aGF0IG5vIHJlbGVhc2Ugb3B0aW9ucyBhcmUgc2VsZWN0ZWQgaWYgdGhlIHJlbGVhc2Ugd29ya2Zsb3cgaXMgZGlzYWJsZWQuXG4gICAgICBpZiAob3B0aW9ucy5yZWxlYXNlVG9OcG0pIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICAgICdcInJlbGVhc2VUb05wbVwiIGlzIG5vdCBzdXBwb3J0ZWQgaWYgXCJyZWxlYXNlXCIgaXMgbm90IHNldCdcbiAgICAgICAgKTtcbiAgICAgIH1cblxuICAgICAgaWYgKG9wdGlvbnMucmVsZWFzZUV2ZXJ5Q29tbWl0KSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihcbiAgICAgICAgICAnXCJyZWxlYXNlRXZlcnlDb21taXRcIiBpcyBub3Qgc3VwcG9ydGVkIGlmIFwicmVsZWFzZVwiIGlzIG5vdCBzZXQnXG4gICAgICAgICk7XG4gICAgICB9XG5cbiAgICAgIGlmIChvcHRpb25zLnJlbGVhc2VTY2hlZHVsZSkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICAgICAgJ1wicmVsZWFzZVNjaGVkdWxlXCIgaXMgbm90IHN1cHBvcnRlZCBpZiBcInJlbGVhc2VcIiBpcyBub3Qgc2V0J1xuICAgICAgICApO1xuICAgICAgfVxuICAgIH1cblxuICAgIGlmIChcbiAgICAgIChvcHRpb25zLmF1dG9NZXJnZSA/PyB0cnVlKSAmJlxuICAgICAgdGhpcy5naXRodWI/Lm1lcmdpZnkgJiZcbiAgICAgIHRoaXMuYnVpbGRXb3JrZmxvdz8uYnVpbGRKb2JJZHNcbiAgICApIHtcbiAgICAgIHRoaXMuYXV0b01lcmdlID0gbmV3IEF1dG9NZXJnZSh0aGlzLmdpdGh1Yiwgb3B0aW9ucy5hdXRvTWVyZ2VPcHRpb25zKTtcbiAgICAgIHRoaXMuYXV0b01lcmdlLmFkZENvbmRpdGlvbnNMYXRlcih7XG4gICAgICAgIHJlbmRlcjogKCkgPT5cbiAgICAgICAgICB0aGlzLmJ1aWxkV29ya2Zsb3c/LmJ1aWxkSm9iSWRzLm1hcCgoaWQpID0+IGBzdGF0dXMtc3VjY2Vzcz0ke2lkfWApID8/XG4gICAgICAgICAgW10sXG4gICAgICB9KTtcbiAgICB9XG5cbiAgICBjb25zdCBkZXBlbmRhYm90ID0gb3B0aW9ucy5kZXBlbmRhYm90ID8/IGZhbHNlO1xuICAgIGNvbnN0IGRlcHNVcGdyYWRlID0gb3B0aW9ucy5kZXBzVXBncmFkZSA/PyAhZGVwZW5kYWJvdDtcblxuICAgIGlmIChkZXBlbmRhYm90ICYmIGRlcHNVcGdyYWRlKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICAgIFwiJ2RlcGVuZGFib3QnIGNhbm5vdCBiZSBjb25maWd1cmVkIHRvZ2V0aGVyIHdpdGggJ2RlcHNVcGdyYWRlJ1wiXG4gICAgICApO1xuICAgIH1cblxuICAgIGNvbnN0IGRlcHNBdXRvQXBwcm92ZSA9IG9wdGlvbnMuYXV0b0FwcHJvdmVVcGdyYWRlcyA/PyBmYWxzZTtcblxuICAgIGlmIChkZXBzQXV0b0FwcHJvdmUgJiYgIXRoaXMuYXV0b0FwcHJvdmUgJiYgdGhpcy5naXRodWIpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihcbiAgICAgICAgXCJBdXRvbWF0aWMgYXBwcm92YWwgb2YgZGVwZW5kZW5jaWVzIHVwZ3JhZGVzIHJlcXVpcmVzIGNvbmZpZ3VyaW5nIGBhdXRvQXBwcm92ZU9wdGlvbnNgXCJcbiAgICAgICk7XG4gICAgfVxuXG4gICAgY29uc3QgYXV0b0FwcHJvdmVMYWJlbCA9IChjb25kaXRpb246IGJvb2xlYW4pID0+XG4gICAgICBjb25kaXRpb24gJiYgdGhpcy5hdXRvQXBwcm92ZT8ubGFiZWxcbiAgICAgICAgPyBbdGhpcy5hdXRvQXBwcm92ZS5sYWJlbF1cbiAgICAgICAgOiB1bmRlZmluZWQ7XG5cbiAgICBpZiAoZGVwZW5kYWJvdCkge1xuICAgICAgY29uc3QgZGVmYXVsdE9wdGlvbnMgPSB7XG4gICAgICAgIGxhYmVsczogYXV0b0FwcHJvdmVMYWJlbChkZXBzQXV0b0FwcHJvdmUpLFxuICAgICAgfTtcbiAgICAgIHRoaXMuZ2l0aHViPy5hZGREZXBlbmRhYm90KFxuICAgICAgICBkZWVwTWVyZ2UoW2RlZmF1bHRPcHRpb25zLCBvcHRpb25zLmRlcGVuZGFib3RPcHRpb25zID8/IHt9XSlcbiAgICAgICk7XG4gICAgfVxuXG4gICAgaWYgKGRlcHNVcGdyYWRlKSB7XG4gICAgICBjb25zdCBkZWZhdWx0T3B0aW9uczogVXBncmFkZURlcGVuZGVuY2llc09wdGlvbnMgPSB7XG4gICAgICAgIHdvcmtmbG93T3B0aW9uczoge1xuICAgICAgICAgIGNvbnRhaW5lcjogb3B0aW9ucy53b3JrZmxvd0NvbnRhaW5lckltYWdlXG4gICAgICAgICAgICA/IHtcbiAgICAgICAgICAgICAgICBpbWFnZTogb3B0aW9ucy53b3JrZmxvd0NvbnRhaW5lckltYWdlLFxuICAgICAgICAgICAgICB9XG4gICAgICAgICAgICA6IHVuZGVmaW5lZCxcbiAgICAgICAgICBsYWJlbHM6IGF1dG9BcHByb3ZlTGFiZWwoZGVwc0F1dG9BcHByb3ZlKSxcbiAgICAgICAgICBnaXRJZGVudGl0eTogdGhpcy53b3JrZmxvd0dpdElkZW50aXR5LFxuICAgICAgICAgIHBlcm1pc3Npb25zOiB3b3JrZmxvd1Blcm1pc3Npb25zLFxuICAgICAgICB9LFxuICAgICAgfTtcbiAgICAgIHRoaXMudXBncmFkZVdvcmtmbG93ID0gbmV3IFVwZ3JhZGVEZXBlbmRlbmNpZXMoXG4gICAgICAgIHRoaXMsXG4gICAgICAgIGRlZXBNZXJnZShbZGVmYXVsdE9wdGlvbnMsIG9wdGlvbnMuZGVwc1VwZ3JhZGVPcHRpb25zID8/IHt9XSlcbiAgICAgICk7XG4gICAgfVxuXG4gICAgaWYgKG9wdGlvbnMucHVsbFJlcXVlc3RUZW1wbGF0ZSA/PyB0cnVlKSB7XG4gICAgICB0aGlzLmdpdGh1Yj8uYWRkUHVsbFJlcXVlc3RUZW1wbGF0ZShcbiAgICAgICAgLi4uKG9wdGlvbnMucHVsbFJlcXVlc3RUZW1wbGF0ZUNvbnRlbnRzID8/IFtdKVxuICAgICAgKTtcbiAgICB9XG5cbiAgICBjb25zdCBwcm9qZW5yY0pzID0gb3B0aW9ucy5wcm9qZW5yY0pzID8/ICFvcHRpb25zLnByb2plbnJjSnNvbjtcbiAgICBpZiAoIXRoaXMucGFyZW50ICYmIHByb2plbnJjSnMpIHtcbiAgICAgIG5ldyBQcm9qZW5yYyh0aGlzLCBvcHRpb25zLnByb2plbnJjSnNPcHRpb25zKTtcbiAgICB9XG5cbiAgICAvLyBhZGQgYSBidW5kbGVyIGNvbXBvbmVudCAtIHRoaXMgZW5hYmxlcyB0aGluZ3MgbGlrZSBMYW1iZGEgYnVuZGxpbmcgYW5kIGluIHRoZSBmdXR1cmUgd2ViIGJ1bmRsaW5nLlxuICAgIHRoaXMuYnVuZGxlciA9IG5ldyBCdW5kbGVyKHRoaXMsIG9wdGlvbnMuYnVuZGxlck9wdGlvbnMpO1xuXG4gICAgaWYgKG9wdGlvbnMucGFja2FnZSA/PyB0cnVlKSB7XG4gICAgICB0aGlzLnBhY2thZ2VUYXNrLmV4ZWMoYG1rZGlyIC1wICR7dGhpcy5hcnRpZmFjdHNKYXZhc2NyaXB0RGlyZWN0b3J5fWApO1xuXG4gICAgICAvLyBhbHdheXMgdXNlIG5wbSBoZXJlIC0geWFybiBkb2Vzbid0IGFkZCBtdWNoIHZhbHVlXG4gICAgICAvLyBzYWRseSB3ZSBjYW5ub3QgdXNlIC0tcGFjay1kZXN0aW5hdGlvbiBiZWNhdXNlIGl0IGlzIG5vdCBzdXBwb3J0ZWQgYnkgb2xkZXIgbnBtXG4gICAgICB0aGlzLnBhY2thZ2VUYXNrLmV4ZWMoXG4gICAgICAgIGBtdiAkKG5wbSBwYWNrKSAke3RoaXMuYXJ0aWZhY3RzSmF2YXNjcmlwdERpcmVjdG9yeX0vYFxuICAgICAgKTtcbiAgICB9XG5cbiAgICBpZiAob3B0aW9ucy5wcmV0dGllciA/PyBmYWxzZSkge1xuICAgICAgdGhpcy5wcmV0dGllciA9IG5ldyBQcmV0dGllcih0aGlzLCB7IC4uLm9wdGlvbnMucHJldHRpZXJPcHRpb25zIH0pO1xuICAgIH1cbiAgfVxuXG4gIHByaXZhdGUgcmVuZGVyVXBsb2FkQ292ZXJhZ2VKb2JTdGVwKG9wdGlvbnM6IE5vZGVQcm9qZWN0T3B0aW9ucyk6IEpvYlN0ZXBbXSB7XG4gICAgLy8gcnVuIGNvZGVjb3YgaWYgZW5hYmxlZCBvciBhIHNlY3JldCB0b2tlbiBuYW1lIGlzIHBhc3NlZCBpblxuICAgIC8vIEFORCBqZXN0IG11c3QgYmUgY29uZmlndXJlZFxuICAgIGlmICgob3B0aW9ucy5jb2RlQ292IHx8IG9wdGlvbnMuY29kZUNvdlRva2VuU2VjcmV0KSAmJiB0aGlzLmplc3Q/LmNvbmZpZykge1xuICAgICAgcmV0dXJuIFtcbiAgICAgICAge1xuICAgICAgICAgIG5hbWU6IFwiVXBsb2FkIGNvdmVyYWdlIHRvIENvZGVjb3ZcIixcbiAgICAgICAgICB1c2VzOiBcImNvZGVjb3YvY29kZWNvdi1hY3Rpb25AdjNcIixcbiAgICAgICAgICB3aXRoOiBvcHRpb25zLmNvZGVDb3ZUb2tlblNlY3JldFxuICAgICAgICAgICAgPyB7XG4gICAgICAgICAgICAgICAgdG9rZW46IGBcXCR7eyBzZWNyZXRzLiR7b3B0aW9ucy5jb2RlQ292VG9rZW5TZWNyZXR9IH19YCxcbiAgICAgICAgICAgICAgICBkaXJlY3Rvcnk6IHRoaXMuamVzdC5jb25maWcuY292ZXJhZ2VEaXJlY3RvcnksXG4gICAgICAgICAgICAgIH1cbiAgICAgICAgICAgIDoge1xuICAgICAgICAgICAgICAgIGRpcmVjdG9yeTogdGhpcy5qZXN0LmNvbmZpZy5jb3ZlcmFnZURpcmVjdG9yeSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgIF07XG4gICAgfSBlbHNlIHtcbiAgICAgIHJldHVybiBbXTtcbiAgICB9XG4gIH1cblxuICBwdWJsaWMgYWRkQmlucyhiaW5zOiBSZWNvcmQ8c3RyaW5nLCBzdHJpbmc+KSB7XG4gICAgdGhpcy5wYWNrYWdlLmFkZEJpbihiaW5zKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXBsYWNlcyB0aGUgY29udGVudHMgb2YgYW4gbnBtIHBhY2thZ2UuanNvbiBzY3JpcHQuXG4gICAqXG4gICAqIEBwYXJhbSBuYW1lIFRoZSBzY3JpcHQgbmFtZVxuICAgKiBAcGFyYW0gY29tbWFuZCBUaGUgY29tbWFuZCB0byBleGVjdXRlXG4gICAqL1xuICBwdWJsaWMgc2V0U2NyaXB0KG5hbWU6IHN0cmluZywgY29tbWFuZDogc3RyaW5nKSB7XG4gICAgdGhpcy5wYWNrYWdlLnNldFNjcmlwdChuYW1lLCBjb21tYW5kKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZW1vdmVzIHRoZSBucG0gc2NyaXB0IChhbHdheXMgc3VjY2Vzc2Z1bCkuXG4gICAqIEBwYXJhbSBuYW1lIFRoZSBuYW1lIG9mIHRoZSBzY3JpcHQuXG4gICAqL1xuICBwdWJsaWMgcmVtb3ZlU2NyaXB0KG5hbWU6IHN0cmluZykge1xuICAgIHRoaXMucGFja2FnZS5yZW1vdmVTY3JpcHQobmFtZSk7XG4gIH1cblxuICAvKipcbiAgICogSW5kaWNhdGVzIGlmIGEgc2NyaXB0IGJ5IHRoZSBuYW1lIG5hbWUgaXMgZGVmaW5lZC5cbiAgICogQHBhcmFtIG5hbWUgVGhlIG5hbWUgb2YgdGhlIHNjcmlwdFxuICAgKi9cbiAgcHVibGljIGhhc1NjcmlwdChuYW1lOiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5wYWNrYWdlLmhhc1NjcmlwdChuYW1lKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBERVBSRUNBVEVEXG4gICAqIEBkZXByZWNhdGVkIHVzZSBgcHJvamVjdC5jb21waWxlVGFzay5leGVjKClgXG4gICAqL1xuICBwdWJsaWMgYWRkQ29tcGlsZUNvbW1hbmQoLi4uY29tbWFuZHM6IHN0cmluZ1tdKSB7XG4gICAgZm9yIChjb25zdCBjIG9mIGNvbW1hbmRzKSB7XG4gICAgICB0aGlzLmNvbXBpbGVUYXNrLmV4ZWMoYyk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIERFUFJFQ0FURURcbiAgICogQGRlcHJlY2F0ZWQgdXNlIGBwcm9qZWN0LnRlc3RUYXNrLmV4ZWMoKWBcbiAgICovXG4gIHB1YmxpYyBhZGRUZXN0Q29tbWFuZCguLi5jb21tYW5kczogc3RyaW5nW10pIHtcbiAgICBmb3IgKGNvbnN0IGMgb2YgY29tbWFuZHMpIHtcbiAgICAgIHRoaXMudGVzdFRhc2suZXhlYyhjKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogRGlyZWN0bHkgc2V0IGZpZWxkcyBpbiBgcGFja2FnZS5qc29uYC5cbiAgICogQHBhcmFtIGZpZWxkcyBUaGUgZmllbGRzIHRvIHNldFxuICAgKi9cbiAgcHVibGljIGFkZEZpZWxkcyhmaWVsZHM6IHsgW25hbWU6IHN0cmluZ106IGFueSB9KSB7XG4gICAgZm9yIChjb25zdCBbbmFtZSwgdmFsdWVdIG9mIE9iamVjdC5lbnRyaWVzKGZpZWxkcykpIHtcbiAgICAgIHRoaXMucGFja2FnZS5hZGRGaWVsZChuYW1lLCB2YWx1ZSk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMga2V5d29yZHMgdG8gcGFja2FnZS5qc29uIChkZWR1cGxpY2F0ZWQpXG4gICAqIEBwYXJhbSBrZXl3b3JkcyBUaGUga2V5d29yZHMgdG8gYWRkXG4gICAqL1xuICBwdWJsaWMgYWRkS2V5d29yZHMoLi4ua2V5d29yZHM6IHN0cmluZ1tdKSB7XG4gICAgdGhpcy5wYWNrYWdlLmFkZEtleXdvcmRzKC4uLmtleXdvcmRzKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHZXQgc3RlcHMgZm9yIHNjb3BlZCBwYWNrYWdlIGFjY2Vzc1xuICAgKlxuICAgKiBAcGFyYW0gY29kZUFydGlmYWN0T3B0aW9ucyBEZXRhaWxzIG9mIGxvZ2dpbmcgaW4gdG8gQVdTXG4gICAqIEByZXR1cm5zIGFycmF5IG9mIGpvYiBzdGVwcyByZXF1aXJlZCBmb3IgZWFjaCBwcml2YXRlIHNjb3BlZCBwYWNrYWdlc1xuICAgKi9cbiAgcHJpdmF0ZSBnZXRTY29wZWRQYWNrYWdlU3RlcHMoXG4gICAgY29kZUFydGlmYWN0T3B0aW9uczogQ29kZUFydGlmYWN0T3B0aW9ucyB8IHVuZGVmaW5lZFxuICApOiBKb2JTdGVwW10ge1xuICAgIGNvbnN0IHBhcnNlZENvZGVBcnRpZmFjdE9wdGlvbnMgPSB7XG4gICAgICBhY2Nlc3NLZXlJZFNlY3JldDpcbiAgICAgICAgY29kZUFydGlmYWN0T3B0aW9ucz8uYWNjZXNzS2V5SWRTZWNyZXQgPz8gXCJBV1NfQUNDRVNTX0tFWV9JRFwiLFxuICAgICAgc2VjcmV0QWNjZXNzS2V5U2VjcmV0OlxuICAgICAgICBjb2RlQXJ0aWZhY3RPcHRpb25zPy5zZWNyZXRBY2Nlc3NLZXlTZWNyZXQgPz8gXCJBV1NfU0VDUkVUX0FDQ0VTU19LRVlcIixcbiAgICAgIHJvbGVUb0Fzc3VtZTogY29kZUFydGlmYWN0T3B0aW9ucz8ucm9sZVRvQXNzdW1lLFxuICAgICAgYXV0aFByb3ZpZGVyOiBjb2RlQXJ0aWZhY3RPcHRpb25zPy5hdXRoUHJvdmlkZXIsXG4gICAgfTtcblxuICAgIGlmIChcbiAgICAgIHBhcnNlZENvZGVBcnRpZmFjdE9wdGlvbnMuYXV0aFByb3ZpZGVyID09PVxuICAgICAgTm9kZVBhY2thZ2VDb2RlQXJ0aWZhY3RBdXRoUHJvdmlkZXIuR0lUSFVCX09JRENcbiAgICApIHtcbiAgICAgIHJldHVybiBbXG4gICAgICAgIHtcbiAgICAgICAgICBuYW1lOiBcIkNvbmZpZ3VyZSBBV1MgQ3JlZGVudGlhbHNcIixcbiAgICAgICAgICB1c2VzOiBcImF3cy1hY3Rpb25zL2NvbmZpZ3VyZS1hd3MtY3JlZGVudGlhbHNAdjJcIixcbiAgICAgICAgICB3aXRoOiB7XG4gICAgICAgICAgICBcImF3cy1yZWdpb25cIjogXCJ1cy1lYXN0LTJcIixcbiAgICAgICAgICAgIFwicm9sZS10by1hc3N1bWVcIjogcGFyc2VkQ29kZUFydGlmYWN0T3B0aW9ucy5yb2xlVG9Bc3N1bWUsXG4gICAgICAgICAgICBcInJvbGUtZHVyYXRpb24tc2Vjb25kc1wiOiA5MDAsXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgICAge1xuICAgICAgICAgIG5hbWU6IFwiQVdTIENvZGVBcnRpZmFjdCBMb2dpblwiLFxuICAgICAgICAgIHJ1bjogYCR7dGhpcy5ydW5TY3JpcHRDb21tYW5kfSBjYTpsb2dpbmAsXG4gICAgICAgIH0sXG4gICAgICBdO1xuICAgIH1cblxuICAgIGlmIChwYXJzZWRDb2RlQXJ0aWZhY3RPcHRpb25zLnJvbGVUb0Fzc3VtZSkge1xuICAgICAgcmV0dXJuIFtcbiAgICAgICAge1xuICAgICAgICAgIG5hbWU6IFwiQ29uZmlndXJlIEFXUyBDcmVkZW50aWFsc1wiLFxuICAgICAgICAgIHVzZXM6IFwiYXdzLWFjdGlvbnMvY29uZmlndXJlLWF3cy1jcmVkZW50aWFsc0B2MlwiLFxuICAgICAgICAgIHdpdGg6IHtcbiAgICAgICAgICAgIFwiYXdzLWFjY2Vzcy1rZXktaWRcIjogc2VjcmV0VG9TdHJpbmcoXG4gICAgICAgICAgICAgIHBhcnNlZENvZGVBcnRpZmFjdE9wdGlvbnMuYWNjZXNzS2V5SWRTZWNyZXRcbiAgICAgICAgICAgICksXG4gICAgICAgICAgICBcImF3cy1zZWNyZXQtYWNjZXNzLWtleVwiOiBzZWNyZXRUb1N0cmluZyhcbiAgICAgICAgICAgICAgcGFyc2VkQ29kZUFydGlmYWN0T3B0aW9ucy5zZWNyZXRBY2Nlc3NLZXlTZWNyZXRcbiAgICAgICAgICAgICksXG4gICAgICAgICAgICBcImF3cy1yZWdpb25cIjogXCJ1cy1lYXN0LTJcIixcbiAgICAgICAgICAgIFwicm9sZS10by1hc3N1bWVcIjogcGFyc2VkQ29kZUFydGlmYWN0T3B0aW9ucy5yb2xlVG9Bc3N1bWUsXG4gICAgICAgICAgICBcInJvbGUtZHVyYXRpb24tc2Vjb25kc1wiOiA5MDAsXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgICAge1xuICAgICAgICAgIG5hbWU6IFwiQVdTIENvZGVBcnRpZmFjdCBMb2dpblwiLFxuICAgICAgICAgIHJ1bjogYCR7dGhpcy5ydW5TY3JpcHRDb21tYW5kfSBjYTpsb2dpbmAsXG4gICAgICAgIH0sXG4gICAgICBdO1xuICAgIH1cblxuICAgIHJldHVybiBbXG4gICAgICB7XG4gICAgICAgIG5hbWU6IFwiQVdTIENvZGVBcnRpZmFjdCBMb2dpblwiLFxuICAgICAgICBydW46IGAke3RoaXMucnVuU2NyaXB0Q29tbWFuZH0gY2E6bG9naW5gLFxuICAgICAgICBlbnY6IHtcbiAgICAgICAgICBBV1NfQUNDRVNTX0tFWV9JRDogc2VjcmV0VG9TdHJpbmcoXG4gICAgICAgICAgICBwYXJzZWRDb2RlQXJ0aWZhY3RPcHRpb25zLmFjY2Vzc0tleUlkU2VjcmV0XG4gICAgICAgICAgKSxcbiAgICAgICAgICBBV1NfU0VDUkVUX0FDQ0VTU19LRVk6IHNlY3JldFRvU3RyaW5nKFxuICAgICAgICAgICAgcGFyc2VkQ29kZUFydGlmYWN0T3B0aW9ucy5zZWNyZXRBY2Nlc3NLZXlTZWNyZXRcbiAgICAgICAgICApLFxuICAgICAgICB9LFxuICAgICAgfSxcbiAgICBdO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgdGhlIHNldCBvZiB3b3JrZmxvdyBzdGVwcyB3aGljaCBzaG91bGQgYmUgZXhlY3V0ZWQgdG8gYm9vdHN0cmFwIGFcbiAgICogd29ya2Zsb3cuXG4gICAqXG4gICAqIEBwYXJhbSBvcHRpb25zIE9wdGlvbnMuXG4gICAqIEByZXR1cm5zIEpvYiBzdGVwc1xuICAgKi9cbiAgcHVibGljIHJlbmRlcldvcmtmbG93U2V0dXAoXG4gICAgb3B0aW9uczogUmVuZGVyV29ya2Zsb3dTZXR1cE9wdGlvbnMgPSB7fVxuICApOiBKb2JTdGVwW10ge1xuICAgIGNvbnN0IGluc3RhbGwgPSBuZXcgQXJyYXk8Sm9iU3RlcD4oKTtcblxuICAgIC8vIGZpcnN0IHJ1biB0aGUgd29ya2Zsb3cgYm9vdHN0cmFwIHN0ZXBzXG4gICAgaW5zdGFsbC5wdXNoKC4uLnRoaXMud29ya2Zsb3dCb290c3RyYXBTdGVwcyk7XG5cbiAgICBpZiAodGhpcy5ub2RlVmVyc2lvbikge1xuICAgICAgaW5zdGFsbC5wdXNoKHtcbiAgICAgICAgbmFtZTogXCJTZXR1cCBOb2RlLmpzXCIsXG4gICAgICAgIHVzZXM6IFwiYWN0aW9ucy9zZXR1cC1ub2RlQHYzXCIsXG4gICAgICAgIHdpdGg6IHsgXCJub2RlLXZlcnNpb25cIjogdGhpcy5ub2RlVmVyc2lvbiB9LFxuICAgICAgfSk7XG4gICAgfVxuXG4gICAgaWYgKHRoaXMucGFja2FnZS5wYWNrYWdlTWFuYWdlciA9PT0gTm9kZVBhY2thZ2VNYW5hZ2VyLlBOUE0pIHtcbiAgICAgIGluc3RhbGwucHVzaCh7XG4gICAgICAgIG5hbWU6IFwiU2V0dXAgcG5wbVwiLFxuICAgICAgICB1c2VzOiBcInBucG0vYWN0aW9uLXNldHVwQHYyLjIuNFwiLFxuICAgICAgICB3aXRoOiB7IHZlcnNpb246IHRoaXMucGFja2FnZS5wbnBtVmVyc2lvbiB9LFxuICAgICAgfSk7XG4gICAgfVxuXG4gICAgY29uc3QgbXV0YWJsZSA9IG9wdGlvbnMubXV0YWJsZSA/PyBmYWxzZTtcblxuICAgIGlmICh0aGlzLnBhY2thZ2Uuc2NvcGVkUGFja2FnZXNPcHRpb25zKSB7XG4gICAgICBpbnN0YWxsLnB1c2goXG4gICAgICAgIC4uLnRoaXMuZ2V0U2NvcGVkUGFja2FnZVN0ZXBzKHRoaXMucGFja2FnZS5jb2RlQXJ0aWZhY3RPcHRpb25zKVxuICAgICAgKTtcbiAgICB9XG5cbiAgICBpbnN0YWxsLnB1c2goe1xuICAgICAgbmFtZTogXCJJbnN0YWxsIGRlcGVuZGVuY2llc1wiLFxuICAgICAgcnVuOiBtdXRhYmxlXG4gICAgICAgID8gdGhpcy5wYWNrYWdlLmluc3RhbGxBbmRVcGRhdGVMb2NrZmlsZUNvbW1hbmRcbiAgICAgICAgOiB0aGlzLnBhY2thZ2UuaW5zdGFsbENvbW1hbmQsXG4gICAgfSk7XG5cbiAgICByZXR1cm4gaW5zdGFsbDtcbiAgfVxuXG4gIC8qKlxuICAgKiBEZWZpbmVzIG5vcm1hbCBkZXBlbmRlbmNpZXMuXG4gICAqXG4gICAqIEBwYXJhbSBkZXBzIE5hbWVzIG1vZHVsZXMgdG8gaW5zdGFsbC4gQnkgZGVmYXVsdCwgdGhlIHRoZSBkZXBlbmRlbmN5IHdpbGxcbiAgICogYmUgaW5zdGFsbGVkIGluIHRoZSBuZXh0IGBucHggcHJvamVuYCBydW4gYW5kIHRoZSB2ZXJzaW9uIHdpbGwgYmUgcmVjb3JkZWRcbiAgICogaW4geW91ciBgcGFja2FnZS5qc29uYCBmaWxlLiBZb3UgY2FuIHVwZ3JhZGUgbWFudWFsbHkgb3IgdXNpbmcgYHlhcm5cbiAgICogYWRkL3VwZ3JhZGVgLiBJZiB5b3Ugd2lzaCB0byBzcGVjaWZ5IGEgdmVyc2lvbiByYW5nZSB1c2UgdGhpcyBzeW50YXg6XG4gICAqIGBtb2R1bGVAXjdgLlxuICAgKi9cbiAgcHVibGljIGFkZERlcHMoLi4uZGVwczogc3RyaW5nW10pIHtcbiAgICByZXR1cm4gdGhpcy5wYWNrYWdlLmFkZERlcHMoLi4uZGVwcyk7XG4gIH1cblxuICAvKipcbiAgICogRGVmaW5lcyBkZXZlbG9wbWVudC90ZXN0IGRlcGVuZGVuY2llcy5cbiAgICpcbiAgICogQHBhcmFtIGRlcHMgTmFtZXMgbW9kdWxlcyB0byBpbnN0YWxsLiBCeSBkZWZhdWx0LCB0aGUgdGhlIGRlcGVuZGVuY3kgd2lsbFxuICAgKiBiZSBpbnN0YWxsZWQgaW4gdGhlIG5leHQgYG5weCBwcm9qZW5gIHJ1biBhbmQgdGhlIHZlcnNpb24gd2lsbCBiZSByZWNvcmRlZFxuICAgKiBpbiB5b3VyIGBwYWNrYWdlLmpzb25gIGZpbGUuIFlvdSBjYW4gdXBncmFkZSBtYW51YWxseSBvciB1c2luZyBgeWFyblxuICAgKiBhZGQvdXBncmFkZWAuIElmIHlvdSB3aXNoIHRvIHNwZWNpZnkgYSB2ZXJzaW9uIHJhbmdlIHVzZSB0aGlzIHN5bnRheDpcbiAgICogYG1vZHVsZUBeN2AuXG4gICAqL1xuICBwdWJsaWMgYWRkRGV2RGVwcyguLi5kZXBzOiBzdHJpbmdbXSkge1xuICAgIHJldHVybiB0aGlzLnBhY2thZ2UuYWRkRGV2RGVwcyguLi5kZXBzKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBEZWZpbmVzIHBlZXIgZGVwZW5kZW5jaWVzLlxuICAgKlxuICAgKiBXaGVuIGFkZGluZyBwZWVyIGRlcGVuZGVuY2llcywgYSBkZXZEZXBlbmRlbmN5IHdpbGwgYWxzbyBiZSBhZGRlZCBvbiB0aGVcbiAgICogcGlubmVkIHZlcnNpb24gb2YgdGhlIGRlY2xhcmVkIHBlZXIuIFRoaXMgd2lsbCBlbnN1cmUgdGhhdCB5b3UgYXJlIHRlc3RpbmdcbiAgICogeW91ciBjb2RlIGFnYWluc3QgdGhlIG1pbmltdW0gdmVyc2lvbiByZXF1aXJlZCBmcm9tIHlvdXIgY29uc3VtZXJzLlxuICAgKlxuICAgKiBAcGFyYW0gZGVwcyBOYW1lcyBtb2R1bGVzIHRvIGluc3RhbGwuIEJ5IGRlZmF1bHQsIHRoZSB0aGUgZGVwZW5kZW5jeSB3aWxsXG4gICAqIGJlIGluc3RhbGxlZCBpbiB0aGUgbmV4dCBgbnB4IHByb2plbmAgcnVuIGFuZCB0aGUgdmVyc2lvbiB3aWxsIGJlIHJlY29yZGVkXG4gICAqIGluIHlvdXIgYHBhY2thZ2UuanNvbmAgZmlsZS4gWW91IGNhbiB1cGdyYWRlIG1hbnVhbGx5IG9yIHVzaW5nIGB5YXJuXG4gICAqIGFkZC91cGdyYWRlYC4gSWYgeW91IHdpc2ggdG8gc3BlY2lmeSBhIHZlcnNpb24gcmFuZ2UgdXNlIHRoaXMgc3ludGF4OlxuICAgKiBgbW9kdWxlQF43YC5cbiAgICovXG4gIHB1YmxpYyBhZGRQZWVyRGVwcyguLi5kZXBzOiBzdHJpbmdbXSkge1xuICAgIHJldHVybiB0aGlzLnBhY2thZ2UuYWRkUGVlckRlcHMoLi4uZGVwcyk7XG4gIH1cblxuICAvKipcbiAgICogRGVmaW5lcyBidW5kbGVkIGRlcGVuZGVuY2llcy5cbiAgICpcbiAgICogQnVuZGxlZCBkZXBlbmRlbmNpZXMgd2lsbCBiZSBhZGRlZCBhcyBub3JtYWwgZGVwZW5kZW5jaWVzIGFzIHdlbGwgYXMgdG8gdGhlXG4gICAqIGBidW5kbGVkRGVwZW5kZW5jaWVzYCBzZWN0aW9uIG9mIHlvdXIgYHBhY2thZ2UuanNvbmAuXG4gICAqXG4gICAqIEBwYXJhbSBkZXBzIE5hbWVzIG1vZHVsZXMgdG8gaW5zdGFsbC4gQnkgZGVmYXVsdCwgdGhlIHRoZSBkZXBlbmRlbmN5IHdpbGxcbiAgICogYmUgaW5zdGFsbGVkIGluIHRoZSBuZXh0IGBucHggcHJvamVuYCBydW4gYW5kIHRoZSB2ZXJzaW9uIHdpbGwgYmUgcmVjb3JkZWRcbiAgICogaW4geW91ciBgcGFja2FnZS5qc29uYCBmaWxlLiBZb3UgY2FuIHVwZ3JhZGUgbWFudWFsbHkgb3IgdXNpbmcgYHlhcm5cbiAgICogYWRkL3VwZ3JhZGVgLiBJZiB5b3Ugd2lzaCB0byBzcGVjaWZ5IGEgdmVyc2lvbiByYW5nZSB1c2UgdGhpcyBzeW50YXg6XG4gICAqIGBtb2R1bGVAXjdgLlxuICAgKi9cbiAgcHVibGljIGFkZEJ1bmRsZWREZXBzKC4uLmRlcHM6IHN0cmluZ1tdKSB7XG4gICAgcmV0dXJuIHRoaXMucGFja2FnZS5hZGRCdW5kbGVkRGVwcyguLi5kZXBzKTtcbiAgfVxuXG4gIHB1YmxpYyBhZGRQYWNrYWdlSWdub3JlKHBhdHRlcm46IHN0cmluZykge1xuICAgIHRoaXMubnBtaWdub3JlPy5hZGRQYXR0ZXJucyhwYXR0ZXJuKTtcbiAgfVxuXG4gIHByaXZhdGUgYWRkTGljZW5zZShvcHRpb25zOiBOb2RlUHJvamVjdE9wdGlvbnMpIHtcbiAgICBpZiAodGhpcy5wYWNrYWdlLmxpY2Vuc2UpIHtcbiAgICAgIG5ldyBMaWNlbnNlKHRoaXMsIHtcbiAgICAgICAgc3BkeDogdGhpcy5wYWNrYWdlLmxpY2Vuc2UsXG4gICAgICAgIGNvcHlyaWdodE93bmVyOiBvcHRpb25zLmNvcHlyaWdodE93bmVyID8/IG9wdGlvbnMuYXV0aG9yTmFtZSxcbiAgICAgICAgY29weXJpZ2h0UGVyaW9kOiBvcHRpb25zLmNvcHlyaWdodFBlcmlvZCxcbiAgICAgIH0pO1xuICAgIH1cbiAgfVxuXG4gIHByaXZhdGUgYWRkRGVmYXVsdEdpdElnbm9yZSgpIHtcbiAgICB0aGlzLmdpdGlnbm9yZS5leGNsdWRlKFxuICAgICAgXCIjIExvZ3NcIixcbiAgICAgIFwibG9nc1wiLFxuICAgICAgXCIqLmxvZ1wiLFxuICAgICAgXCJucG0tZGVidWcubG9nKlwiLFxuICAgICAgXCJ5YXJuLWRlYnVnLmxvZypcIixcbiAgICAgIFwieWFybi1lcnJvci5sb2cqXCIsXG4gICAgICBcImxlcm5hLWRlYnVnLmxvZypcIixcblxuICAgICAgXCIjIERpYWdub3N0aWMgcmVwb3J0cyAoaHR0cHM6Ly9ub2RlanMub3JnL2FwaS9yZXBvcnQuaHRtbClcIixcbiAgICAgIFwicmVwb3J0LlswLTldKi5bMC05XSouWzAtOV0qLlswLTldKi5qc29uXCIsXG5cbiAgICAgIFwiIyBSdW50aW1lIGRhdGFcIixcbiAgICAgIFwicGlkc1wiLFxuICAgICAgXCIqLnBpZFwiLFxuICAgICAgXCIqLnNlZWRcIixcbiAgICAgIFwiKi5waWQubG9ja1wiLFxuXG4gICAgICBcIiMgRGlyZWN0b3J5IGZvciBpbnN0cnVtZW50ZWQgbGlicyBnZW5lcmF0ZWQgYnkganNjb3ZlcmFnZS9KU0NvdmVyXCIsXG4gICAgICBcImxpYi1jb3ZcIixcblxuICAgICAgXCIjIENvdmVyYWdlIGRpcmVjdG9yeSB1c2VkIGJ5IHRvb2xzIGxpa2UgaXN0YW5idWxcIixcbiAgICAgIFwiY292ZXJhZ2VcIixcbiAgICAgIFwiKi5sY292XCIsXG5cbiAgICAgIFwiIyBueWMgdGVzdCBjb3ZlcmFnZVwiLFxuICAgICAgXCIubnljX291dHB1dFwiLFxuXG4gICAgICBcIiMgQ29tcGlsZWQgYmluYXJ5IGFkZG9ucyAoaHR0cHM6Ly9ub2RlanMub3JnL2FwaS9hZGRvbnMuaHRtbClcIixcbiAgICAgIFwiYnVpbGQvUmVsZWFzZVwiLFxuXG4gICAgICBcIiMgRGVwZW5kZW5jeSBkaXJlY3Rvcmllc1wiLFxuICAgICAgXCJub2RlX21vZHVsZXMvXCIsXG4gICAgICBcImpzcG1fcGFja2FnZXMvXCIsXG5cbiAgICAgIFwiIyBUeXBlU2NyaXB0IGNhY2hlXCIsXG4gICAgICBcIioudHNidWlsZGluZm9cIixcblxuICAgICAgXCIjIE9wdGlvbmFsIGVzbGludCBjYWNoZVwiLFxuICAgICAgXCIuZXNsaW50Y2FjaGVcIixcblxuICAgICAgXCIjIE91dHB1dCBvZiAnbnBtIHBhY2snXCIsXG4gICAgICBcIioudGd6XCIsXG5cbiAgICAgIFwiIyBZYXJuIEludGVncml0eSBmaWxlXCIsXG4gICAgICBcIi55YXJuLWludGVncml0eVwiLFxuXG4gICAgICBcIiMgcGFyY2VsLWJ1bmRsZXIgY2FjaGUgKGh0dHBzOi8vcGFyY2VsanMub3JnLylcIixcbiAgICAgIFwiLmNhY2hlXCJcbiAgICApO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgdGhlIHNoZWxsIGNvbW1hbmQgdG8gZXhlY3V0ZSBpbiBvcmRlciB0byBydW4gYSB0YXNrLiBUaGlzIHdpbGxcbiAgICogdHlwaWNhbGx5IGJlIGBucHggcHJvamVuIFRBU0tgLlxuICAgKlxuICAgKiBAcGFyYW0gdGFzayBUaGUgdGFzayBmb3Igd2hpY2ggdGhlIGNvbW1hbmQgaXMgcmVxdWlyZWRcbiAgICovXG4gIHB1YmxpYyBydW5UYXNrQ29tbWFuZCh0YXNrOiBUYXNrKSB7XG4gICAgcmV0dXJuIGAke3RoaXMucGFja2FnZS5wcm9qZW5Db21tYW5kfSAke3Rhc2submFtZX1gO1xuICB9XG5cbiAgLyoqXG4gICAqIFRoZSBqb2IgSUQgb2YgdGhlIGJ1aWxkIHdvcmtmbG93LlxuICAgKi9cbiAgcHVibGljIGdldCBidWlsZFdvcmtmbG93Sm9iSWQoKSB7XG4gICAgcmV0dXJuIHRoaXMuYnVpbGRXb3JrZmxvdz8uYnVpbGRKb2JJZHNbMF07XG4gIH1cbn1cblxuLyoqXG4gKiBPcHRpb25zIGZvciBgcmVuZGVySW5zdGFsbFN0ZXBzKClgLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIFJlbmRlcldvcmtmbG93U2V0dXBPcHRpb25zIHtcbiAgLyoqXG4gICAqIFNob3VsZCB0aGUgcGFjYWtnZSBsb2NrZmlsZSBiZSB1cGRhdGVkP1xuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgbXV0YWJsZT86IGJvb2xlYW47XG59XG4iXX0=