"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.defaultNpmToken = exports.NpmAccess = exports.NodePackageManager = exports.NodePackage = exports.CodeArtifactAuthProvider = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs_1 = require("fs");
const path_1 = require("path");
const url_1 = require("url");
const util_1 = require("./util");
const _resolve_1 = require("../_resolve");
const component_1 = require("../component");
const dependencies_1 = require("../dependencies");
const json_1 = require("../json");
const release_1 = require("../release");
const task_runtime_1 = require("../task-runtime");
const util_2 = require("../util");
const UNLICENSED = "UNLICENSED";
const DEFAULT_NPM_REGISTRY_URL = "https://registry.npmjs.org/";
const GITHUB_PACKAGES_REGISTRY = "npm.pkg.github.com";
const DEFAULT_NPM_TOKEN_SECRET = "NPM_TOKEN";
const DEFAULT_GITHUB_TOKEN_SECRET = "GITHUB_TOKEN";
/**
 * Options for authorizing requests to a AWS CodeArtifact npm repository.
 */
var CodeArtifactAuthProvider;
(function (CodeArtifactAuthProvider) {
    /**
     * Fixed credentials provided via Github secrets.
     */
    CodeArtifactAuthProvider["ACCESS_AND_SECRET_KEY_PAIR"] = "ACCESS_AND_SECRET_KEY_PAIR";
    /**
     * Ephemeral credentials provided via Github's OIDC integration with an IAM role.
     * See:
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_providers_create_oidc.html
     * https://docs.github.com/en/actions/deployment/security-hardening-your-deployments/configuring-openid-connect-in-amazon-web-services
     */
    CodeArtifactAuthProvider["GITHUB_OIDC"] = "GITHUB_OIDC";
})(CodeArtifactAuthProvider = exports.CodeArtifactAuthProvider || (exports.CodeArtifactAuthProvider = {}));
/**
 * Represents the npm `package.json` file.
 */
class NodePackage extends component_1.Component {
    constructor(project, options = {}) {
        super(project);
        this.keywords = new Set();
        this.bin = {};
        this.engines = {};
        this.packageName = options.packageName ?? project.name;
        this.peerDependencyOptions = {
            pinnedDevDependency: true,
            ...options.peerDependencyOptions,
        };
        this.allowLibraryDependencies = options.allowLibraryDependencies ?? true;
        this.packageManager = options.packageManager ?? NodePackageManager.YARN;
        this.entrypoint = options.entrypoint ?? "lib/index.js";
        this.lockFile = determineLockfile(this.packageManager);
        this.project.annotateGenerated(`/${this.lockFile}`);
        const { npmAccess, npmRegistry, npmRegistryUrl, npmTokenSecret, codeArtifactOptions, scopedPackagesOptions, } = this.parseNpmOptions(options);
        this.npmAccess = npmAccess;
        this.npmRegistry = npmRegistry;
        this.npmRegistryUrl = npmRegistryUrl;
        this.npmTokenSecret = npmTokenSecret;
        this.codeArtifactOptions = codeArtifactOptions;
        this.scopedPackagesOptions = scopedPackagesOptions;
        this.processDeps(options);
        this.addCodeArtifactLoginScript();
        const prev = this.readPackageJson() ?? {};
        // empty objects are here to preserve order for backwards compatibility
        this.manifest = {
            name: this.packageName,
            description: options.description,
            repository: !options.repository
                ? undefined
                : {
                    type: "git",
                    url: options.repository,
                    directory: options.repositoryDirectory,
                },
            bin: () => this.renderBin(),
            scripts: () => this.renderScripts(),
            author: this.renderAuthor(options),
            devDependencies: {},
            peerDependencies: {},
            dependencies: {},
            bundledDependencies: [],
            keywords: () => this.renderKeywords(),
            engines: () => this.renderEngines(),
            main: this.entrypoint !== "" ? this.entrypoint : undefined,
            license: () => this.license ?? UNLICENSED,
            homepage: options.homepage,
            publishConfig: () => this.renderPublishConfig(),
            // in release CI builds we bump the version before we run "build" so we want
            // to preserve the version number. otherwise, we always set it to 0.0.0
            version: this.determineVersion(prev?.version),
            bugs: options.bugsEmail || options.bugsUrl
                ? {
                    email: options.bugsEmail,
                    url: options.bugsUrl,
                }
                : undefined,
        };
        // override any scripts from options (if specified)
        for (const [cmdname, shell] of Object.entries(options.scripts ?? {})) {
            project.addTask(cmdname, { exec: shell });
        }
        this.file = new json_1.JsonFile(this.project, "package.json", {
            obj: this.manifest,
            readonly: false,
            newline: true,
            committed: true,
        });
        this.addKeywords(...(options.keywords ?? []));
        this.addBin(options.bin ?? {});
        // automatically add all executable files under "bin"
        if (options.autoDetectBin ?? true) {
            this.autoDiscoverBinaries();
        }
        // node version
        this.minNodeVersion = options.minNodeVersion;
        this.maxNodeVersion = options.maxNodeVersion;
        this.pnpmVersion = options.pnpmVersion ?? "7";
        this.addNodeEngine();
        // license
        if (options.licensed ?? true) {
            this.license = options.license ?? "Apache-2.0";
        }
        this.installTask = project.addTask("install", {
            description: "Install project dependencies and update lockfile (non-frozen)",
            exec: this.installAndUpdateLockfileCommand,
        });
        this.installCiTask = project.addTask("install:ci", {
            description: "Install project dependencies using frozen lockfile",
            exec: this.installCommand,
        });
    }
    /**
     * Defines normal dependencies.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addDeps(...deps) {
        for (const dep of deps) {
            this.project.deps.addDependency(dep, dependencies_1.DependencyType.RUNTIME);
        }
    }
    /**
     * Defines development/test dependencies.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addDevDeps(...deps) {
        for (const dep of deps) {
            this.project.deps.addDependency(dep, dependencies_1.DependencyType.BUILD);
        }
    }
    /**
     * Defines peer dependencies.
     *
     * When adding peer dependencies, a devDependency will also be added on the
     * pinned version of the declared peer. This will ensure that you are testing
     * your code against the minimum version required from your consumers.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addPeerDeps(...deps) {
        if (Object.keys(deps).length && !this.allowLibraryDependencies) {
            throw new Error(`cannot add peer dependencies to an APP project: ${Object.keys(deps).join(",")}`);
        }
        for (const dep of deps) {
            this.project.deps.addDependency(dep, dependencies_1.DependencyType.PEER);
        }
    }
    /**
     * Defines bundled dependencies.
     *
     * Bundled dependencies will be added as normal dependencies as well as to the
     * `bundledDependencies` section of your `package.json`.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addBundledDeps(...deps) {
        if (deps.length && !this.allowLibraryDependencies) {
            throw new Error(`cannot add bundled dependencies to an APP project: ${deps.join(",")}`);
        }
        for (const dep of deps) {
            this.project.deps.addDependency(dep, dependencies_1.DependencyType.BUNDLED);
        }
    }
    /**
     * Adds an `engines` requirement to your package.
     * @param engine The engine (e.g. `node`)
     * @param version The semantic version requirement (e.g. `^10`)
     */
    addEngine(engine, version) {
        this.engines[engine] = version;
    }
    /**
     * Adds keywords to package.json (deduplicated)
     * @param keywords The keywords to add
     */
    addKeywords(...keywords) {
        for (const k of keywords) {
            this.keywords.add(k);
        }
    }
    addBin(bins) {
        for (const [k, v] of Object.entries(bins)) {
            this.bin[k] = v;
        }
    }
    /**
     * Override the contents of an npm package.json script.
     *
     * @param name The script name
     * @param command The command to execute
     */
    setScript(name, command) {
        this.file.addOverride(`scripts.${name}`, command);
    }
    /**
     * Removes the npm script (always successful).
     * @param name The name of the script.
     */
    removeScript(name) {
        this.file.addDeletionOverride(`scripts.${name}`);
    }
    /**
     * Indicates if a script by the given name is defined.
     * @param name The name of the script
     * @deprecated Use `project.tasks.tryFind(name)`
     */
    hasScript(name) {
        return this.project.tasks.tryFind(name) !== undefined;
    }
    /**
     * Directly set fields in `package.json`.
     * @escape
     * @param name field name
     * @param value field value
     */
    addField(name, value) {
        this.manifest[name] = value;
    }
    /**
     * Sets the package version.
     * @param version Package version.
     */
    addVersion(version) {
        this.manifest.version = version;
    }
    /**
     * Defines resolutions for dependencies to change the normally resolved
     * version of a dependency to something else.
     *
     * @param resolutions Names resolutions to be added. Specify a version or
     * range with this syntax:
     * `module@^7`.
     */
    addPackageResolutions(...resolutions) {
        const fieldName = util_1.packageResolutionsFieldName(this.packageManager);
        for (const resolution of resolutions) {
            this.project.deps.addDependency(resolution, dependencies_1.DependencyType.OVERRIDE);
            const { name, version = "*" } = dependencies_1.Dependencies.parseDependency(resolution);
            this.file.addOverride(`${fieldName}.${name}`, version);
        }
    }
    /**
     * Returns the command to execute in order to install all dependencies (always frozen).
     */
    get installCommand() {
        return this.renderInstallCommand(true);
    }
    /**
     * Renders `yarn install` or `npm install` with lockfile update (not frozen)
     */
    get installAndUpdateLockfileCommand() {
        return this.renderInstallCommand(false);
    }
    /**
     * Render a package manager specific command to upgrade all requested dependencies.
     */
    renderUpgradePackagesCommand(exclude, include) {
        const project = this.project;
        function upgradePackages(command) {
            return () => {
                if (exclude.length === 0 && !include) {
                    // request to upgrade all packages
                    // separated for asthetic reasons.
                    return command;
                }
                // filter by exclude and include.
                return `${command} ${project.deps.all
                    .filter((d) => d.type !== dependencies_1.DependencyType.OVERRIDE)
                    .map((d) => d.name)
                    .filter((d) => (include ? include.includes(d) : true))
                    .filter((d) => !exclude.includes(d))
                    .join(" ")}`;
            };
        }
        let lazy = undefined;
        switch (this.packageManager) {
            case NodePackageManager.YARN:
            case NodePackageManager.YARN2:
                lazy = upgradePackages("yarn upgrade");
                break;
            case NodePackageManager.NPM:
                lazy = upgradePackages("npm update");
                break;
            case NodePackageManager.PNPM:
                lazy = upgradePackages("pnpm update");
                break;
            default:
                throw new Error(`unexpected package manager ${this.packageManager}`);
        }
        // return a lazy function so that dependencies include ones that were
        // added post project instantiation (i.e using project.addDeps)
        return lazy;
    }
    // ---------------------------------------------------------------------------------------
    preSynthesize() {
        super.preSynthesize();
        this._renderedDeps = this.renderDependencies();
    }
    postSynthesize() {
        super.postSynthesize();
        // only run "install" if package.json has changed or if we don't have a
        // `node_modules` directory.
        if (this.file.changed ||
            !fs_1.existsSync(path_1.join(this.project.outdir, "node_modules"))) {
            this.installDependencies();
        }
        // resolve "*" deps in package.json and update it. if it was changed,
        // install deps again so that lockfile is updated.
        if (this.resolveDepsAndWritePackageJson()) {
            this.installDependencies();
        }
    }
    /**
     * The command which executes "projen".
     */
    get projenCommand() {
        return this.project.projenCommand;
    }
    /**
     * Returns `true` if we are running within a CI build.
     */
    get isAutomatedBuild() {
        return util_2.isTruthy(process.env.CI);
    }
    determineVersion(currVersion) {
        if (!this.isReleaseBuild) {
            return "0.0.0";
        }
        return currVersion ?? "0.0.0";
    }
    /**
     * Returns `true` if this is a CI release build.
     */
    get isReleaseBuild() {
        return util_2.isTruthy(process.env.RELEASE);
    }
    // -------------------------------------------------------------------------------------------
    parseNpmOptions(options) {
        let npmRegistryUrl = options.npmRegistryUrl;
        if (options.npmRegistry) {
            if (npmRegistryUrl) {
                throw new Error('cannot use the deprecated "npmRegistry" together with "npmRegistryUrl". please use "npmRegistryUrl" instead.');
            }
            npmRegistryUrl = `https://${options.npmRegistry}`;
        }
        const npmr = url_1.parse(npmRegistryUrl ?? DEFAULT_NPM_REGISTRY_URL);
        if (!npmr || !npmr.hostname || !npmr.href) {
            throw new Error(`unable to determine npm registry host from url ${npmRegistryUrl}. Is this really a URL?`);
        }
        const npmAccess = options.npmAccess ?? defaultNpmAccess(this.packageName);
        if (!isScoped(this.packageName) && npmAccess === NpmAccess.RESTRICTED) {
            throw new Error(`"npmAccess" cannot be RESTRICTED for non-scoped npm package "${this.packageName}"`);
        }
        const isAwsCodeArtifact = release_1.isAwsCodeArtifactRegistry(npmRegistryUrl);
        const hasScopedPackage = options.scopedPackagesOptions &&
            options.scopedPackagesOptions.length !== 0;
        if (isAwsCodeArtifact) {
            if (options.npmTokenSecret) {
                throw new Error('"npmTokenSecret" must not be specified when publishing AWS CodeArtifact.');
            }
            else if (options.codeArtifactOptions?.authProvider ===
                CodeArtifactAuthProvider.GITHUB_OIDC) {
                if (options.codeArtifactOptions.accessKeyIdSecret ||
                    options.codeArtifactOptions.secretAccessKeySecret) {
                    throw new Error("access and secret key pair should not be provided when using GITHUB_OIDC auth provider for AWS CodeArtifact");
                }
                else if (!options.codeArtifactOptions.roleToAssume) {
                    throw new Error('"roleToAssume" property is required when using GITHUB_OIDC for AWS CodeArtifact options');
                }
            }
        }
        else {
            if ((options.codeArtifactOptions?.accessKeyIdSecret ||
                options.codeArtifactOptions?.secretAccessKeySecret ||
                options.codeArtifactOptions?.roleToAssume) &&
                !hasScopedPackage) {
                throw new Error("codeArtifactOptions must only be specified when publishing AWS CodeArtifact or used in scoped packages.");
            }
        }
        // apply defaults for AWS CodeArtifact
        let codeArtifactOptions;
        if (isAwsCodeArtifact || hasScopedPackage) {
            const authProvider = options.codeArtifactOptions?.authProvider ??
                CodeArtifactAuthProvider.ACCESS_AND_SECRET_KEY_PAIR;
            const isAccessSecretKeyPairAuth = authProvider === CodeArtifactAuthProvider.ACCESS_AND_SECRET_KEY_PAIR;
            codeArtifactOptions = {
                authProvider,
                accessKeyIdSecret: options.codeArtifactOptions?.accessKeyIdSecret ??
                    (isAccessSecretKeyPairAuth ? "AWS_ACCESS_KEY_ID" : undefined),
                secretAccessKeySecret: options.codeArtifactOptions?.secretAccessKeySecret ??
                    (isAccessSecretKeyPairAuth ? "AWS_SECRET_ACCESS_KEY" : undefined),
                roleToAssume: options.codeArtifactOptions?.roleToAssume,
            };
        }
        return {
            npmAccess,
            npmRegistry: npmr.hostname + this.renderNpmRegistryPath(npmr.pathname),
            npmRegistryUrl: npmr.href,
            npmTokenSecret: defaultNpmToken(options.npmTokenSecret, npmr.hostname),
            codeArtifactOptions,
            scopedPackagesOptions: this.parseScopedPackagesOptions(options.scopedPackagesOptions),
        };
    }
    parseScopedPackagesOptions(scopedPackagesOptions) {
        if (!scopedPackagesOptions) {
            return undefined;
        }
        return scopedPackagesOptions.map((option) => {
            if (!isScoped(option.scope)) {
                throw new Error(`Scope must start with "@" in options, found ${option.scope}`);
            }
            if (!release_1.isAwsCodeArtifactRegistry(option.registryUrl)) {
                throw new Error(`Only AWS Code artifact scoped registry is supported for now, found ${option.registryUrl}`);
            }
            const result = {
                registryUrl: option.registryUrl,
                scope: option.scope,
            };
            return result;
        });
    }
    addCodeArtifactLoginScript() {
        if (!this.scopedPackagesOptions ||
            this.scopedPackagesOptions.length === 0) {
            return;
        }
        this.project.addTask("ca:login", {
            requiredEnv: ["AWS_ACCESS_KEY_ID", "AWS_SECRET_ACCESS_KEY"],
            steps: [
                { exec: "which aws" },
                ...this.scopedPackagesOptions.map((scopedPackagesOption) => {
                    const { registryUrl, scope } = scopedPackagesOption;
                    const { domain, region, accountId, registry } = util_1.extractCodeArtifactDetails(registryUrl);
                    // reference: https://docs.aws.amazon.com/codeartifact/latest/ug/npm-auth.html
                    const commands = [
                        `npm config set ${scope}:registry ${registryUrl}`,
                        `CODEARTIFACT_AUTH_TOKEN=$(aws codeartifact get-authorization-token --domain ${domain} --region ${region} --domain-owner ${accountId} --query authorizationToken --output text)`,
                        `npm config set //${registry}:_authToken=$CODEARTIFACT_AUTH_TOKEN`,
                        `npm config set //${registry}:always-auth=true`,
                    ];
                    return {
                        exec: commands.join("; "),
                    };
                }),
            ],
        });
    }
    addNodeEngine() {
        if (!this.minNodeVersion && !this.maxNodeVersion) {
            return;
        }
        let nodeVersion = "";
        if (this.minNodeVersion) {
            nodeVersion += `>= ${this.minNodeVersion}`;
        }
        if (this.maxNodeVersion) {
            nodeVersion += ` <= ${this.maxNodeVersion}`;
        }
        this.addEngine("node", nodeVersion);
    }
    renderNpmRegistryPath(path) {
        if (!path || path == "/") {
            return "";
        }
        else {
            return path;
        }
    }
    renderInstallCommand(frozen) {
        switch (this.packageManager) {
            case NodePackageManager.YARN:
                return [
                    "yarn install",
                    "--check-files",
                    ...(frozen ? ["--frozen-lockfile"] : []),
                ].join(" ");
            case NodePackageManager.YARN2:
                return ["yarn install", ...(frozen ? ["--immutable"] : [])].join(" ");
            case NodePackageManager.NPM:
                return frozen ? "npm ci" : "npm install";
            case NodePackageManager.PNPM:
                return frozen
                    ? "pnpm i --frozen-lockfile"
                    : "pnpm i --no-frozen-lockfile";
            default:
                throw new Error(`unexpected package manager ${this.packageManager}`);
        }
    }
    processDeps(options) {
        this.addDeps(...(options.deps ?? []));
        this.addDevDeps(...(options.devDeps ?? []));
        this.addPeerDeps(...(options.peerDeps ?? []));
        this.addBundledDeps(...(options.bundledDeps ?? []));
    }
    renderDependencies() {
        const devDependencies = {};
        const peerDependencies = {};
        const dependencies = {};
        const bundledDependencies = new Array();
        // synthetic dependencies: add a pinned build dependency to ensure we are
        // testing against the minimum requirement of the peer.
        if (this.peerDependencyOptions.pinnedDevDependency) {
            for (const dep of this.project.deps.all.filter((d) => d.type === dependencies_1.DependencyType.PEER)) {
                let req = dep.name;
                // skip if we already have a runtime dependency on this peer
                if (this.project.deps.tryGetDependency(dep.name, dependencies_1.DependencyType.RUNTIME)) {
                    continue;
                }
                if (dep.version) {
                    const ver = util_1.minVersion(dep.version);
                    if (!ver) {
                        throw new Error(`unable to determine minimum semver for peer dependency ${dep.name}@${dep.version}`);
                    }
                    req += "@" + ver;
                }
                this.addDevDeps(req);
            }
        }
        for (const dep of this.project.deps.all) {
            let version = dep.version ?? "*";
            let name = dep.name;
            if (name.startsWith("file:")) {
                const localDependencyPath = name.substring(5);
                const depPackageJson = path_1.resolve(this.project.outdir, localDependencyPath, "package.json");
                const pkgFile = fs_1.readFileSync(depPackageJson, "utf8");
                const pkg = JSON.parse(pkgFile);
                version = localDependencyPath;
                name = pkg.name;
            }
            switch (dep.type) {
                case dependencies_1.DependencyType.BUNDLED:
                    bundledDependencies.push(name);
                    if (this.project.deps.all.find((d) => d.name === name && d.type === dependencies_1.DependencyType.PEER)) {
                        throw new Error(`unable to bundle "${name}". it cannot appear as a peer dependency`);
                    }
                    // also add as a runtime dependency
                    dependencies[name] = version;
                    break;
                case dependencies_1.DependencyType.PEER:
                    peerDependencies[name] = version;
                    break;
                case dependencies_1.DependencyType.RUNTIME:
                    dependencies[name] = version;
                    break;
                case dependencies_1.DependencyType.TEST:
                case dependencies_1.DependencyType.DEVENV:
                case dependencies_1.DependencyType.BUILD:
                    devDependencies[name] = version;
                    break;
            }
        }
        // returns a lazy value to normalize during synthesis
        const normalize = (obj) => () => util_2.sorted(obj);
        // update the manifest we are about to save into `package.json`
        this.manifest.devDependencies = normalize(devDependencies);
        this.manifest.peerDependencies = normalize(peerDependencies);
        this.manifest.dependencies = normalize(dependencies);
        this.manifest.bundledDependencies = util_2.sorted(bundledDependencies);
        // nothing further to do if package.json file does not exist
        const pkg = this.readPackageJson();
        if (!pkg) {
            return { devDependencies, peerDependencies, dependencies };
        }
        const readDeps = (user, current = {}) => {
            for (const [name, userVersion] of Object.entries(user)) {
                const currentVersion = current[name];
                // respect user version if it's not '*' or if current version is undefined
                if (userVersion !== "*" || !currentVersion || currentVersion === "*") {
                    continue;
                }
                // memoize current version in memory so it is preserved when saving
                user[name] = currentVersion;
            }
            // report removals
            for (const name of Object.keys(current ?? {})) {
                if (!user[name]) {
                    this.project.logger.verbose(`${name}: removed`);
                }
            }
        };
        readDeps(devDependencies, pkg.devDependencies);
        readDeps(dependencies, pkg.dependencies);
        readDeps(peerDependencies, pkg.peerDependencies);
        return { devDependencies, dependencies, peerDependencies };
    }
    /**
     * Resolves any deps that do not have a specified version (e.g. `*`) and
     * update `package.json` if needed.
     *
     * @returns `true` if package.json was updated or `false` if not.
     */
    resolveDepsAndWritePackageJson() {
        const outdir = this.project.outdir;
        const rootPackageJson = path_1.join(outdir, "package.json");
        const original = fs_1.readFileSync(rootPackageJson, "utf8");
        const pkg = JSON.parse(original);
        const resolveDeps = (current, user) => {
            const result = {};
            current = current ?? {};
            user = user ?? {};
            for (const [name, currentDefinition] of Object.entries(user)) {
                // find actual version from node_modules
                let desiredVersion = currentDefinition;
                if (currentDefinition === "*") {
                    try {
                        const modulePath = require.resolve(`${name}/package.json`, {
                            paths: [outdir],
                        });
                        const module = JSON.parse(fs_1.readFileSync(modulePath, "utf-8"));
                        desiredVersion = `^${module.version}`;
                    }
                    catch (e) { }
                    if (!desiredVersion) {
                        this.project.logger.warn(`unable to resolve version for ${name} from installed modules`);
                        continue;
                    }
                }
                if (currentDefinition !== desiredVersion) {
                    this.project.logger.verbose(`${name}: ${currentDefinition} => ${desiredVersion}`);
                }
                result[name] = desiredVersion;
            }
            // print removed packages
            for (const name of Object.keys(current)) {
                if (!result[name]) {
                    this.project.logger.verbose(`${name} removed`);
                }
            }
            return result;
        };
        const rendered = this._renderedDeps;
        if (!rendered) {
            throw new Error("assertion failed");
        }
        const deps = resolveDeps(pkg.dependencies, rendered.dependencies);
        const devDeps = resolveDeps(pkg.devDependencies, rendered.devDependencies);
        const peerDeps = resolveDeps(pkg.peerDependencies, rendered.peerDependencies);
        if (this.peerDependencyOptions.pinnedDevDependency) {
            for (const [name, version] of Object.entries(peerDeps)) {
                // Skip if we already have a runtime dependency on this peer
                // or if devDependency version is already set.
                // Relies on the "*" devDependency added in the presynth step
                if (deps[name] || rendered.devDependencies[name] !== "*") {
                    continue;
                }
                // Take version and pin as dev dependency
                const ver = util_1.minVersion(version);
                if (!ver) {
                    throw new Error(`unable to determine minimum semver for peer dependency ${name}@${version}`);
                }
                devDeps[name] = ver;
            }
        }
        pkg.dependencies = util_2.sorted(deps);
        pkg.devDependencies = util_2.sorted(devDeps);
        pkg.peerDependencies = util_2.sorted(peerDeps);
        const updated = JSON.stringify(pkg, undefined, 2) + "\n";
        if (original === updated) {
            return false;
        }
        util_2.writeFile(rootPackageJson, updated);
        return true;
    }
    renderPublishConfig() {
        // omit values if they are the same as the npm defaults
        return _resolve_1.resolve({
            registry: this.npmRegistryUrl !== DEFAULT_NPM_REGISTRY_URL
                ? this.npmRegistryUrl
                : undefined,
            access: this.npmAccess !== defaultNpmAccess(this.packageName)
                ? this.npmAccess
                : undefined,
        }, { omitEmpty: true });
    }
    renderKeywords() {
        const kwds = Array.from(this.keywords);
        return util_2.sorted(kwds.sort());
    }
    renderEngines() {
        return util_2.sorted(this.engines);
    }
    autoDiscoverBinaries() {
        const binrel = "bin";
        const bindir = path_1.join(this.project.outdir, binrel);
        if (fs_1.existsSync(bindir)) {
            for (const file of fs_1.readdirSync(bindir)) {
                try {
                    fs_1.accessSync(path_1.join(bindir, file), fs_1.constants.X_OK);
                    this.bin[file] = path_1.join(binrel, file).replace(/\\/g, "/");
                }
                catch (e) {
                    // not executable, skip
                }
            }
        }
    }
    renderAuthor(options) {
        let author;
        if (options.authorName) {
            author = {
                name: options.authorName,
                email: options.authorEmail,
                url: options.authorUrl,
                organization: options.authorOrganization ?? false,
            };
        }
        else {
            if (options.authorEmail ||
                options.authorUrl ||
                options.authorOrganization !== undefined) {
                throw new Error('"authorName" is required if specifying "authorEmail" or "authorUrl"');
            }
        }
        return author;
    }
    renderBin() {
        return util_2.sorted(this.bin);
    }
    renderScripts() {
        const result = {};
        const tasks = this.project.tasks.all
            .filter((t) => 
        // Must remove to prevent overriding built-in npm command (which would loop)
        t.name !== this.installTask.name && t.name !== this.installCiTask.name)
            .sort((x, y) => x.name.localeCompare(y.name));
        for (const task of tasks) {
            result[task.name] = this.npmScriptForTask(task);
        }
        return result;
    }
    npmScriptForTask(task) {
        return `${this.projenCommand} ${task.name}`;
    }
    readPackageJson() {
        const file = path_1.join(this.project.outdir, "package.json");
        if (!fs_1.existsSync(file)) {
            return undefined;
        }
        return JSON.parse(fs_1.readFileSync(file, "utf-8"));
    }
    installDependencies() {
        this.project.logger.info("Installing dependencies...");
        const runtime = new task_runtime_1.TaskRuntime(this.project.outdir);
        const taskToRun = this.isAutomatedBuild
            ? this.installCiTask
            : this.installTask;
        runtime.runTask(taskToRun.name);
    }
}
exports.NodePackage = NodePackage;
_a = JSII_RTTI_SYMBOL_1;
NodePackage[_a] = { fqn: "projen.javascript.NodePackage", version: "0.69.3" };
/**
 * The node package manager to use.
 */
var NodePackageManager;
(function (NodePackageManager) {
    /**
     * Use `yarn` as the package manager.
     */
    NodePackageManager["YARN"] = "yarn";
    /**
     * Use `yarn` versions >= 2 as the package manager.
     */
    NodePackageManager["YARN2"] = "yarn2";
    /**
     * Use `npm` as the package manager.
     */
    NodePackageManager["NPM"] = "npm";
    /**
     * Use `pnpm` as the package manager.
     */
    NodePackageManager["PNPM"] = "pnpm";
})(NodePackageManager = exports.NodePackageManager || (exports.NodePackageManager = {}));
/**
 * Npm package access level
 */
var NpmAccess;
(function (NpmAccess) {
    /**
     * Package is public.
     */
    NpmAccess["PUBLIC"] = "public";
    /**
     * Package can only be accessed with credentials.
     */
    NpmAccess["RESTRICTED"] = "restricted";
})(NpmAccess = exports.NpmAccess || (exports.NpmAccess = {}));
/**
 * Determines if an npm package is "scoped" (i.e. it starts with "xxx@").
 */
function isScoped(packageName) {
    return packageName.includes("@");
}
function defaultNpmAccess(packageName) {
    return isScoped(packageName) ? NpmAccess.RESTRICTED : NpmAccess.PUBLIC;
}
function defaultNpmToken(npmToken, registry) {
    // if we are publishing to AWS CdodeArtifact, no NPM_TOKEN used (will be requested using AWS CLI later).
    if (release_1.isAwsCodeArtifactRegistry(registry)) {
        return undefined;
    }
    // if we are publishing to GitHub Packages, default to GITHUB_TOKEN.
    const isGitHubPackages = registry === GITHUB_PACKAGES_REGISTRY;
    return (npmToken ??
        (isGitHubPackages ? DEFAULT_GITHUB_TOKEN_SECRET : DEFAULT_NPM_TOKEN_SECRET));
}
exports.defaultNpmToken = defaultNpmToken;
function determineLockfile(packageManager) {
    if (packageManager === NodePackageManager.YARN ||
        packageManager === NodePackageManager.YARN2) {
        return "yarn.lock";
    }
    else if (packageManager === NodePackageManager.NPM) {
        return "package-lock.json";
    }
    else if (packageManager === NodePackageManager.PNPM) {
        return "pnpm-lock.yaml";
    }
    throw new Error(`unsupported package manager ${packageManager}`);
}
//# sourceMappingURL=data:application/json;base64,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