"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Jest = exports.UpdateSnapshot = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const semver = require("semver");
const component_1 = require("../component");
const javascript_1 = require("../javascript");
const json_1 = require("../json");
const DEFAULT_TEST_REPORTS_DIR = "test-reports";
var UpdateSnapshot;
(function (UpdateSnapshot) {
    /**
     * Always update snapshots in "test" task.
     */
    UpdateSnapshot["ALWAYS"] = "always";
    /**
     * Never update snapshots in "test" task and create a separate "test:update" task.
     */
    UpdateSnapshot["NEVER"] = "never";
})(UpdateSnapshot = exports.UpdateSnapshot || (exports.UpdateSnapshot = {}));
/**
 * Installs the following npm scripts:
 *
 * - `test`, intended for testing locally and in CI. Will update snapshots unless `updateSnapshot: UpdateSnapshot: NEVER` is set.
 * - `test:watch`, intended for automatically rerunning tests when files change.
 * - `test:update`, intended for testing locally and updating snapshots to match the latest unit under test. Only available when `updateSnapshot: UpdateSnapshot: NEVER`.
 *
 */
class Jest extends component_1.Component {
    constructor(project, options = {}) {
        super(project);
        // hard deprecation
        if (options.typescriptConfig) {
            throw new Error('"jestOptions.typescriptConfig" is deprecated. Use "typescriptProject.tsconfigDev" instead');
        }
        // Jest snapshot files are generated files!
        project.root.annotateGenerated("*.snap");
        this.jestVersion = options.jestVersion ? `@${options.jestVersion}` : "";
        project.addDevDeps(`jest${this.jestVersion}`);
        this.jestConfig = options.jestConfig;
        this.extraCliOptions = options.extraCliOptions ?? [];
        this.ignorePatterns = this.jestConfig?.testPathIgnorePatterns ??
            options.ignorePatterns ?? ["/node_modules/"];
        this.watchIgnorePatterns = this.jestConfig?.watchPathIgnorePatterns ?? [
            "/node_modules/",
        ];
        this.coverageReporters = this.jestConfig?.coverageReporters ?? [
            "json",
            "lcov",
            "clover",
            "cobertura",
        ];
        this.testMatch = this.jestConfig?.testMatch ?? [
            "**/__tests__/**/*.[jt]s?(x)",
            "**/?(*.)+(spec|test).[tj]s?(x)",
        ];
        const coverageDirectory = this.jestConfig?.coverageDirectory ?? "coverage";
        this.reporters = [];
        if (options.preserveDefaultReporters ?? true) {
            this.reporters.unshift("default");
        }
        this.config = {
            ...this.jestConfig,
            clearMocks: this.jestConfig?.clearMocks ?? true,
            collectCoverage: options.coverage ?? this.jestConfig?.collectCoverage ?? true,
            coverageReporters: this.coverageReporters,
            coverageDirectory: coverageDirectory,
            coveragePathIgnorePatterns: this.jestConfig?.coveragePathIgnorePatterns ?? this.ignorePatterns,
            testPathIgnorePatterns: this.ignorePatterns,
            watchPathIgnorePatterns: this.watchIgnorePatterns,
            testMatch: this.testMatch,
            reporters: this.reporters,
            snapshotResolver: (() => this._snapshotResolver),
        };
        if (options.junitReporting ?? true) {
            const reportsDir = DEFAULT_TEST_REPORTS_DIR;
            this.addReporter(["jest-junit", { outputDirectory: reportsDir }]);
            project.addDevDeps("jest-junit@^15");
            project.gitignore.exclude("# jest-junit artifacts", `/${reportsDir}/`, "junit.xml");
            project.npmignore?.exclude("# jest-junit artifacts", `/${reportsDir}/`, "junit.xml");
        }
        if (this.jestConfig?.reporters) {
            for (const reporter of this.jestConfig.reporters) {
                this.addReporter(reporter);
            }
        }
        if (this.jestConfig?.coverageThreshold) {
            this.config.coverageThreshold = {
                global: this.jestConfig?.coverageThreshold,
            };
        }
        this.configureTestCommand(options.updateSnapshot ?? UpdateSnapshot.ALWAYS);
        if (options.configFilePath) {
            this.file = new json_1.JsonFile(project, options.configFilePath, {
                obj: this.config,
            });
            project.npmignore?.addPatterns(`/${this.file.path}`);
        }
        else {
            project.addFields({ jest: this.config });
        }
        const coverageDirectoryPath = path.posix.join("/", coverageDirectory, "/");
        project.npmignore?.exclude(coverageDirectoryPath);
        project.gitignore.exclude(coverageDirectoryPath);
        if (options.coverageText ?? true) {
            this.coverageReporters.push("text");
        }
    }
    /**
     * Returns the singletone Jest component of a project or undefined if there is none.
     */
    static of(project) {
        const isJest = (c) => c instanceof Jest;
        return project.components.find(isJest);
    }
    /**
     * Adds a test match pattern.
     * @param pattern glob pattern to match for tests
     */
    addTestMatch(pattern) {
        this.testMatch.push(pattern);
    }
    /**
     * Adds a watch ignore pattern.
     * @param pattern The pattern (regular expression).
     */
    addWatchIgnorePattern(pattern) {
        this.watchIgnorePatterns.push(pattern);
    }
    addIgnorePattern(pattern) {
        this.ignorePatterns.push(pattern);
    }
    addReporter(reporter) {
        this.reporters.push(reporter);
    }
    /**
     * Adds a a setup file to Jest's setupFiles configuration.
     * @param file File path to setup file
     */
    addSetupFile(file) {
        if (!this.config.setupFiles) {
            this.config.setupFiles = [];
        }
        this.config.setupFiles.push(file);
    }
    /**
     * Adds a a setup file to Jest's setupFilesAfterEnv configuration.
     * @param file File path to setup file
     */
    addSetupFileAfterEnv(file) {
        if (!this.config.setupFilesAfterEnv) {
            this.config.setupFilesAfterEnv = [];
        }
        this.config.setupFilesAfterEnv.push(file);
    }
    addSnapshotResolver(file) {
        this._snapshotResolver = file;
    }
    configureTestCommand(updateSnapshot) {
        const jestOpts = ["--passWithNoTests", ...this.extraCliOptions];
        const jestConfigOpts = this.file && this.file.path != "jest.config.json"
            ? ` -c ${this.file.path}`
            : "";
        // as recommended in the jest docs, node > 14 may use native v8 coverage collection
        // https://jestjs.io/docs/en/cli#--coverageproviderprovider
        if (this.project instanceof javascript_1.NodeProject &&
            this.project.package.minNodeVersion &&
            semver.gte(this.project.package.minNodeVersion, "14.0.0")) {
            jestOpts.push("--coverageProvider=v8");
        }
        if (updateSnapshot === UpdateSnapshot.ALWAYS) {
            jestOpts.push("--updateSnapshot");
        }
        else {
            jestOpts.push("--ci"); // to prevent accepting new snapshots
            const testUpdate = this.project.tasks.tryFind("test:update");
            if (!testUpdate) {
                this.project.addTask("test:update", {
                    description: "Update jest snapshots",
                    exec: `jest --updateSnapshot ${jestOpts.join(" ")}${jestConfigOpts}`,
                    receiveArgs: true,
                });
            }
        }
        this.project.testTask.exec(`jest ${jestOpts.join(" ")}${jestConfigOpts}`, {
            receiveArgs: true,
        });
        const testWatch = this.project.tasks.tryFind("test:watch");
        if (!testWatch) {
            this.project.addTask("test:watch", {
                description: "Run jest in watch mode",
                exec: `jest --watch${jestConfigOpts}`,
            });
        }
    }
}
exports.Jest = Jest;
_a = JSII_RTTI_SYMBOL_1;
Jest[_a] = { fqn: "projen.javascript.Jest", version: "0.69.3" };
//# sourceMappingURL=data:application/json;base64,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