"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Eslint = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const prettier_1 = require("./prettier");
const common_1 = require("../common");
const component_1 = require("../component");
const json_1 = require("../json");
const yaml_1 = require("../yaml");
/**
 * Represents eslint configuration.
 */
class Eslint extends component_1.Component {
    constructor(project, options) {
        super(project);
        /**
         * eslint overrides.
         */
        this.overrides = [];
        this._plugins = new Array();
        this._extends = new Array();
        this.nodeProject = project;
        project.addDevDeps("eslint@^8", "@typescript-eslint/eslint-plugin@^5", "@typescript-eslint/parser@^5", "eslint-import-resolver-node", "eslint-import-resolver-typescript", "eslint-plugin-import");
        if (options.aliasMap) {
            project.addDevDeps("eslint-import-resolver-alias");
        }
        const lintProjenRc = options.lintProjenRc ?? true;
        const lintProjenRcFile = options.lintProjenRcFile ?? common_1.PROJEN_RC;
        const devdirs = options.devdirs ?? [];
        this._lintPatterns = [
            ...options.dirs,
            ...devdirs,
            ...(lintProjenRc && lintProjenRcFile ? [lintProjenRcFile] : []),
        ];
        this._fileExtensions = options.fileExtensions ?? [".ts"];
        this._allowDevDeps = new Set((devdirs ?? []).map((dir) => `**/${dir}/**`));
        this.eslintTask = project.addTask("eslint", {
            description: "Runs eslint against the codebase",
        });
        this.updateTask();
        project.testTask.spawn(this.eslintTask);
        // exclude some files
        project.npmignore?.exclude("/.eslintrc.json");
        this._formattingRules = {
            // see https://github.com/typescript-eslint/typescript-eslint/blob/master/packages/eslint-plugin/docs/rules/indent.md
            indent: ["off"],
            "@typescript-eslint/indent": ["error", 2],
            // Style
            quotes: ["error", "single", { avoidEscape: true }],
            "comma-dangle": ["error", "always-multiline"],
            "comma-spacing": ["error", { before: false, after: true }],
            "no-multi-spaces": ["error", { ignoreEOLComments: false }],
            "array-bracket-spacing": ["error", "never"],
            "array-bracket-newline": ["error", "consistent"],
            "object-curly-spacing": ["error", "always"],
            "object-curly-newline": ["error", { multiline: true, consistent: true }],
            "object-property-newline": [
                "error",
                { allowAllPropertiesOnSameLine: true },
            ],
            "keyword-spacing": ["error"],
            "brace-style": ["error", "1tbs", { allowSingleLine: true }],
            "space-before-blocks": ["error"],
            curly: ["error", "multi-line", "consistent"],
            "@typescript-eslint/member-delimiter-style": ["error"],
            // Require semicolons
            semi: ["error", "always"],
            // Max line lengths
            "max-len": [
                "error",
                {
                    code: 150,
                    ignoreUrls: true,
                    ignoreStrings: true,
                    ignoreTemplateLiterals: true,
                    ignoreComments: true,
                    ignoreRegExpLiterals: true,
                },
            ],
            // Don't unnecessarily quote properties
            "quote-props": ["error", "consistent-as-needed"],
        };
        this.rules = {
            // Require use of the `import { foo } from 'bar';` form instead of `import foo = require('bar');`
            "@typescript-eslint/no-require-imports": ["error"],
            // Require all imported dependencies are actually declared in package.json
            "import/no-extraneous-dependencies": [
                "error",
                {
                    // Only allow importing devDependencies from "devdirs".
                    devDependencies: () => this.renderDevDepsAllowList(),
                    optionalDependencies: false,
                    peerDependencies: true,
                },
            ],
            // Require all imported libraries actually resolve (!!required for import/no-extraneous-dependencies to work!!)
            "import/no-unresolved": ["error"],
            // Require an ordering on all imports
            "import/order": [
                "warn",
                {
                    groups: ["builtin", "external"],
                    alphabetize: { order: "asc", caseInsensitive: true },
                },
            ],
            // Cannot import from the same module twice
            "no-duplicate-imports": ["error"],
            // Cannot shadow names
            "no-shadow": ["off"],
            "@typescript-eslint/no-shadow": ["error"],
            // Required spacing in property declarations (copied from TSLint, defaults are good)
            "key-spacing": ["error"],
            // No multiple empty lines
            "no-multiple-empty-lines": ["error"],
            // One of the easiest mistakes to make
            "@typescript-eslint/no-floating-promises": ["error"],
            // Make sure that inside try/catch blocks, promises are 'return await'ed
            // (must disable the base rule as it can report incorrect errors)
            "no-return-await": ["off"],
            "@typescript-eslint/return-await": ["error"],
            // Useless diff results
            "no-trailing-spaces": ["error"],
            // Must use foo.bar instead of foo['bar'] if possible
            "dot-notation": ["error"],
            // Are you sure | is not a typo for || ?
            "no-bitwise": ["error"],
            // Member ordering
            "@typescript-eslint/member-ordering": [
                "error",
                {
                    default: [
                        "public-static-field",
                        "public-static-method",
                        "protected-static-field",
                        "protected-static-method",
                        "private-static-field",
                        "private-static-method",
                        "field",
                        // Constructors
                        "constructor",
                        // Methods
                        "method",
                    ],
                },
            ],
        };
        // Overrides for .projenrc.js
        // @deprecated
        if (lintProjenRc) {
            this.overrides = [
                {
                    files: [lintProjenRcFile || common_1.PROJEN_RC],
                    rules: {
                        "@typescript-eslint/no-require-imports": "off",
                        "import/no-extraneous-dependencies": "off",
                    },
                },
            ];
        }
        this.ignorePatterns = options.ignorePatterns ?? [
            "*.js",
            // @deprecated
            ...(lintProjenRc ? [`!${lintProjenRcFile || common_1.PROJEN_RC}`] : []),
            "*.d.ts",
            "node_modules/",
            "*.generated.ts",
            "coverage",
        ];
        const tsconfig = options.tsconfigPath ?? "./tsconfig.json";
        this.addPlugins("@typescript-eslint");
        this.addPlugins("import");
        this.addExtends("plugin:import/typescript");
        this.config = {
            env: {
                jest: true,
                node: true,
            },
            root: true,
            plugins: () => this._plugins,
            parser: "@typescript-eslint/parser",
            parserOptions: {
                ecmaVersion: 2018,
                sourceType: "module",
                project: tsconfig,
            },
            extends: () => this._extends,
            settings: {
                "import/parsers": {
                    "@typescript-eslint/parser": [".ts", ".tsx"],
                },
                "import/resolver": {
                    ...(options.aliasMap && {
                        alias: {
                            map: Object.entries(options.aliasMap).map(([k, v]) => [k, v]),
                            extensions: options.aliasExtensions,
                        },
                    }),
                    node: {},
                    typescript: {
                        project: tsconfig,
                        ...(options.tsAlwaysTryTypes !== false && { alwaysTryTypes: true }),
                    },
                },
            },
            ignorePatterns: this.ignorePatterns,
            rules: () => ({ ...this._formattingRules, ...this.rules }),
            overrides: this.overrides,
        };
        if (options.yaml) {
            new yaml_1.YamlFile(project, ".eslintrc.yml", {
                obj: this.config,
                marker: true,
            });
        }
        else {
            new json_1.JsonFile(project, ".eslintrc.json", {
                obj: this.config,
                // https://eslint.org/docs/latest/user-guide/configuring/configuration-files#comments-in-configuration-files
                marker: true,
                allowComments: true,
            });
        }
        // if the user enabled prettier explicitly _or_ if the project has a
        // `Prettier` component, we shall tweak our configuration accordingly.
        if (options.prettier || prettier_1.Prettier.of(project)) {
            this.enablePrettier();
        }
    }
    /**
     * Returns the singletone Eslint component of a project or undefined if there is none.
     */
    static of(project) {
        const isEslint = (c) => c instanceof Eslint;
        return project.components.find(isEslint);
    }
    /**
     * Add a file, glob pattern or directory with source files to lint (e.g. [ "src" ])
     */
    addLintPattern(pattern) {
        this._lintPatterns.push(pattern);
        this.updateTask();
    }
    /**
     * Add an eslint rule.
     */
    addRules(rules) {
        for (const [k, v] of Object.entries(rules)) {
            this.rules[k] = v;
        }
    }
    /**
     * Adds an eslint plugin
     * @param plugins The names of plugins to add
     */
    addPlugins(...plugins) {
        this._plugins.push(...plugins);
    }
    /**
     * Add an eslint override.
     */
    addOverride(override) {
        this.overrides.push(override);
    }
    /**
     * Do not lint these files.
     */
    addIgnorePattern(pattern) {
        this.ignorePatterns.push(pattern);
    }
    /**
     * Adds an `extends` item to the eslint configuration.
     * @param extendList The list of "extends" to add.
     */
    addExtends(...extendList) {
        this._extends.push(...extendList);
    }
    /**
     * Add a glob file pattern which allows importing dev dependencies.
     * @param pattern glob pattern.
     */
    allowDevDeps(pattern) {
        this._allowDevDeps.add(pattern);
    }
    /**
     * Enables prettier for code formatting.
     */
    enablePrettier() {
        this.nodeProject.addDevDeps("prettier", "eslint-plugin-prettier", "eslint-config-prettier");
        this.addPlugins("prettier");
        this._formattingRules = {
            "prettier/prettier": ["error"],
        };
        this.addExtends("prettier", "plugin:prettier/recommended");
    }
    renderDevDepsAllowList() {
        return Array.from(this._allowDevDeps);
    }
    /**
     * Update the task with the current list of lint patterns and file extensions
     */
    updateTask() {
        this.eslintTask.reset([
            "eslint",
            `--ext ${this._fileExtensions.join(",")}`,
            "--fix",
            "--no-error-on-unmatched-pattern",
            ...this._lintPatterns,
        ].join(" "));
    }
}
exports.Eslint = Eslint;
_a = JSII_RTTI_SYMBOL_1;
Eslint[_a] = { fqn: "projen.javascript.Eslint", version: "0.69.3" };
//# sourceMappingURL=data:application/json;base64,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