"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.getJavaImport = exports.generateJavaOptionNames = exports.Projenrc = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs_1 = require("fs");
const path_1 = require("path");
const common_1 = require("../common");
const dependencies_1 = require("../dependencies");
const inventory_1 = require("../inventory");
const projenrc_1 = require("../projenrc");
/**
 * Allows writing projenrc files in java.
 *
 * This will install `org.projen/projen` as a Maven dependency and will add a
 * `synth` task which will compile & execute `main()` from
 * `src/main/java/projenrc.java`.
 */
class Projenrc extends projenrc_1.ProjenrcFile {
    constructor(project, pom, options = {}) {
        super(project);
        const projenVersion = options.projenVersion ?? common_1.PROJEN_VERSION;
        this.className = options.className ?? "projenrc";
        this.testScope = options.testScope ?? true;
        const depType = this.testScope
            ? dependencies_1.DependencyType.TEST
            : dependencies_1.DependencyType.RUNTIME;
        const execOpts = this.testScope ? ' -Dexec.classpathScope="test"' : "";
        const compileGoal = this.testScope
            ? "compiler:testCompile"
            : "compiler:compile";
        project.deps.addDependency(`io.github.cdklabs/projen@${projenVersion}`, depType);
        pom.addPlugin("org.codehaus.mojo/exec-maven-plugin@3.0.0");
        // set up the "default" task which is the task executed when `projen` is executed for this project.
        project.defaultTask?.exec(`mvn ${compileGoal} --quiet`);
        project.defaultTask?.exec(`mvn exec:java --quiet -Dexec.mainClass=${this.className}${execOpts}`);
        // if this is a new project, generate a skeleton for projenrc.java
        this.generateProjenrc();
    }
    get javaClass() {
        const split = this.className.split(".");
        if (split.length === 1) {
            return split[0];
        }
        return split[split.length - 1];
    }
    get javaPackage() {
        const split = this.className.split(".");
        if (split.length === 1) {
            return [];
        }
        return split.slice(0, split.length - 2);
    }
    /**
     * The path of the projenrc file.
     */
    get filePath() {
        const dir = this.testScope ? "src/test/java" : "src/main/java";
        const javaFile = path_1.join(this.project.outdir, dir, ...this.javaPackage, this.javaClass + ".java");
        return path_1.relative(this.project.outdir, javaFile);
    }
    generateProjenrc() {
        const bootstrap = this.project.initProject;
        if (!bootstrap) {
            return;
        }
        const jsiiFqn = bootstrap.fqn;
        const jsiiManifest = inventory_1.readJsiiManifest(jsiiFqn);
        const jsiiType = jsiiManifest.types[jsiiFqn];
        const javaTarget = jsiiManifest.targets.java;
        const optionsTypeFqn = jsiiType.initializer?.parameters?.[0].type?.fqn;
        if (!optionsTypeFqn) {
            this.project.logger.warn("cannot determine jsii type for project options");
            return;
        }
        const jsiiOptionsType = jsiiManifest.types[optionsTypeFqn];
        if (!jsiiOptionsType) {
            this.project.logger.warn(`cannot find jsii type for project options: ${optionsTypeFqn}`);
            return;
        }
        // skip if file exists
        if (fs_1.existsSync(this.filePath)) {
            return;
        }
        const lines = new Array();
        let indent = 0;
        const emit = (line = "") => lines.push(" ".repeat(indent * 4) + line);
        const openBlock = (line = "") => {
            emit(line + " {");
            indent++;
        };
        const closeBlock = () => {
            indent--;
            emit("}");
        };
        const optionFqns = generateJavaOptionNames(bootstrap.type.options, jsiiManifest);
        if (this.javaPackage.length > 0) {
            emit(`package ${this.javaPackage.join(".")};`);
            emit();
        }
        const { renderedOptions, imports } = renderJavaOptions(2, jsiiOptionsType.name, optionFqns, bootstrap.args);
        emit(`import ${getJavaImport(jsiiType, jsiiManifest)};`);
        emit(`import ${getJavaImport(jsiiOptionsType, jsiiManifest)};`);
        for (const optionTypeName of imports) {
            emit(`import ${javaTarget.package}.${optionTypeName};`);
        }
        emit();
        openBlock(`public class ${this.javaClass}`);
        openBlock("public static void main(String[] args)");
        emit(`${jsiiType.name} project = new ${jsiiType.name}(${renderedOptions});`);
        emit("project.synth();");
        closeBlock();
        closeBlock();
        const filePath = path_1.join(this.project.outdir, this.filePath);
        fs_1.mkdirSync(path_1.dirname(filePath), { recursive: true });
        fs_1.writeFileSync(filePath, lines.join("\n"));
        this.project.logger.info(`Project definition file was created at ${this.filePath}`);
    }
}
exports.Projenrc = Projenrc;
_a = JSII_RTTI_SYMBOL_1;
Projenrc[_a] = { fqn: "projen.java.Projenrc", version: "0.69.3" };
function generateJavaOptionNames(options, jsiiManifest) {
    const optionFqns = {};
    for (const option of options) {
        if (option.fqn && jsiiManifest.types[option.fqn]) {
            optionFqns[option.name] = toJavaFullTypeName(jsiiManifest.types[option.fqn]);
        }
    }
    return optionFqns;
}
exports.generateJavaOptionNames = generateJavaOptionNames;
function renderJavaOptions(indent, optionsTypeName, optionFqns, initOptions) {
    const imports = new Set();
    if (!initOptions || Object.keys(initOptions).length === 0) {
        return { renderedOptions: "", imports }; // no options
    }
    const lines = [`${optionsTypeName}.builder()`];
    for (const [name, value] of Object.entries(initOptions)) {
        const { javaValue, importName } = toJavaValue(value, name, optionFqns);
        if (importName)
            imports.add(importName);
        lines.push(`.${toJavaProperty(name)}(${javaValue})`);
    }
    lines.push(".build()");
    const renderedOptions = lines.join(`\n${" ".repeat((indent + 1) * 4)}`);
    return { renderedOptions, imports };
}
function toJavaProperty(prop) {
    return prop;
}
function toJavaValue(value, name, optionFqns) {
    if (typeof value === "string" && optionFqns[name] !== undefined) {
        const parts = optionFqns[name].split(".");
        const base = parts[parts.length - 1];
        const choice = String(value).toUpperCase().replace(/-/g, "_");
        return { javaValue: `${base}.${choice}`, importName: optionFqns[name] };
    }
    else {
        return { javaValue: JSON.stringify(value) };
    }
}
function toJavaFullTypeName(jsiiType) {
    return [jsiiType.namespace, jsiiType.name].filter((x) => x).join(".");
}
function getJavaImport(jsiiType, jsiiManifest) {
    const packageName = jsiiManifest?.submodules?.[`${jsiiType.assembly}.${jsiiType?.namespace}`]
        ?.targets?.java?.package ||
        [jsiiManifest.targets.java.package, jsiiType.namespace]
            .filter((x) => x)
            .join(".");
    return `${packageName}.${jsiiType.name}`;
}
exports.getJavaImport = getJavaImport;
//# sourceMappingURL=data:application/json;base64,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