"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.readJsiiManifest = exports.resolveProjectType = exports.readManifest = exports.discover = void 0;
const fs = require("fs");
const path = require("path");
const zlib_1 = require("zlib");
const case_1 = require("case");
const PROJEN_MODULE_ROOT = path.join(__dirname, "..");
const PROJECT_BASE_FQN = "projen.Project";
/**
 * Returns a list of project types exported the modules defined in `moduleDirs`.
 * This list will always also include the built-in projen project types.
 * Modules without a .jsii manifest are skipped.
 *
 * @param moduleDirs A list of npm module directories
 */
function discover(...moduleDirs) {
    const jsii = discoverJsiiTypes(...moduleDirs);
    const result = new Array();
    for (const fqn of Object.keys(jsii)) {
        if (isProjectType(jsii, fqn)) {
            const p = toProjectType(jsii, fqn);
            result.push(p);
        }
    }
    return result.sort((r1, r2) => r1.pjid.localeCompare(r2.pjid));
}
exports.discover = discover;
function readManifest(dir) {
    const jsiiFile = path.join(dir, ".jsii");
    if (!fs.existsSync(jsiiFile)) {
        return undefined;
    } // no jsii manifest
    let manifest = JSON.parse(fs.readFileSync(jsiiFile, "utf-8"));
    if (manifest.schema === "jsii/file-redirect") {
        const compressedFile = path.join(dir, manifest.filename);
        if (!fs.existsSync(compressedFile)) {
            throw new Error(`${compressedFile} does not exist.`);
        }
        switch (manifest.compression) {
            case "gzip":
                manifest = JSON.parse(zlib_1.unzipSync(fs.readFileSync(compressedFile)).toString());
                break;
            default:
                throw new Error(`Unsupported compression format: ${manifest.compression}`);
        }
    }
    return manifest;
}
exports.readManifest = readManifest;
/**
 * Resolve all jsii types from @modulesDirs.
 * When a jsii module is found it will recusively list the types from the dependant module as well
 *
 * @param moduleDirs
 * @returns
 */
function discoverJsiiTypes(...moduleDirs) {
    const jsii = {};
    const discoveredManifests = [];
    const discoverJsii = (dir) => {
        const manifest = readManifest(dir);
        if (!manifest) {
            return;
        }
        if (discoveredManifests.includes(manifest.fingerprint)) {
            return;
        }
        discoveredManifests.push(manifest.fingerprint);
        for (const [fqn, type] of Object.entries(manifest.types)) {
            jsii[fqn] = {
                ...type,
            };
        }
        // Also search recursively in nested project dependencies. If the requested module is an external module
        // this will also end-up in the projen module and add the projen types
        if (manifest.dependencies) {
            for (const dependency of Object.keys(manifest.dependencies)) {
                const nestedDependencyFolder = path.dirname(require.resolve(`${dependency}/package.json`, {
                    paths: [dir],
                }));
                if (fs.existsSync(nestedDependencyFolder)) {
                    discoverJsii(nestedDependencyFolder);
                }
            }
        }
    };
    // read all .jsii manifests from all requested modules and merge
    // them all into a single map of fqn->type.
    for (const dir of [...moduleDirs, PROJEN_MODULE_ROOT]) {
        discoverJsii(dir);
        // Read from scoped packages
        if (dir.includes("@") && fs.lstatSync(dir).isDirectory()) {
            const childDirs = fs.readdirSync(dir).map((file) => path.join(dir, file));
            for (const child of childDirs) {
                discoverJsii(child);
            }
        }
    }
    return jsii;
}
function resolveProjectType(projectFqn) {
    let [moduleName] = projectFqn.split(".");
    if (moduleName === "projen") {
        moduleName = PROJEN_MODULE_ROOT;
    }
    // try picking the manifest. We only need the base folder but this is directly a nice check if we request from a valid jsii package
    const jsiiManifestFile = require.resolve(`${moduleName}/.jsii`, {
        paths: [process.cwd()],
    });
    const moduleFolder = path.dirname(jsiiManifestFile);
    // Read all jsii types that can be loaded from this project type
    const jsii = discoverJsiiTypes(moduleFolder);
    return toProjectType(jsii, projectFqn);
}
exports.resolveProjectType = resolveProjectType;
function toProjectType(jsii, fqn) {
    if (!isProjectType(jsii, fqn)) {
        throw new Error(`Fully qualified name "${fqn}" is not a valid project type.`);
    }
    const typeinfo = jsii[fqn];
    // projen.web.ReactProject -> web.ReactProject
    const typename = fqn.substring(fqn.indexOf(".") + 1);
    const docsurl = `https://projen.io/api/API.html#projen-${typename
        .toLocaleLowerCase()
        .replace(/\./g, "-")}`;
    let pjid = typeinfo.docs?.custom?.pjid ?? case_1.snake(typename).replace(/_project$/, "");
    return {
        moduleName: typeinfo.assembly,
        typename,
        pjid,
        fqn,
        options: discoverOptions(jsii, fqn).sort((o1, o2) => o1.name.localeCompare(o2.name)),
        docs: typeinfo.docs?.summary,
        docsurl,
    };
}
function readJsiiManifest(jsiiFqn) {
    let [moduleName] = jsiiFqn.split(".");
    if (moduleName === "projen") {
        moduleName = PROJEN_MODULE_ROOT;
    }
    const jsiiManifestFile = require.resolve(`${moduleName}/.jsii`);
    return JSON.parse(fs.readFileSync(jsiiManifestFile, "utf-8"));
}
exports.readJsiiManifest = readJsiiManifest;
function discoverOptions(jsii, fqn) {
    const options = {};
    const params = jsii[fqn]?.initializer?.parameters ?? [];
    const optionsParam = params[0];
    const optionsTypeFqn = optionsParam?.type?.fqn;
    if (params.length > 1 ||
        (params.length === 1 && optionsParam?.name !== "options")) {
        throw new Error(`constructor for project ${fqn} must have a single "options" argument of a struct type. got ${JSON.stringify(params)}`);
    }
    addOptions(optionsTypeFqn);
    const opts = Object.values(options);
    return opts.sort((a, b) => a.switch.localeCompare(b.switch));
    function addOptions(ofqn, basePath = [], optional = false) {
        if (!ofqn) {
            return;
        }
        const struct = jsii[ofqn];
        if (!struct) {
            throw new Error(`unable to find options type ${ofqn} for project ${fqn}`);
        }
        for (const prop of struct.properties ?? []) {
            const propPath = [...basePath, prop.name];
            // protect against double-booking
            if (prop.name in options) {
                throw new Error(`duplicate option "${prop.name}" in ${fqn} (already declared in ${options[prop.name].parent})`);
            }
            let jsiiKind;
            if (prop.type?.fqn) {
                jsiiKind = jsii[prop.type?.fqn].kind; // e.g. 'class', 'interface', 'enum'
            }
            const isOptional = optional || prop.optional;
            let defaultValue = prop.docs?.default;
            if (defaultValue === "undefined") {
                defaultValue = undefined;
            }
            // if this is a mandatory option and we have a default value, it has to be JSON-parsable to the correct type
            if (!isOptional && defaultValue) {
                if (!prop.type?.primitive) {
                    throw new Error(`required option "${prop.name}" with a @default must use primitive types (string, number or boolean). type found is: ${JSON.stringify(prop.type)}`);
                }
                checkDefaultIsParsable(prop.name, defaultValue, prop.type?.primitive);
            }
            options[prop.name] = filterUndefined({
                path: propPath,
                parent: struct.name,
                name: prop.name,
                fqn: prop.type?.fqn,
                docs: prop.docs.summary,
                simpleType: prop.type ? getSimpleTypeName(prop.type) : "unknown",
                fullType: prop.type,
                kind: jsiiKind,
                jsonLike: prop.type ? isJsonLike(jsii, prop.type) : undefined,
                switch: propPath.map((p) => case_1.snake(p).replace(/_/g, "-")).join("-"),
                default: defaultValue,
                optional: isOptional,
                featured: prop.docs?.custom?.featured === "true",
                deprecated: prop.docs.stability === "deprecated" ? true : undefined,
            });
        }
        for (const ifc of struct.interfaces ?? []) {
            addOptions(ifc);
        }
    }
}
function getSimpleTypeName(type) {
    if (type?.primitive) {
        return type.primitive; // e.g. 'string', 'boolean', 'number'
    }
    else if (type?.fqn) {
        return type.fqn.split(".").pop(); // projen.NodeProjectOptions -> NodeProjectOptions
    }
    else {
        // any other types such as collection types
        return "unknown";
    }
}
/**
 * Whether a value of this type is serializable into JSON.
 */
function isJsonLike(jsii, type) {
    if (type.primitive) {
        // string, boolean, number, any
        return true;
    }
    else if (type.fqn) {
        const kind = jsii[type.fqn].kind;
        if (["interface", "enum"].includes(kind)) {
            // not 'class'
            return true;
        }
    }
    else if (type.collection) {
        return isJsonLike(jsii, type.collection.elementtype);
    }
    return false;
}
function filterUndefined(obj) {
    const ret = {};
    for (const [k, v] of Object.entries(obj)) {
        if (v !== undefined) {
            ret[k] = v;
        }
    }
    return ret;
}
function isProjectType(jsii, fqn) {
    const type = jsii[fqn];
    if (!type) {
        throw new Error(`Could not find project type with fqn "${fqn}" in  .jsii file.`);
    }
    if (type.kind !== "class") {
        return false;
    }
    if (type.abstract) {
        return false;
    }
    if (type.docs?.deprecated) {
        return false;
    }
    let curr = type;
    while (true) {
        if (curr.fqn === PROJECT_BASE_FQN) {
            return true;
        }
        if (!curr.base) {
            return false;
        }
        curr = jsii[curr.base];
        if (!curr) {
            return false;
        }
    }
}
function checkDefaultIsParsable(prop, value, type) {
    // macros are pass-through
    if (value.startsWith("$")) {
        return;
    }
    try {
        const parsed = JSON.parse(value);
        if (typeof parsed !== type) {
            throw new Error(`cannot parse @default value for mandatory option ${prop} as a ${type}: ${parsed}`);
        }
    }
    catch (e) {
        throw new Error(`unable to JSON.parse() value "${value}" specified as @default for mandatory option "${prop}": ${e.message}`);
    }
}
//# sourceMappingURL=data:application/json;base64,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